#!/usr/bin/env node
/**
 * Generate RSA key pair for JWT signing
 * Usage: node scripts/generate-jwt-keys.js
 */

const crypto = require("crypto");
const fs = require("fs");
const path = require("path");

const KEYS_DIR = process.env.JWT_KEYS_DIR || path.join(__dirname, "../keys");
const PRIVATE_KEY_PATH = path.join(KEYS_DIR, "jwt-private.pem");
const PUBLIC_KEY_PATH = path.join(KEYS_DIR, "jwt-public.pem");

console.log("🔐 Generating RSA key pair for JWT...\n");

// Create keys directory
if (!fs.existsSync(KEYS_DIR)) {
	fs.mkdirSync(KEYS_DIR, { recursive: true });
}

// Check if keys already exist
if (fs.existsSync(PRIVATE_KEY_PATH) || fs.existsSync(PUBLIC_KEY_PATH)) {
	console.log("⚠️  Keys already exist!");
	console.log(`   Private: ${PRIVATE_KEY_PATH}`);
	console.log(`   Public:  ${PUBLIC_KEY_PATH}`);
	console.log("\nTo regenerate, delete the existing keys first.");
	console.log("⚠️  WARNING: Regenerating keys will invalidate all existing tokens!\n");
	process.exit(1);
}

// Generate 2048-bit RSA key pair
const { privateKey, publicKey } = crypto.generateKeyPairSync("rsa", {
	modulusLength: 2048,
	publicKeyEncoding: {
		type: "spki",
		format: "pem",
	},
	privateKeyEncoding: {
		type: "pkcs8",
		format: "pem",
	},
});

// Save keys
fs.writeFileSync(PRIVATE_KEY_PATH, privateKey, { mode: 0o600 });
fs.writeFileSync(PUBLIC_KEY_PATH, publicKey, { mode: 0o644 });

console.log("✅ Keys generated successfully!\n");
console.log(`   Private Key: ${PRIVATE_KEY_PATH}`);
console.log(`   Public Key:  ${PUBLIC_KEY_PATH}`);
console.log("\n📋 Next steps:");
console.log("   1. Keep the private key SECRET (server only)");
console.log("   2. Share the public key with frontend for token verification");
console.log("   3. Add 'keys/' to .gitignore in production");
console.log("\n🔒 Security Notes:");
console.log("   - Private key is used to SIGN tokens (server only)");
console.log("   - Public key is used to VERIFY tokens (can be shared)");
console.log("   - Never commit private keys to version control\n");
