#!/usr/bin/env node

/**
 * Generate VAPID Keys for Web Push Notifications
 * 
 * This script generates a pair of VAPID keys required for web push notifications.
 * Run this script once during initial setup and save the keys in your .env file.
 * 
 * Usage:
 *   node scripts/generate-vapid-keys.js
 * 
 * Output:
 *   - VAPID Public Key (share with clients)
 *   - VAPID Private Key (keep secret on server)
 *   - Example .env configuration
 */

const webpush = require('web-push');
const fs = require('fs');
const path = require('path');

console.log('\n' + '='.repeat(70));
console.log('🔐 VAPID Key Generator for Web Push Notifications');
console.log('='.repeat(70) + '\n');

try {
  // Generate VAPID keys
  console.log('Generating VAPID keys...\n');
  const vapidKeys = webpush.generateVAPIDKeys();

  // Display keys
  console.log('✅ VAPID Keys Generated Successfully!\n');
  
  console.log('━'.repeat(70));
  console.log('📋 Your VAPID Keys:');
  console.log('━'.repeat(70) + '\n');
  
  console.log('Public Key (share with clients):');
  console.log('\x1b[32m%s\x1b[0m', vapidKeys.publicKey);
  console.log();
  
  console.log('Private Key (keep secret):');
  console.log('\x1b[31m%s\x1b[0m', vapidKeys.privateKey);
  console.log();

  // Generate .env format
  console.log('━'.repeat(70));
  console.log('📝 Add these to your .env file:');
  console.log('━'.repeat(70) + '\n');

  const envContent = `# Web Push Notification Configuration
VAPID_PUBLIC_KEY=${vapidKeys.publicKey}
VAPID_PRIVATE_KEY=${vapidKeys.privateKey}
VAPID_SUBJECT=mailto:admin@example.com  # Change to your email`;

  console.log(envContent);
  console.log();

  // Offer to save to file
  const outputPath = path.join(__dirname, '..', 'vapid-keys.txt');
  
  try {
    fs.writeFileSync(outputPath, envContent);
    console.log('━'.repeat(70));
    console.log(`✅ Keys saved to: ${outputPath}`);
    console.log('━'.repeat(70) + '\n');
    console.log('\x1b[33m%s\x1b[0m', '⚠️  WARNING: Keep vapid-keys.txt secure and add it to .gitignore!');
    console.log('\x1b[33m%s\x1b[0m', '⚠️  Never commit your private key to version control!');
    console.log();
  } catch (writeError) {
    console.log('\x1b[33m%s\x1b[0m', '⚠️  Could not save to file, please copy manually.');
    console.log();
  }

  // Display instructions
  console.log('━'.repeat(70));
  console.log('📚 Next Steps:');
  console.log('━'.repeat(70) + '\n');
  console.log('1. Copy the keys above to your .env file');
  console.log('2. Update VAPID_SUBJECT with your actual email');
  console.log('3. Restart your application');
  console.log('4. Share the Public Key with your frontend application');
  console.log('5. Keep the Private Key secret on your server\n');

  console.log('━'.repeat(70));
  console.log('🔒 Security Notes:');
  console.log('━'.repeat(70) + '\n');
  console.log('• The Private Key must be kept SECRET');
  console.log('• Never expose the Private Key in client-side code');
  console.log('• Never commit keys to version control');
  console.log('• Add vapid-keys.txt to .gitignore');
  console.log('• If compromised, regenerate new keys immediately\n');

  console.log('━'.repeat(70));
  console.log('✨ Generation Complete!');
  console.log('━'.repeat(70) + '\n');

  process.exit(0);

} catch (error) {
  console.error('\x1b[31m%s\x1b[0m', '❌ Error generating VAPID keys:');
  console.error(error.message);
  console.log();
  console.log('Make sure web-push is installed:');
  console.log('  npm install web-push');
  console.log();
  process.exit(1);
}
