const express = require("express");
const helmet = require("helmet");
const cors = require("cors");
const path = require("path");
const { routes } = require("./routes");
const { errorMiddleware } = require("./middlewares/error.middleware");
const { logger } = require("./core/logger/logger");
const cookieParser = require("cookie-parser");

const app = express();

app.use(helmet());
app.use(
	cors({
		origin: function (origin, callback) {
			const allowedOrigins = [
				"http://localhost:3001",
				"http://localhost:3000",
				"http://localhost:3002",
				"http://127.0.0.1:3001",
				"http://127.0.0.1:3000",
				process.env.CORS_ORIGIN,
			].filter(Boolean);

			if (!origin) return callback(null, true);

			if (allowedOrigins.includes(origin)) {
				callback(null, true);
			} else {
				callback(new Error("Not allowed by CORS"));
			}
		},
		credentials: true,
		methods: ["GET", "POST", "PUT", "DELETE", "PATCH", "OPTIONS"],
		allowedHeaders: [
			"Content-Type",
			"Accept",
			"Accept-Language",
			"Accept-Encoding",
			"Authorization",
			"X-Request-Id",
			"X-Session-Id",
			"X-Device-Id",
			"X-Origin",
			"X-Signature",
			"X-Nonce",
			"X-Timestamp",
			"X-CSRF-Token",
			"X-API-Key",
			"X-Forwarded-For",
			"X-Real-IP",
		],
		exposedHeaders: [
			"X-Request-Id",
			"X-Session-Id",
			"X-RateLimit-Limit",
			"X-RateLimit-Remaining",
			"X-RateLimit-Reset",
			"Retry-After",
		],
	}),
);

app.use(express.json({ limit: "1mb" }));
app.use(express.urlencoded({ extended: true, limit: "1mb" }));
app.use(cookieParser());

// Serve static files from uploads directory
app.use("/uploads", express.static(path.join(__dirname, "..", "uploads")));

app.use((req, res, next) => {
	const start = Date.now();
	res.on("finish", () => {
		const duration = Date.now() - start;
		const meta = {
			method: req.method,
			path: req.originalUrl,
			status: res.statusCode,
			durationMs: duration,
			requestId: req.requestId,
		};
		if (res.statusCode >= 500) {
			logger.error("Request completed", meta);
		} else if (res.statusCode >= 400) {
			logger.warn("Request completed", meta);
		} else if (duration > 1000) {
			logger.warn("Slow request", meta);
		} else {
			logger.info("Request completed", meta);
		}
	});
	next();
});

app.use("/api", routes);

app.use((req, res) =>
	res.status(404).json({ error: { message: "Not found" } }),
);
app.use(errorMiddleware);

module.exports = { app };
