const fs = require("fs");
const path = require("path");
const crypto = require("crypto");

const isProd = process.env.NODE_ENV === "production";

// RSA Key Paths (can be overridden via environment)
const KEYS_DIR = process.env.JWT_KEYS_DIR || path.join(__dirname, "../../keys");
const PRIVATE_KEY_PATH = path.join(KEYS_DIR, "jwt-private.pem");
const PUBLIC_KEY_PATH = path.join(KEYS_DIR, "jwt-public.pem");

// Generate RSA keys if they don't exist (dev only)
const ensureKeys = () => {
	if (!fs.existsSync(KEYS_DIR)) {
		fs.mkdirSync(KEYS_DIR, { recursive: true });
	}

	if (!fs.existsSync(PRIVATE_KEY_PATH) || !fs.existsSync(PUBLIC_KEY_PATH)) {
		if (isProd) {
			throw new Error(
				"JWT RSA keys not found. Generate keys first.",
			);
		}

		console.log("⚠️  Generating development JWT RSA keys...");
		const { privateKey, publicKey } = crypto.generateKeyPairSync("rsa", {
			modulusLength: 2048,
			publicKeyEncoding: { type: "spki", format: "pem" },
			privateKeyEncoding: { type: "pkcs8", format: "pem" },
		});

		fs.writeFileSync(PRIVATE_KEY_PATH, privateKey);
		fs.writeFileSync(PUBLIC_KEY_PATH, publicKey);
		console.log("✅ JWT RSA keys generated at:", KEYS_DIR);
	}
};

// Initialize keys
ensureKeys();

// Load keys
const privateKey = fs.readFileSync(PRIVATE_KEY_PATH, "utf8");
const publicKey = fs.readFileSync(PUBLIC_KEY_PATH, "utf8");

const jwtConfig = {
	accessTokenTtl: 15 * 60, // 15 minutes
	refreshTokenTtl: 30 * 24 * 60 * 60, // 30 days
	issuer: process.env.JWT_ISSUER || "nodemeta",
	audience: process.env.JWT_AUDIENCE || "nodemeta-clients",
	algorithm: "RS256",
	privateKey,
	publicKey,
	// Legacy secret (deprecated - for migration only)
	secret: process.env.JWT_SECRET || "dev-secret-change-in-production",
};

module.exports = { jwtConfig, PUBLIC_KEY_PATH };
