const { logger } = require("../logger/logger");
const { CacheEntry } = require("../models/cacheEntry.model");

class CacheService {
	constructor() {
		this.store = new Map(); // Keep in-memory for performance

		// Background cleanup every minute
		this.cleanupInterval = setInterval(() => this.#cleanup(), 60000);

		// Load persisted cache on startup
		this.#loadFromDatabase();
	}

	#now() {
		return Date.now();
	}

	#cleanup() {
		const now = this.#now();
		let cleaned = 0;
		for (const [key, record] of this.store) {
			if (record.expiresAt && record.expiresAt <= now) {
				this.store.delete(key);
				cleaned++;
			}
		}
		if (cleaned > 0) {
			logger.debug("Cache cleanup", {
				removed: cleaned,
				remaining: this.store.size,
			});
		}

		// Also clean up expired entries in database
		this.#cleanupDatabase().catch((err) => {
			logger.error("Database cleanup failed", { error: err.message });
		});
	}

	#isExpired(record) {
		if (!record) return true;
		return record.expiresAt && record.expiresAt <= this.#now();
	}

	async get(key) {
		// Check in-memory first
		const record = this.store.get(key);
		if (record && !this.#isExpired(record)) {
			return record.value;
		}

		// If not in memory or expired, check database
		try {
			const dbEntry = await CacheEntry.findOne({ key });
			if (!dbEntry) {
				if (record) this.store.delete(key); // Clean up expired in-memory
				return null;
			}

			if (this.#isExpired({ expiresAt: dbEntry.expiresAt?.getTime() })) {
				// Clean up expired entry
				await CacheEntry.deleteOne({ key });
				this.store.delete(key);
				return null;
			}

			// Cache in memory for future use
			const memoryRecord = {
				value: dbEntry.value,
				expiresAt: dbEntry.expiresAt?.getTime(),
			};
			this.store.set(key, memoryRecord);

			return dbEntry.value;
		} catch (err) {
			logger.error("Failed to get from cache database", {
				key,
				error: err.message,
			});
			return record && !this.#isExpired(record) ? record.value : null;
		}
	}

	async set(key, value, ttlSeconds) {
		const expiresAt = ttlSeconds ? this.#now() + ttlSeconds * 1000 : null;
		const memoryRecord = { value, expiresAt };
		this.store.set(key, memoryRecord);

		try {
			await CacheEntry.findOneAndUpdate(
				{ key },
				{
					value,
					expiresAt: expiresAt ? new Date(expiresAt) : null,
				},
				{ upsert: true, new: true },
			);
		} catch (err) {
			logger.error("Failed to save to cache database", {
				key,
				error: err.message,
			});
		}

		return true;
	}

	// Atomic set-if-not-exists - returns true if set, false if key already exists
	async setNX(key, value, ttlSeconds) {
		// Check in-memory first
		const existing = this.store.get(key);
		if (existing && !this.#isExpired(existing)) return false;

		// Check database
		try {
			const dbEntry = await CacheEntry.findOne({ key });
			if (
				dbEntry &&
				!this.#isExpired({ expiresAt: dbEntry.expiresAt?.getTime() })
			) {
				return false;
			}
		} catch (err) {
			logger.error("Failed to check cache database for setNX", {
				key,
				error: err.message,
			});
			return false;
		}

		// Set in both memory and database
		const expiresAt = ttlSeconds ? this.#now() + ttlSeconds * 1000 : null;
		const memoryRecord = { value, expiresAt };
		this.store.set(key, memoryRecord);

		try {
			await CacheEntry.findOneAndUpdate(
				{ key },
				{
					value,
					expiresAt: expiresAt ? new Date(expiresAt) : null,
				},
				{ upsert: true, new: true },
			);
			return true;
		} catch (err) {
			logger.error("Failed to save to cache database in setNX", {
				key,
				error: err.message,
			});
			this.store.delete(key); // Rollback memory
			return false;
		}
	}

	async increment(key, ttlSeconds) {
		try {
			// Use MongoDB atomic increment
			const expiresAt = ttlSeconds
				? new Date(this.#now() + ttlSeconds * 1000)
				: null;

			const result = await CacheEntry.findOneAndUpdate(
				{ key },
				[
					{
						$set: {
							value: { $add: ["$value", 1] },
							expiresAt: expiresAt,
						},
					},
				],
				{
					upsert: true,
					new: true,
					setDefaultsOnInsert: { value: 0 },
				},
			);

			// Update in-memory cache
			const memoryRecord = {
				value: result.value,
				expiresAt: result.expiresAt?.getTime(),
			};
			this.store.set(key, memoryRecord);

			return result.value;
		} catch (err) {
			logger.error("Failed to increment in cache database", {
				key,
				error: err.message,
			});
			// Fallback to in-memory only
			const record = this.store.get(key);
			if (!record || this.#isExpired(record)) {
				const expiresAt = ttlSeconds ? this.#now() + ttlSeconds * 1000 : null;
				this.store.set(key, { value: 1, expiresAt });
				return 1;
			}
			const next = (record.value || 0) + 1;
			record.value = next;
			return next;
		}
	}

	async delete(key) {
		this.store.delete(key);

		try {
			await CacheEntry.deleteOne({ key });
		} catch (err) {
			logger.error("Failed to delete from cache database", {
				key,
				error: err.message,
			});
		}

		return true;
	}

	async exists(key) {
		// Check in-memory first
		const record = this.store.get(key);
		if (record && !this.#isExpired(record)) {
			return true;
		}

		// Check database
		try {
			const dbEntry = await CacheEntry.findOne({ key });
			if (!dbEntry) {
				if (record) this.store.delete(key); // Clean up expired
				return false;
			}

			if (this.#isExpired({ expiresAt: dbEntry.expiresAt?.getTime() })) {
				await CacheEntry.deleteOne({ key });
				this.store.delete(key);
				return false;
			}

			// Cache in memory
			const memoryRecord = {
				value: dbEntry.value,
				expiresAt: dbEntry.expiresAt?.getTime(),
			};
			this.store.set(key, memoryRecord);

			return true;
		} catch (err) {
			logger.error("Failed to check existence in cache database", {
				key,
				error: err.message,
			});
			return record && !this.#isExpired(record);
		}
	}

	// Get cache stats
	getStats() {
		return { size: this.store.size };
	}

	async #loadFromDatabase() {
		try {
			const now = new Date();
			const entries = await CacheEntry.find({
				$or: [{ expiresAt: null }, { expiresAt: { $gt: now } }],
			}).limit(1000); // Limit initial load

			for (const entry of entries) {
				this.store.set(entry.key, {
					value: entry.value,
					expiresAt: entry.expiresAt?.getTime(),
				});
			}

			logger.info("Cache loaded from database", {
				keys: this.store.size,
				source: "database",
			});
		} catch (err) {
			logger.error("Failed to load cache from database", {
				error: err.message,
			});
			logger.warn("Starting with empty cache");
		}
	}

	async #cleanupDatabase() {
		try {
			const now = new Date();
			const result = await CacheEntry.deleteMany({
				expiresAt: { $lte: now },
			});

			if (result.deletedCount > 0) {
				logger.debug("Database cache cleanup", {
					removed: result.deletedCount,
				});
			}
		} catch (err) {
			logger.error("Database cleanup failed", { error: err.message });
		}
	}
}

const cacheService = new CacheService();

module.exports = { cacheService };
