const isDev = process.env.NODE_ENV !== "production";

const LOG_ICONS = {
	debug: "🔍", // Magnifying glass for debug/inspection
	info: "ℹ️", // Information symbol
	warn: "⚠️", // Warning sign
	error: "❌", // Error/cross mark
};

const format = (level, message, meta = {}) => {
	const timestamp = new Date().toISOString();
	const { requestId, ...rest } = meta;
	const icon = LOG_ICONS[level] || "📝";
	return JSON.stringify({
		timestamp,
		level: `${icon} ${level.toUpperCase()}`,
		message,
		...(requestId && { requestId }),
		...rest,
	});
};

const createLogger = (defaultMeta = {}) => ({
	debug(message, meta = {}) {
		if (isDev)
			console.debug(format("debug", message, { ...defaultMeta, ...meta }));
	},
	info(message, meta = {}) {
		console.info(format("info", message, { ...defaultMeta, ...meta }));
	},
	warn(message, meta = {}) {
		console.warn(format("warn", message, { ...defaultMeta, ...meta }));
	},
	error(message, meta = {}) {
		console.error(format("error", message, { ...defaultMeta, ...meta }));
	},
	// Create child logger with persistent context
	child(meta) {
		return createLogger({ ...defaultMeta, ...meta });
	},
});

const logger = createLogger();

module.exports = { logger, createLogger };
