const mongoose = require('mongoose');

const jobSchema = new mongoose.Schema({
  id: {
    type: String,
    required: true,
    unique: true,
    index: true
  },
  name: {
    type: String,
    required: true,
    index: true
  },
  data: {
    type: mongoose.Schema.Types.Mixed,
    required: true
  },
  opts: {
    type: mongoose.Schema.Types.Mixed,
    default: {}
  },
  attemptsMade: {
    type: Number,
    default: 0
  },
  status: {
    type: String,
    enum: ['pending', 'processing', 'completed', 'failed'],
    default: 'pending',
    index: true
  },
  createdAt: {
    type: Date,
    default: Date.now,
    index: true
  },
  startedAt: {
    type: Date
  },
  completedAt: {
    type: Date
  },
  failedAt: {
    type: Date
  },
  error: {
    type: String
  },
  priority: {
    type: Number,
    default: 0,
    index: true
  }
}, {
  timestamps: true
});

// Compound indexes for efficient queries
jobSchema.index({ status: 1, createdAt: 1 });
jobSchema.index({ status: 1, priority: -1, createdAt: 1 });

// TTL index for automatic cleanup of old completed jobs (30 days)
jobSchema.index({ completedAt: 1 }, {
  expireAfterSeconds: 30 * 24 * 60 * 60,
  partialFilterExpression: { status: 'completed' }
});

// TTL index for failed jobs (7 days)
jobSchema.index({ failedAt: 1 }, {
  expireAfterSeconds: 7 * 24 * 60 * 60,
  partialFilterExpression: { status: 'failed' }
});

const Job = mongoose.model('Job', jobSchema);

module.exports = { Job };