const { rateLimitService } = require("./rateLimit.service");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../logger/logger");

// Rate limiting - Prevents abuse by limiting requests per user/IP
// Policies: 'default' (normal), 'sensitive' (stricter), 'strict' (very strict)
// Returns 429 if limit exceeded with Retry-After header

const rateLimitMiddleware =
	(policyType = "default") =>
	async (req, res, next) => {
		try {
			const policy = rateLimitService.getPolicy(policyType);
			if (!policy) return next();

			const userId = req.user ? req.user._id : null;
			const identifier = userId
				? `user:${userId}`
				: req.clientIp !== "unknown"
				? `ip:${req.clientIp}`
				: "anonymous";
			const { blocked, remaining, resetSeconds } =
				await rateLimitService.attempt(identifier, policy);

			res.setHeader("X-RateLimit-Limit", policy.max);
			res.setHeader("X-RateLimit-Remaining", Math.max(remaining, 0));
			res.setHeader("X-RateLimit-Reset", resetSeconds);

			if (blocked) {
				res.setHeader("Retry-After", resetSeconds);
				logger.warn("Rate limit exceeded", {
					identifier,
					policy: policyType,
					requestId: req.requestId,
				});
				throw new ApiError(
					429,
					`Rate limit exceeded. Retry in ${resetSeconds}s`,
				);
			}

			next();
		} catch (err) {
			if (err instanceof ApiError) return next(err);
			next();
		}
	};

module.exports = { rateLimitMiddleware };
