const cron = require("node-cron");
const CronManager = require("./cronManager");
const rankManagementService = require("../../modules/ranks/rankManagement.service");
const {
	updateAllExpiredPackages,
} = require("../../modules/bonuses/bonus.service");
const {
	pancakeswapService: pancakeswapTransactionService,
} = require("../../modules/pancakeswapTransactions/pancakeswapTransaction.service");
const { adminService } = require("../../modules/admin/admin.service");
const {
	updateUsersWithActivePackagesCache,
	updatePackageUserCountsCache,
} = require("../../modules/config/globalConfig.service");
const {
	getPackageOwners,
} = require("../../modules/purchase/services/packageOwners.service");
const dbeProgramAttendeeService = require("../../modules/dbe/dbeProgramAttendee.service");
const { logger } = require("../logger/logger");
const { CronJobLog } = require("../../modules/system/cronJobLog.model");
const { createDatabaseBackup } = require("../../modules/system/databaseBackup.service");
const { RankCronJobResult } = require("../../modules/ranks/rankCronJobResult.model");

const loggerInstance = {
	info: (message, data = {}) => {
		const logEntry = {
			timestamp: new Date().toISOString(),
			level: "INFO",
			message,
			...data,
		};
		logger.info(message, logEntry);
		return logEntry;
	},

	error: (message, error = null, data = {}) => {
		const logEntry = {
			timestamp: new Date().toISOString(),
			level: "ERROR",
			message,
			error: error
				? {
						message: error.message,
						stack: error.stack,
						name: error.name,
				  }
				: null,
			...data,
		};
		logger.error(message, logEntry);
		return logEntry;
	},

	warn: (message, data = {}) => {
		const logEntry = {
			timestamp: new Date().toISOString(),
			level: "WARN",
			message,
			...data,
		};
		logger.warn(message, logEntry);
		return logEntry;
	},

	success: (message, data = {}) => {
		const logEntry = {
			timestamp: new Date().toISOString(),
			level: "INFO",
			message,
			...data,
		};
		logger.info(message, logEntry);
		return logEntry;
	},
};

// Log cron job execution to database
const logCronJobExecution = async (jobData) => {
	try {
		const logEntry = new CronJobLog({
			jobType: jobData.jobType,
			jobId: jobData.jobId || `${jobData.jobType}_${Date.now()}`,
			status: jobData.status,
			isSuccess: jobData.isSuccess,
			timestamp: new Date().toISOString(),
			startTime: jobData.startTime,
			endTime: jobData.endTime,
			durationMs: jobData.durationMs,
			totalRetries: jobData.totalRetries || 0,
			attempts: jobData.attempts || [],
			finalResult: jobData.finalResult || {},
			finalError: jobData.isSuccess ? {} : jobData.finalError || {},
			summary: jobData.summary || {
				totalAttempts: 1,
				successfulAttempts: jobData.isSuccess ? 1 : 0,
				failedAttempts: jobData.isSuccess ? 0 : 1,
				avgAttemptDuration: jobData.durationMs,
			},
			systemContext: {
				platform: process.platform,
				nodeVersion: process.version,
				environment: process.env.NODE_ENV || "development",
				memoryUsage: process.memoryUsage(),
			},
		});

		await logEntry.save();
		loggerInstance.info(`Cron job logged to database: ${jobData.jobType}`, {
			jobId: logEntry.jobId,
			status: jobData.status,
			duration: jobData.durationMs,
		});
	} catch (error) {
		loggerInstance.error("Failed to log cron job execution to database", error);
	}
};

const updatePackageUserCountsJob = async () => {
	try {
		const result = await updatePackageUserCountsCache();
		return {
			success: true,
			message: "Package user counts updated successfully",
			...result,
		};
	} catch (error) {
		loggerInstance.error("Failed to update package user counts:", error);
		return {
			success: false,
			error: error.message,
		};
	}
};

const updateUsersWithActivePackagesJob = async () => {
	try {
		const result = await updateUsersWithActivePackagesCache();
		return {
			size: result?.length || 0,
			success: true,
			message: "Users with active packages cache updated successfully",
		};
	} catch (error) {
		loggerInstance.error(
			"Failed to update users with active packages cache:",
			error,
		);
		return {
			success: false,
			error: error.message,
			size: 0,
		};
	}
};

const updateAllExpiredPackageJob = async () => {
	try {
		const result = await updateAllExpiredPackages();
		return result;
	} catch (error) {
		loggerInstance.error("Failed to update expired packages:", error);
		return {
			success: false,
			error: error.message,
			updatedCount: 0,
		};
	}
};

const updatePackageOwnersCache = async () => {
	try {
		const result = await getPackageOwners();
		return result;
	} catch (error) {
		loggerInstance.error("Failed to update package owners cache:", error);
		return {
			success: false,
			error: error.message,
			cachedPackages: 0,
		};
	}
};

const performDailyDBEVerification = async () => {
	try {
		const result =
			await dbeProgramAttendeeService.performDailyDBEVerification();
		return result;
	} catch (error) {
		loggerInstance.error("Daily DBE verification failed:", error);
		return {
			success: false,
			error: error.message,
			processed: 0,
			disqualified: 0,
		};
	}
};

// Cron job schedules from environment variables
const CRON_SCHEDULES = {
	RANK_UPDATE: process.env.RANK_UPDATE_SCHEDULE,
	BONUS_DISTRIBUTION: process.env.BONUS_DISTRIBUTION_SCHEDULE,
	PACKAGE_EXPIRATION: process.env.PACKAGE_EXPIRATION_SCHEDULE,
	PACKAGE_USER_COUNTS: process.env.PACKAGE_USER_COUNTS_SCHEDULE,
	USERS_WITH_PACKAGES: process.env.USERS_WITH_PACKAGES_SCHEDULE,
	RANK_SALARY: process.env.RANK_SALARY_SCHEDULE,
	MONTHLY_BONUS: process.env.MONTHLY_BONUS_SCHEDULE,
	DATABASE_BACKUP: process.env.DATABASE_BACKUP_SCHEDULE,
	PACKAGE_OWNERS: process.env.PACKAGE_OWNERS_SCHEDULE,
	PANCAKESWAP_TRACKER: process.env.PANCAKESWAP_TRACKER_SCHEDULE,
	DBE_DAILY_VERIFICATION: process.env.DBE_DAILY_VERIFICATION_SCHEDULE,
};

const RETRY_CONFIG = {
	MAX_RETRIES: parseInt(process.env.CRON_MAX_RETRIES) || 3,
	RETRY_DELAY_BASE: parseInt(process.env.CRON_RETRY_DELAY_BASE) || 1000,
	RETRY_DELAY_MULTIPLIER:
		parseInt(process.env.CRON_RETRY_DELAY_MULTIPLIER) || 2,
};

const jobMetrics = {
	executions: {},
	failures: {},
	lastSuccess: {},
	averageDuration: {},
	retryCount: {},
};

const performanceMonitor = {
	slowJobs: [],
	memoryUsage: [],
	jobDurations: {},

	trackPerformance(jobName, startTime, endTime) {
		const duration = endTime - startTime;
		if (duration > 300000) {
			this.slowJobs.push({ jobName, duration, timestamp: new Date() });
		}

		const memUsage = process.memoryUsage();
		this.memoryUsage.push({
			jobName,
			rss: memUsage.rss,
			heapUsed: memUsage.heapUsed,
			timestamp: new Date(),
		});

		if (!this.jobDurations[jobName]) {
			this.jobDurations[jobName] = [];
		}
		this.jobDurations[jobName].push({ duration, timestamp: new Date() });

		if (this.jobDurations[jobName].length > 100) {
			this.jobDurations[jobName] = this.jobDurations[jobName].slice(-100);
		}
	},

	getPerformanceStats() {
		const stats = {
			slowJobs: this.slowJobs.slice(-10),
			memoryUsage: this.memoryUsage.slice(-10),
			jobDurations: {},
		};

		Object.keys(this.jobDurations).forEach((jobName) => {
			const durations = this.jobDurations[jobName].map((d) => d.duration);
			if (durations.length > 0) {
				stats.jobDurations[jobName] = {
					average: durations.reduce((a, b) => a + b, 0) / durations.length,
					min: Math.min(...durations),
					max: Math.max(...durations),
					count: durations.length,
					lastDuration: durations[durations.length - 1],
				};
			}
		});

		return stats;
	},
};

async function executeWithRetry(
	jobName,
	jobFunction,
	maxRetries = RETRY_CONFIG.MAX_RETRIES,
) {
	let lastError;
	const jobId = `${jobName.toLowerCase().replace(/\s+/g, "-")}-${Date.now()}`;
	const startTime = Date.now();

	let jobResult = {
		jobType: jobName.toLowerCase().replace(/\s+/g, "_"),
		jobId,
		status: "running",
		startTime: new Date(),
		attempts: [],
		totalRetries: 0,
		durationMs: 0,
		isSuccess: false,
		finalResult: null,
		finalError: null,
	};

	for (let attempt = 1; attempt <= maxRetries; attempt++) {
		const attemptStartTime = Date.now();

		try {
			const result = await jobFunction();
			const attemptDuration = Date.now() - attemptStartTime;
			const totalDuration = Date.now() - startTime;

			jobResult.attempts.push({
				attemptNumber: attempt,
				status: "success",
				duration: attemptDuration,
				timestamp: new Date(),
				result: typeof result === "object" ? result : { value: result },
			});

			updateJobMetrics(jobName, true, totalDuration);
			performanceMonitor.trackPerformance(jobName, startTime, Date.now());

			jobResult.status = "success";
			jobResult.isSuccess = true;
			jobResult.durationMs = totalDuration;
			jobResult.endTime = new Date();
			jobResult.finalResult = result;

			// Calculate summary statistics
			jobResult.summary = {
				totalAttempts: jobResult.attempts.length,
				successfulAttempts: jobResult.attempts.filter(
					(a) => a.status === "success",
				).length,
				failedAttempts: jobResult.attempts.filter((a) => a.status === "failed")
					.length,
				avgAttemptDuration:
					jobResult.attempts.reduce((sum, a) => sum + a.duration, 0) /
					jobResult.attempts.length,
			};

			await logCronJobExecution(jobResult);

			return result;
		} catch (error) {
			lastError = error;
			const attemptDuration = Date.now() - attemptStartTime;

			jobResult.attempts.push({
				attemptNumber: attempt,
				status: "failed",
				duration: attemptDuration,
				timestamp: new Date(),
				error: {
					message: error.message,
					stack: error.stack,
					name: error.name,
				},
			});

			jobResult.totalRetries++;
			updateJobMetrics(jobName, false, 0);

			loggerInstance.error(
				`${jobName} failed (attempt ${attempt}/${maxRetries})`,
				error,
			);

			if (attempt === maxRetries) {
				const totalDuration = Date.now() - startTime;

				jobResult.status = "failed";
				jobResult.isSuccess = false;
				jobResult.durationMs = totalDuration;
				jobResult.endTime = new Date();
				jobResult.finalResult = {};
				jobResult.finalError = {
					message: error.message,
					stack: error.stack,
					name: error.name,
				};

				// Calculate summary statistics
				jobResult.summary = {
					totalAttempts: jobResult.attempts.length,
					successfulAttempts: jobResult.attempts.filter(
						(a) => a.status === "success",
					).length,
					failedAttempts: jobResult.attempts.filter(
						(a) => a.status === "failed",
					).length,
					avgAttemptDuration:
						jobResult.attempts.reduce((sum, a) => sum + a.duration, 0) /
						jobResult.attempts.length,
				};

				await logCronJobExecution(jobResult);

				loggerInstance.error(
					`${jobName} failed permanently after ${maxRetries} attempts`,
					error,
				);
				throw error;
			}

			const delay =
				RETRY_CONFIG.RETRY_DELAY_BASE *
				Math.pow(RETRY_CONFIG.RETRY_DELAY_MULTIPLIER, attempt - 1);
			await new Promise((resolve) => setTimeout(resolve, delay));
		}
	}

	throw lastError;
}

function updateJobMetrics(jobName, success, duration) {
	if (!jobMetrics.executions[jobName]) {
		jobMetrics.executions[jobName] = 0;
		jobMetrics.failures[jobName] = 0;
		jobMetrics.retryCount[jobName] = 0;
	}

	jobMetrics.executions[jobName]++;

	if (!success) {
		jobMetrics.failures[jobName]++;
	} else {
		jobMetrics.lastSuccess[jobName] = new Date();

		if (!jobMetrics.averageDuration[jobName]) {
			jobMetrics.averageDuration[jobName] = duration;
		} else {
			jobMetrics.averageDuration[jobName] =
				(jobMetrics.averageDuration[jobName] + duration) / 2;
		}
	}
}

function getHealthStatus() {
	const now = Date.now();
	const health = {
		status: "healthy",
		timestamp: new Date().toISOString(),
		jobs: {},
		metrics: jobMetrics,
		uptime: process.uptime(),
	};

	Object.keys(jobMetrics.lastSuccess).forEach((jobName) => {
		const lastSuccess = jobMetrics.lastSuccess[jobName];
		const executions = jobMetrics.executions[jobName] || 0;
		const failures = jobMetrics.failures[jobName] || 0;
		const averageDuration = jobMetrics.averageDuration[jobName] || 0;

		let hoursSinceLastSuccess = null;
		if (lastSuccess) {
			hoursSinceLastSuccess = (now - lastSuccess) / (1000 * 60 * 60);
		}

		let jobHealth = "healthy";
		if (!lastSuccess) {
			jobHealth = "never_run";
		} else if (hoursSinceLastSuccess > 25) {
			jobHealth = "unhealthy";
			health.status = "unhealthy";
		} else if (failures > executions * 0.5) {
			jobHealth = "degraded";
			if (health.status === "healthy") health.status = "degraded";
		}

		health.jobs[jobName] = {
			lastSuccess: lastSuccess ? new Date(lastSuccess).toISOString() : null,
			hoursSinceLastSuccess,
			executions,
			failures,
			averageDuration,
			successRate:
				executions > 0
					? (((executions - failures) / executions) * 100).toFixed(1)
					: 0,
			health: jobHealth,
		};
	});

	health.system = {
		memoryUsage: process.memoryUsage(),
		platform: process.platform,
		nodeVersion: process.version,
		environment: process.env.NODE_ENV || "development",
	};

	return health;
}

let jobsStarted = false;
let bonusJobsStarted = false;

let rankUpdateTask = null;
let bonusDistributionTask = null;
let rankSalaryTask = null;
let monthlyBonusTask = null;
let packageUserCountsTask = null;
let usersWithPackagesTask = null;
let packageExpirationTask = null;
let databaseBackupTask = null;
let packageOwnersTask = null;
let pancakeswapTrackerTask = null;
let dbeVerificationTask = null;
let startingJobs = false;

// Get the CronManager instance
const cronManager = CronManager.getInstance();

// Clean up all existing cron tasks to avoid duplicates
function destroyAllExistingTasks() {
	try {
		// Use CronManager to handle task cleanup
		cronManager.destroyAllTasks();

		// Also manually destroy our tracked tasks if they exist (just in case)
		if (rankUpdateTask) {
			rankUpdateTask.destroy();
			rankUpdateTask = null;
		}
		if (bonusDistributionTask) {
			bonusDistributionTask.destroy();
			bonusDistributionTask = null;
		}
		if (rankSalaryTask) {
			rankSalaryTask.destroy();
			rankSalaryTask = null;
		}
		if (monthlyBonusTask) {
			monthlyBonusTask.destroy();
			monthlyBonusTask = null;
		}
		if (packageUserCountsTask) {
			packageUserCountsTask.destroy();
			packageUserCountsTask = null;
		}
		if (usersWithPackagesTask) {
			usersWithPackagesTask.destroy();
			usersWithPackagesTask = null;
		}
		if (packageExpirationTask) {
			packageExpirationTask.destroy();
			packageExpirationTask = null;
		}
		if (databaseBackupTask) {
			databaseBackupTask.destroy();
			databaseBackupTask = null;
		}
		if (packageOwnersTask) {
			packageOwnersTask.destroy();
			packageOwnersTask = null;
		}
		if (pancakeswapTrackerTask) {
			pancakeswapTrackerTask.destroy();
			pancakeswapTrackerTask = null;
		}
		if (dbeVerificationTask) {
			dbeVerificationTask.destroy();
			dbeVerificationTask = null;
		}

		loggerInstance.success("All existing cron tasks destroyed successfully!");
	} catch (error) {
		loggerInstance.error("Error destroying existing tasks:", error);
	}
}

function startCronJobs() {
	// Skip cron jobs in development
	if (process.env.NODE_ENV === "development") {
		loggerInstance.warn("Cron jobs are disabled in development environment", {
			environment: process.env.NODE_ENV,
			message: "Set NODE_ENV to 'production' to enable cron jobs",
		});
		return;
	}

	// Prevent multiple simultaneous starts
	if (startingJobs) {
		loggerInstance.warn(
			"Cron jobs are already being started, ignoring duplicate call",
		);
		return;
	}

	// Check if all jobs are already running - don't start if they are
	if (jobsStarted) {
		loggerInstance.info("Checking individual job status...");

		const jobStatus = {
			packageUserCountsTask: !!packageUserCountsTask,
			bonusDistributionTask: !!bonusDistributionTask,
			packageExpirationTask: !!packageExpirationTask,
			packageOwnersTask: !!packageOwnersTask,
			rankUpdateTask: !!rankUpdateTask,
			rankSalaryTask: !!rankSalaryTask,
			monthlyBonusTask: !!monthlyBonusTask,
			databaseBackupTask: !!databaseBackupTask,
			pancakeswapTrackerTask: !!pancakeswapTrackerTask,
			dbeVerificationTask: !!dbeVerificationTask,
		};

		const runningJobs = Object.entries(jobStatus).filter(
			([name, isRunning]) => isRunning,
		);
		const missingJobs = Object.entries(jobStatus).filter(
			([name, isRunning]) => !isRunning,
		);

		loggerInstance.info("Individual job status:", {
			runningJobs: runningJobs.map(([name]) => name),
			missingJobs: missingJobs.map(([name]) => name),
			runningCount: runningJobs.length,
			missingCount: missingJobs.length,
		});

		// If all jobs are running, don't start anything new
		if (runningJobs.length === 11 && missingJobs.length === 0) {
			loggerInstance.info(
				"All 11 individual cron jobs are already running - no action needed",
			);
			return;
		} else if (missingJobs.length > 0) {
			loggerInstance.warn(
				`${missingJobs.length} jobs are missing:`,
				missingJobs.map(([name]) => name),
			);
			loggerInstance.info(
				"Stopping all jobs and restarting to ensure clean state...",
			);
			// Clean up existing tasks first
			stopCronJobs();
		}
	}

	startingJobs = true;

	// Clean up all existing tasks first
	loggerInstance.info(
		"Cleaning up all existing cron tasks before starting new ones...",
	);
	destroyAllExistingTasks();

	loggerInstance.info("Starting all cron jobs...");

	// Package User Counts Update
	packageUserCountsTask = cronManager.registerTask(
		"Package User Counts Update",
		CRON_SCHEDULES.PACKAGE_USER_COUNTS,
		async () => {
			loggerInstance.info(
				"Starting scheduled package user counts update at:",
				new Date().toISOString(),
			);
			try {
				const result = await executeWithRetry(
					"Package User Counts Update",
					updatePackageUserCountsJob,
				);
				loggerInstance.success("Package user counts updated successfully", {
					result,
				});
			} catch (error) {
				loggerInstance.error("Failed to update package user counts:", error);
			}
		},
	);
	loggerInstance.success("Package User Counts cron job scheduled", {
		schedule: CRON_SCHEDULES.PACKAGE_USER_COUNTS || "Not configured",
		priority: 4,
	});

	// Users With Active Packages Cache Update
	usersWithPackagesTask = cronManager.registerTask(
		"Users With Active Packages Cache Update",
		CRON_SCHEDULES.USERS_WITH_PACKAGES,
		async () => {
			loggerInstance.info(
				"Starting scheduled users with active packages cache update at:",
				new Date().toISOString(),
			);
			try {
				const result = await executeWithRetry(
					"Users With Active Packages Cache Update",
					updateUsersWithActivePackagesJob,
				);
				loggerInstance.success(
					"Users with active packages cache updated successfully",
					{ usersCount: result.size },
				);
			} catch (error) {
				loggerInstance.error(
					"Failed to update users with active packages cache:",
					error,
				);
			}
		},
	);
	loggerInstance.success(
		"Users With Active Packages Cache Update cron job scheduled",
		{
			schedule: CRON_SCHEDULES.USERS_WITH_PACKAGES || "Not configured",
			priority: 4.5,
		},
	);

	// Daily Bonus Distribution - runs at 00:03 UTC
	bonusDistributionTask = cronManager.registerTask(
		"Daily Bonus Distribution",
		CRON_SCHEDULES.BONUS_DISTRIBUTION,
		async () => {
			loggerInstance.info(
				"Starting scheduled daily bonus distribution at:",
				new Date().toISOString(),
			);

			try {
				// Disable withdrawals before starting
				await adminService.disableWithdrawalsForCronJob("BONUS_DISTRIBUTION");
				loggerInstance.info("Withdrawals disabled for bonus distribution");

				// Add detailed logging before execution
				loggerInstance.info(
					"Executing bonus distribution with enhanced logging...",
				);

				const result = await executeWithRetry(
					"Daily Bonus Distribution",
					async () => {
						const {
							bonusService,
						} = require("../../modules/bonuses/bonus.service");
						const distributionResult =
							await bonusService.distributeBonusesToAllUsers();

						// Log detailed results immediately after execution
						loggerInstance.info("Bonus Distribution - Raw Results:", {
							success: distributionResult.success !== false,
							totalUsers: distributionResult.totalUsers,
							successful: distributionResult.successful,
							failed: distributionResult.failed,
							alreadyDistributed: distributionResult.alreadyDistributed,
							totalBonusesDistributed:
								distributionResult.totalBonusesDistributed,
							totalTransactions: distributionResult.totalTransactions,
							errors: distributionResult.errors?.slice(0, 5) || [], // Log first 5 errors
							hasErrors: (distributionResult.errors?.length || 0) > 0,
							duration: distributionResult.endTime
								? (distributionResult.endTime - distributionResult.startTime) /
								  1000
								: 0,
						});

						// If there are errors, log them separately with more detail
						if (
							distributionResult.errors &&
							distributionResult.errors.length > 0
						) {
							loggerInstance.warn(
								`Bonus Distribution had ${distributionResult.errors.length} errors:`,
								{
									errorSample: distributionResult.errors
										.slice(0, 3)
										.map((err) => ({
											username: err.username,
											error: err.error,
											timestamp: err.timestamp,
										})),
									totalErrors: distributionResult.errors.length,
									errorUsernames: distributionResult.errors
										.slice(0, 10)
										.map((err) => err.username),
								},
							);
						}

						return distributionResult;
					},
				);

				loggerInstance.success(
					"Daily bonus distribution completed successfully",
					{
						totalUsers: result.totalUsers,
						successful: result.successful,
						failed: result.failed,
						alreadyDistributed: result.alreadyDistributed,
						totalBonusesDistributed: result.totalBonusesDistributed,
						totalTransactions: result.totalTransactions,
						duration: result.endTime
							? (result.endTime - result.startTime) / 1000
							: 0,
						timestamp: new Date().toISOString(),
						successRate:
							result.totalUsers > 0
								? ((result.successful / result.totalUsers) * 100).toFixed(1) +
								  "%"
								: "0%",
					},
				);

				// Re-enable withdrawals after successful completion
				await adminService.enableWithdrawalsAfterCronJob("BONUS_DISTRIBUTION");
				loggerInstance.info("Withdrawals re-enabled after bonus distribution");
			} catch (error) {
				loggerInstance.error("Daily bonus distribution failed:", {
					error: error.message,
					stack: error.stack,
					timestamp: new Date().toISOString(),
				});

				// Re-enable withdrawals even if the job failed
				try {
					await adminService.enableWithdrawalsAfterCronJob(
						"BONUS_DISTRIBUTION",
					);
					loggerInstance.info(
						"Withdrawals re-enabled after failed bonus distribution",
					);
				} catch (withdrawalError) {
					loggerInstance.error(
						"Failed to re-enable withdrawals after bonus distribution error:",
						withdrawalError,
					);
				}
			}
		},
	);
	loggerInstance.success("Daily Bonus Distribution cron job scheduled", {
		schedule: CRON_SCHEDULES.BONUS_DISTRIBUTION || "Not configured",
		priority: 6,
	});

	// Package Expiration Check
	packageExpirationTask = cronManager.registerTask(
		"Package Expiration Check",
		CRON_SCHEDULES.PACKAGE_EXPIRATION,
		async () => {
			loggerInstance.info(
				"Starting scheduled package expiration check at:",
				new Date().toISOString(),
			);
			try {
				const result = await executeWithRetry(
					"Package Expiration Check",
					updateAllExpiredPackageJob,
				);
				if (result.success && result.updatedCount > 0) {
					loggerInstance.success(
						`Updated ${result.updatedCount} expired packages`,
					);
				} else if (result.success) {
					loggerInstance.info("No expired packages found");
				} else {
					loggerInstance.error(
						"Error updating expired packages:",
						result.error,
					);
				}
			} catch (error) {
				loggerInstance.error("Package expiration job error:", error);
			}
		},
	);
	loggerInstance.success("Package Expiration Check cron job scheduled", {
		schedule: CRON_SCHEDULES.PACKAGE_EXPIRATION || "Not configured",
		priority: 5,
	});

	// Package Owners Cache Update
	packageOwnersTask = cronManager.registerTask(
		"Package Owners Cache Update",
		CRON_SCHEDULES.PACKAGE_OWNERS,
		async () => {
			loggerInstance.info(
				"Starting scheduled package owners cache update at:",
				new Date().toISOString(),
			);
			try {
				const result = await executeWithRetry(
					"Package Owners Cache Update",
					async () => await updatePackageOwnersCache(),
				);
				loggerInstance.success("Package owners cache updated successfully", {
					cachedPackages: result?.cachedPackages || 0,
				});
			} catch (error) {
				loggerInstance.error("Failed to update package owners cache:", error);
			}
		},
		{
			scheduled: true,
			timezone: "UTC",
		},
	);
	loggerInstance.success("Package Owners Cache Update cron job scheduled", {
		schedule: CRON_SCHEDULES.PACKAGE_OWNERS || "Not configured",
		priority: 3,
	});

	// Rank Update Job
	rankUpdateTask = cronManager.registerTask(
		"Rank Update Job",
		CRON_SCHEDULES.RANK_UPDATE,
		async () => {
			loggerInstance.info("Starting scheduled rank update job at:", {
				timestamp: new Date().toISOString(),
			});

			try {
				const result = await executeWithRetry("Rank Update Job", async () => {
					const rankResult = await rankManagementService.updateAllUsersRanks();

					// Log detailed rank update results
					loggerInstance.info("Rank Update Job - Detailed Results:", {
						totalUsers: rankResult.totalUsers,
						usersUpdated: rankResult.usersUpdated,
						duration: rankResult.duration,
						completedAt: rankResult.completedAt,
						hasErrors: !!rankResult.errors,
						errorCount: rankResult.errors || 0,
					});

					return rankResult;
				});

				loggerInstance.success("Rank update job completed successfully", {
					totalUsers: result.totalUsers,
					usersUpdated: result.usersUpdated,
					duration: result.duration,
					timestamp: new Date().toISOString(),
				});
			} catch (error) {
				loggerInstance.error("Rank update job failed:", {
					error: error.message,
					timestamp: new Date().toISOString(),
				});
			}
		},
	);
	loggerInstance.success("Rank Update Job cron job scheduled", {
		schedule: CRON_SCHEDULES.RANK_UPDATE || "Not configured",
		priority: 2,
	});

	// Monthly Rank Salary Distribution
	rankSalaryTask = cronManager.registerTask(
		"Monthly Rank Salary Distribution",
		CRON_SCHEDULES.RANK_SALARY,
		async () => {
			loggerInstance.info(
				"Starting scheduled monthly rank salary distribution at:",
				new Date().toISOString(),
			);

			try {
				// Disable withdrawals before starting
				await adminService.disableWithdrawalsForCronJob("RANK_SALARY");
				loggerInstance.info(
					"Withdrawals disabled for rank salary distribution",
				);

				const result = await executeWithRetry(
					"Monthly Rank Salary Distribution",
					async () => {
						const rankSalaryService = require("../../modules/ranks/rankSalary.service");
						return await rankSalaryService.distributeRankSalariesToAllUsers();
					},
				);
				loggerInstance.success(
					"Monthly rank salary distribution completed successfully",
					{
						totalEligible: result.totalEligible,
						successful: result.successful,
						failed: result.failed,
						alreadyDistributed: result.alreadyDistributed,
						totalSalariesDistributed: result.totalSalariesDistributed,
						timestamp: new Date().toISOString(),
					},
				);

				// Re-enable withdrawals after successful completion
				await adminService.enableWithdrawalsAfterCronJob("RANK_SALARY");
				loggerInstance.info(
					"Withdrawals re-enabled after rank salary distribution",
				);
			} catch (error) {
				loggerInstance.error("Monthly rank salary distribution failed:", {
					error: error.message,
					stack: error.stack,
					timestamp: new Date().toISOString(),
				});

				// Re-enable withdrawals even if the job failed
				try {
					await adminService.enableWithdrawalsAfterCronJob("RANK_SALARY");
					loggerInstance.info(
						"Withdrawals re-enabled after failed rank salary distribution",
					);
				} catch (withdrawalError) {
					loggerInstance.error(
						"Failed to re-enable withdrawals after rank salary error:",
						withdrawalError,
					);
				}
			}
		},
		{
			scheduled: true,
			timezone: "UTC",
		},
	);
	loggerInstance.success(
		"Monthly Rank Salary Distribution cron job scheduled",
		{
			schedule: CRON_SCHEDULES.RANK_SALARY || "Not configured",
			priority: 7,
		},
	);

	// Monthly Bonus Distribution
	monthlyBonusTask = cronManager.registerTask(
		"Monthly Bonus Distribution",
		CRON_SCHEDULES.MONTHLY_BONUS,
		async () => {
			loggerInstance.info(
				"Starting scheduled monthly bonus distribution at:",
				new Date().toISOString(),
			);

			try {
				// Disable withdrawals before starting
				await adminService.disableWithdrawalsForCronJob("MONTHLY_BONUS");
				loggerInstance.info(
					"Withdrawals disabled for monthly bonus distribution",
				);

				const result = await executeWithRetry(
					"Monthly Bonus Distribution",
					async () => {
						const {
							bonusService,
						} = require("../../modules/bonuses/bonus.service");
						return await bonusService.distributeMonthlyBonusesToAllUsers();
					},
				);
				loggerInstance.success(
					"Monthly bonus distribution completed successfully",
					{
						totalEligible: result.totalEligible,
						successful: result.successful,
						failed: result.failed,
						alreadyDistributed: result.alreadyDistributed,
						totalBonusesDistributed: result.totalBonusesDistributed,
						timestamp: new Date().toISOString(),
					},
				);

				// Re-enable withdrawals after successful completion
				await adminService.enableWithdrawalsAfterCronJob("MONTHLY_BONUS");
				loggerInstance.info(
					"Withdrawals re-enabled after monthly bonus distribution",
				);
			} catch (error) {
				loggerInstance.error("Monthly bonus distribution failed:", {
					error: error.message,
					stack: error.stack,
					timestamp: new Date().toISOString(),
				});

				// Re-enable withdrawals even if the job failed
				try {
					await adminService.enableWithdrawalsAfterCronJob("MONTHLY_BONUS");
					loggerInstance.info(
						"Withdrawals re-enabled after failed monthly bonus distribution",
					);
				} catch (withdrawalError) {
					loggerInstance.error(
						"Failed to re-enable withdrawals after monthly bonus error:",
						withdrawalError,
					);
				}
			}
		},
	);
	loggerInstance.success("Monthly Bonus Distribution cron job scheduled", {
		schedule: CRON_SCHEDULES.MONTHLY_BONUS || "Not configured",
		priority: 8,
	});

	// Database Backup
	databaseBackupTask = cronManager.registerTask(
		"Database Backup",
		CRON_SCHEDULES.DATABASE_BACKUP,
		async () => {
			loggerInstance.info(
				"Starting scheduled database backup at:",
				new Date().toISOString(),
			);

			try {
				const result = await executeWithRetry(
					"Database Backup",
					createDatabaseBackup,
				);
				loggerInstance.success("Database backup completed successfully", {
					backupPath: result.backupPath,
					originalSize: result.originalSize,
					compressedSize: result.compressedSize,
					compressionRatio: result.compressionRatio,
					fileCount: result.fileCount,
					duration: result.duration,
					timestamp: new Date().toISOString(),
				});
			} catch (error) {
				loggerInstance.error("Database backup failed:", {
					error: error.message,
					stack: error.stack,
					timestamp: new Date().toISOString(),
				});
			}
		},
		{
			scheduled: true,
			timezone: "UTC",
		},
	);
	loggerInstance.success("Database Backup cron job scheduled", {
		schedule: CRON_SCHEDULES.DATABASE_BACKUP || "Not configured",
		priority: 1,
	});

	// PancakeSwap Transaction Tracker
	pancakeswapTrackerTask = cronManager.registerTask(
		"PancakeSwap Transaction Tracker",
		CRON_SCHEDULES.PANCAKESWAP_TRACKER,
		async () => {
			loggerInstance.info(
				"Starting PancakeSwap transaction tracking at:",
				new Date().toISOString(),
			);

			try {
				const result = await executeWithRetry(
					"PancakeSwap Transaction Tracker",
					pancakeswapTransactionService.trackTransactions,
				);

				if (result.success) {
					loggerInstance.success(
						"PancakeSwap tracking completed successfully",
						{
							buys: result.buys,
							sells: result.sells,
							transfers: result.transfers,
							totalTransactions: result.totalTransactions,
							blocksProcessed: result.blocksProcessed,
							fromBlock: result.fromBlock,
							toBlock: result.toBlock,
							timestamp: new Date().toISOString(),
						},
					);
				} else {
					loggerInstance.warn("PancakeSwap tracking completed with issues", {
						message: result.message || result.error,
						timestamp: new Date().toISOString(),
					});
				}
			} catch (error) {
				loggerInstance.error("PancakeSwap tracking failed:", {
					error: error.message,
					stack: error.stack,
					timestamp: new Date().toISOString(),
				});
			}
		},
		{
			scheduled: true,
			timezone: "UTC",
		},
	);
	loggerInstance.success("PancakeSwap Transaction Tracker cron job scheduled", {
		schedule: CRON_SCHEDULES.PANCAKESWAP_TRACKER,
		priority: 10,
		note: "Tracks buys/sells on PancakeSwap every 5 minutes",
	});

	// DBE Daily Verification
	dbeVerificationTask = cronManager.registerTask(
		"DBE Daily Verification",
		CRON_SCHEDULES.DBE_DAILY_VERIFICATION,
		async () => {
			loggerInstance.info(
				"Starting DBE daily verification at:",
				new Date().toISOString(),
			);

			try {
				const result = await executeWithRetry(
					"DBE Daily Verification",
					performDailyDBEVerification,
				);

				if (result.success) {
					loggerInstance.success(
						"DBE daily verification completed successfully",
						{
							processed: result.processed,
							disqualified: result.disqualified,
							date: result.date,
							timestamp: new Date().toISOString(),
						},
					);
				} else {
					loggerInstance.error("DBE daily verification failed", {
						error: result.error,
						timestamp: new Date().toISOString(),
					});
				}
			} catch (error) {
				loggerInstance.error("DBE daily verification error:", {
					error: error.message,
					stack: error.stack,
					timestamp: new Date().toISOString(),
				});
			}
		},
		{
			scheduled: true,
			timezone: "UTC",
		},
	);
	loggerInstance.success("DBE Daily Verification cron job scheduled", {
		schedule: CRON_SCHEDULES.DBE_DAILY_VERIFICATION || "Not configured",
		priority: 11,
		note: "Verifies daily DBE program compliance and disqualifies violators",
	});

	jobsStarted = true;
	bonusJobsStarted = true;
	startingJobs = false; // Reset the starting flag

	// Final summary log
	loggerInstance.success("All cron jobs started successfully!", {
		totalJobs: 11,
		jobsStarted: true,
		bonusJobsStarted: true,
		schedules: {
			databaseBackup: CRON_SCHEDULES.DATABASE_BACKUP || "Not configured",
			rankUpdate: CRON_SCHEDULES.RANK_UPDATE || "Not configured",
			packageOwners: CRON_SCHEDULES.PACKAGE_OWNERS || "Not configured",
			packageUserCounts: CRON_SCHEDULES.PACKAGE_USER_COUNTS || "Not configured",
			usersWithPackages: CRON_SCHEDULES.USERS_WITH_PACKAGES || "Not configured",
			packageExpiration: CRON_SCHEDULES.PACKAGE_EXPIRATION || "Not configured",
			bonusDistribution: CRON_SCHEDULES.BONUS_DISTRIBUTION || "Not configured",
			rankSalary: CRON_SCHEDULES.RANK_SALARY || "Not configured",
			monthlyBonus: CRON_SCHEDULES.MONTHLY_BONUS || "Not configured",
			pancakeswapTracker: CRON_SCHEDULES.PANCAKESWAP_TRACKER || "*/5 * * * *",
			dbeVerification:
				CRON_SCHEDULES.DBE_DAILY_VERIFICATION || "Not configured",
		},
		timezone: "UTC",
		startedAt: new Date().toISOString(),
	});
}

function stopCronJobs() {
	loggerInstance.info("Stopping all cron jobs...");

	if (rankUpdateTask) {
		rankUpdateTask.destroy();
		rankUpdateTask = null;
		loggerInstance.info("Rank Update Task stopped");
	}
	if (bonusDistributionTask) {
		bonusDistributionTask.destroy();
		bonusDistributionTask = null;
		loggerInstance.info("Bonus Distribution Task stopped");
	}
	if (rankSalaryTask) {
		rankSalaryTask.destroy();
		rankSalaryTask = null;
		loggerInstance.info("Rank Salary Task stopped");
	}
	if (monthlyBonusTask) {
		monthlyBonusTask.destroy();
		monthlyBonusTask = null;
		loggerInstance.info("Monthly Bonus Task stopped");
	}
	if (packageUserCountsTask) {
		packageUserCountsTask.destroy();
		packageUserCountsTask = null;
		loggerInstance.info("Package User Counts Task stopped");
	}
	if (usersWithPackagesTask) {
		usersWithPackagesTask.destroy();
		usersWithPackagesTask = null;
		loggerInstance.info("Users With Packages Task stopped");
	}
	if (packageExpirationTask) {
		packageExpirationTask.destroy();
		packageExpirationTask = null;
		loggerInstance.info("Package Expiration Task stopped");
	}
	if (packageOwnersTask) {
		packageOwnersTask.destroy();
		packageOwnersTask = null;
		loggerInstance.info("Package Owners Task stopped");
	}
	if (databaseBackupTask) {
		databaseBackupTask.destroy();
		databaseBackupTask = null;
		loggerInstance.info("Database Backup Task stopped");
	}
	if (pancakeswapTrackerTask) {
		pancakeswapTrackerTask.destroy();
		pancakeswapTrackerTask = null;
		loggerInstance.info("PancakeSwap Tracker Task stopped");
	}
	if (dbeVerificationTask) {
		dbeVerificationTask.destroy();
		dbeVerificationTask = null;
		loggerInstance.info("DBE Verification Task stopped");
	}

	jobsStarted = false;
	bonusJobsStarted = false;
	startingJobs = false;

	loggerInstance.success("All cron jobs stopped successfully!");
}

function getCronJobStatus() {
	const tasks = cron.getTasks();
	const nowUTC = new Date();
	const cronManagerStatus = cronManager.getTaskStatus();

	const getNextExecutionUTC = (schedule) => {
		try {
			if (!schedule) return null;

			const cronFields = schedule.split(" ");
			if (cronFields.length !== 5) return null;

			const [minute, hour, dayOfMonth, month, dayOfWeek] = cronFields;

			let next = new Date(nowUTC.getTime());

			// Handle minute field
			if (minute === "*") {
			} else if (minute.startsWith("*/")) {
				const step = parseInt(minute.split("/")[1]);
				const currentMinute = next.getUTCMinutes();
				const nextMinute = Math.ceil((currentMinute + 1) / step) * step;
				if (nextMinute >= 60) {
					next.setUTCHours(next.getUTCHours() + 1);
					next.setUTCMinutes(0, 0, 0);
				} else {
					next.setUTCMinutes(nextMinute, 0, 0);
				}
			} else if (minute !== "*") {
				const targetMinute = parseInt(minute);
				next.setUTCMinutes(targetMinute, 0, 0);

				if (nowUTC.getUTCMinutes() >= targetMinute) {
					next.setUTCHours(next.getUTCHours() + 1);
				}
			}

			if (hour === "*") {
			} else if (hour.startsWith("*/")) {
				const step = parseInt(hour.split("/")[1]);
				const currentHour = next.getUTCHours();
				const nextHour = Math.ceil((currentHour + 1) / step) * step;
				if (nextHour >= 24) {
					next.setUTCDate(next.getUTCDate() + 1);
					next.setUTCHours(0, next.getUTCMinutes(), 0, 0);
				} else {
					next.setUTCHours(nextHour, next.getUTCMinutes(), 0, 0);
				}
			} else if (hour !== "*") {
				const targetHour = parseInt(hour);
				next.setUTCHours(targetHour, next.getUTCMinutes(), 0, 0);

				if (
					nowUTC.getUTCHours() > targetHour ||
					(nowUTC.getUTCHours() === targetHour &&
						nowUTC.getUTCMinutes() >= next.getUTCMinutes())
				) {
					next.setUTCDate(next.getUTCDate() + 1);
				}
			}

			return {
				utc: next.toISOString(),
				formatted: next
					.toISOString()
					.replace("T", " ")
					.replace(".000Z", " UTC"),
				timeUntil: getTimeUntilExecution(next),
			};
		} catch (error) {
			return null;
		}
	};

	const getTimeUntilExecution = (nextExecution) => {
		const timeDiff = nextExecution.getTime() - nowUTC.getTime();
		if (timeDiff <= 0) return "Now";

		const hours = Math.floor(timeDiff / (1000 * 60 * 60));
		const minutes = Math.floor((timeDiff % (1000 * 60 * 60)) / (1000 * 60));
		const seconds = Math.floor((timeDiff % (1000 * 60)) / 1000);

		if (hours > 0) {
			return `${hours}h ${minutes}m`;
		} else if (minutes > 0) {
			return `${minutes}m ${seconds}s`;
		} else {
			return `${seconds}s`;
		}
	};

	const getRealTimeJobStatus = (taskVariable, taskName) => {
		if (!taskVariable) return "Inactive";

		try {
			if (taskVariable && typeof taskVariable.destroy === "function") {
				return "Active";
			}
			return "Inactive";
		} catch (error) {
			loggerInstance.warn(
				`Task ${taskName} appears to be corrupted:`,
				error.message,
			);
			return "Error";
		}
	};

	return {
		jobsStarted,
		bonusJobsStarted,
		activeTasks: tasks.size,
		totalSystemCronTasks: tasks.size,
		cronManagerStatus,
		isDuplicateDetected: cronManagerStatus.hasDuplicates,
		recommendation: cronManagerStatus.recommendation,
		currentTimeUTC: nowUTC.toISOString(),
		systemTime: new Date().toISOString(),
		jobs: [
			{
				name: "Database Backup",
				schedule: CRON_SCHEDULES.DATABASE_BACKUP || "Not configured",
				status: getRealTimeJobStatus(databaseBackupTask, "Database Backup"),
				nextExecution: getNextExecutionUTC(CRON_SCHEDULES.DATABASE_BACKUP),
				priority: 1,
				taskExists: !!databaseBackupTask,
				isRunning: !!databaseBackupTask,
			},
			{
				name: "Rank Update Job",
				schedule: CRON_SCHEDULES.RANK_UPDATE || "Not configured",
				status: getRealTimeJobStatus(rankUpdateTask, "Rank Update Job"),
				nextExecution: getNextExecutionUTC(CRON_SCHEDULES.RANK_UPDATE),
				priority: 2,
				taskExists: !!rankUpdateTask,
				isRunning: !!rankUpdateTask,
			},
			{
				name: "Package Owners Cache Update",
				schedule: CRON_SCHEDULES.PACKAGE_OWNERS || "Not configured",
				status: getRealTimeJobStatus(
					packageOwnersTask,
					"Package Owners Cache Update",
				),
				nextExecution: getNextExecutionUTC(CRON_SCHEDULES.PACKAGE_OWNERS),
				priority: 3,
				taskExists: !!packageOwnersTask,
				isRunning: !!packageOwnersTask,
			},
			{
				name: "Package User Counts Update",
				schedule: CRON_SCHEDULES.PACKAGE_USER_COUNTS || "Not configured",
				status: getRealTimeJobStatus(
					packageUserCountsTask,
					"Package User Counts Update",
				),
				nextExecution: getNextExecutionUTC(CRON_SCHEDULES.PACKAGE_USER_COUNTS),
				priority: 4,
				taskExists: !!packageUserCountsTask,
				isRunning: !!packageUserCountsTask,
			},
			{
				name: "Users With Active Packages Cache Update",
				schedule: CRON_SCHEDULES.USERS_WITH_PACKAGES || "Not configured",
				status: getRealTimeJobStatus(
					usersWithPackagesTask,
					"Users With Active Packages Cache Update",
				),
				nextExecution: getNextExecutionUTC(CRON_SCHEDULES.USERS_WITH_PACKAGES),
				priority: 4.5,
				taskExists: !!usersWithPackagesTask,
				isRunning: !!usersWithPackagesTask,
			},
			{
				name: "Package Expiration Check",
				schedule: CRON_SCHEDULES.PACKAGE_EXPIRATION || "Not configured",
				status: getRealTimeJobStatus(
					packageExpirationTask,
					"Package Expiration Check",
				),
				nextExecution: getNextExecutionUTC(CRON_SCHEDULES.PACKAGE_EXPIRATION),
				priority: 5,
				taskExists: !!packageExpirationTask,
				isRunning: !!packageExpirationTask,
			},
			{
				name: "Daily Bonus Distribution",
				schedule: CRON_SCHEDULES.BONUS_DISTRIBUTION || "Not configured",
				status: getRealTimeJobStatus(
					bonusDistributionTask,
					"Daily Bonus Distribution",
				),
				nextExecution: getNextExecutionUTC(CRON_SCHEDULES.BONUS_DISTRIBUTION),
				priority: 6,
				taskExists: !!bonusDistributionTask,
				isRunning: !!bonusDistributionTask,
			},
			{
				name: "Monthly Rank Salary Distribution",
				schedule: CRON_SCHEDULES.RANK_SALARY || "Not configured",
				status: getRealTimeJobStatus(
					rankSalaryTask,
					"Monthly Rank Salary Distribution",
				),
				nextExecution: getNextExecutionUTC(CRON_SCHEDULES.RANK_SALARY),
				priority: 7,
				taskExists: !!rankSalaryTask,
				isRunning: !!rankSalaryTask,
			},
			{
				name: "Monthly Bonus Distribution",
				schedule: CRON_SCHEDULES.MONTHLY_BONUS || "Not configured",
				status: getRealTimeJobStatus(
					monthlyBonusTask,
					"Monthly Bonus Distribution",
				),
				nextExecution: getNextExecutionUTC(CRON_SCHEDULES.MONTHLY_BONUS),
				priority: 8,
				taskExists: !!monthlyBonusTask,
				isRunning: !!monthlyBonusTask,
			},
			{
				name: "PancakeSwap Transaction Tracker",
				schedule: CRON_SCHEDULES.PANCAKESWAP_TRACKER || "Not configured",
				status: getRealTimeJobStatus(
					pancakeswapTrackerTask,
					"PancakeSwap Transaction Tracker",
				),
				nextExecution: getNextExecutionUTC(CRON_SCHEDULES.PANCAKESWAP_TRACKER),
				priority: 9,
				taskExists: !!pancakeswapTrackerTask,
				isRunning: !!pancakeswapTrackerTask,
			},
			{
				name: "DBE Daily Verification",
				schedule: CRON_SCHEDULES.DBE_DAILY_VERIFICATION || "Not configured",
				status: getRealTimeJobStatus(
					dbeVerificationTask,
					"DBE Daily Verification",
				),
				nextExecution: getNextExecutionUTC(
					CRON_SCHEDULES.DBE_DAILY_VERIFICATION,
				),
				priority: 11,
				taskExists: !!dbeVerificationTask,
				isRunning: !!dbeVerificationTask,
			},
		],
	};
}

async function getCronJobResults(jobType = null, limit = 10) {
	try {
		const query = {};
		if (jobType) query.jobType = jobType;

		const results = await CronJobLog.find(query)
			.sort({ createdAt: -1 })
			.limit(limit)
			.lean();

		return results.map((result) => ({
			...result,
			_id: result._id.toString(),
		}));
	} catch (error) {
		loggerInstance.error(
			"Failed to get cron job results from database:",
			error,
		);
		return [];
	}
}

async function getCronJobAnalytics(jobType = null, limit = 10) {
	try {
		const query = {};
		if (jobType) query.jobType = jobType;

		const jobResults = await CronJobLog.find(query)
			.sort({ createdAt: -1 })
			.limit(limit)
			.lean();

		const summary = {
			totalJobs: jobResults.length,
			successfulJobs: jobResults.filter((j) => j.isSuccess).length,
			failedJobs: jobResults.filter((j) => !j.isSuccess).length,
			avgDuration:
				jobResults.length > 0
					? jobResults.reduce((sum, j) => sum + (j.durationMs || 0), 0) /
					  jobResults.length
					: 0,
			jobTypes: [...new Set(jobResults.map((j) => j.jobType))],
			lastExecutionTime: jobResults.length > 0 ? jobResults[0].timestamp : null,
			databaseModel: "CronJobLog",
		};

		loggerInstance.info("Generated cron job analytics", {
			totalJobs: summary.totalJobs,
		});

		return {
			summary,
			jobs: jobResults.map((job) => ({
				...job,
				_id: job._id.toString(),
			})),
		};
	} catch (error) {
		loggerInstance.error(
			"Failed to get cron job analytics from database:",
			error,
		);
		return {
			summary: { error: error.message },
			jobs: [],
		};
	}
}

function forceRestartAllCronJobs() {
	loggerInstance.info("Force restarting all cron jobs...");

	// First, completely stop everything
	stopCronJobs();

	// Give it a moment to properly clean up
	setTimeout(() => {
		// Force destroy any remaining tasks
		destroyAllExistingTasks();

		// Reset all flags
		jobsStarted = false;
		bonusJobsStarted = false;
		startingJobs = false;

		// Wait a bit more then start fresh
		setTimeout(() => {
			startCronJobs();
		}, 1000);
	}, 2000);

	loggerInstance.success(
		"Force restart initiated - jobs will restart in 3 seconds",
	);
}

function getSystemCronJobInfo() {
	const tasks = cron?.getTasks() || new Map();
	const systemInfo = {
		totalSystemTasks: tasks.size,
		ourTrackedTasks: {
			packageUserCountsTask: !!packageUserCountsTask,
			usersWithPackagesTask: !!usersWithPackagesTask,
			bonusDistributionTask: !!bonusDistributionTask,
			packageExpirationTask: !!packageExpirationTask,
			packageOwnersTask: !!packageOwnersTask,
			rankUpdateTask: !!rankUpdateTask,
			rankSalaryTask: !!rankSalaryTask,
			monthlyBonusTask: !!monthlyBonusTask,
			databaseBackupTask: !!databaseBackupTask,
			pancakeswapTrackerTask: !!pancakeswapTrackerTask,
		},
		ourActiveCount: [
			packageUserCountsTask,
			usersWithPackagesTask,
			bonusDistributionTask,
			packageExpirationTask,
			packageOwnersTask,
			rankUpdateTask,
			rankSalaryTask,
			monthlyBonusTask,
			databaseBackupTask,
			pancakeswapTrackerTask,
		].filter((task) => task !== null).length,
		flags: {
			jobsStarted,
			bonusJobsStarted,
			startingJobs,
		},
		recommendation: tasks.size > 10 ? "DUPLICATE_DETECTED" : "NORMAL",
	};

	loggerInstance.info("System Cron Job Information:", systemInfo);
	return systemInfo;
}

async function getRankCronJobResults(
	limit = 20,
	page = 1,
	jobType = "",
	status = "",
	sortBy = "startTime",
	sortOrder = "desc",
) {
	try {
		const filter = {};

		if (jobType) {
			filter.jobType = jobType;
		}

		if (status) {
			filter.status = status;
		}

		const sortOptions = {};
		sortOptions[sortBy] = sortOrder === "desc" ? -1 : 1;

		const skip = (page - 1) * limit;

		const [results, total] = await Promise.all([
			RankCronJobResult.find(filter)
				.sort(sortOptions)
				.skip(skip)
				.limit(limit)
				.lean(),
			RankCronJobResult.countDocuments(filter),
		]);

		loggerInstance.info("Retrieved rank cron job results", {
			total,
			page,
			limit,
			filters: filter,
		});

		return {
			results: results.map((result) => ({
				...result,
				_id: result._id.toString(),
			})),
			pagination: {
				total,
				page,
				limit,
				totalPages: Math.ceil(total / limit),
				hasNextPage: page < Math.ceil(total / limit),
				hasPrevPage: page > 1,
			},
		};
	} catch (error) {
		loggerInstance.error("Failed to get rank cron job results:", error);
		throw error;
	}
}

module.exports = {
	startCronJobs,
	stopCronJobs,
	getCronJobStatus,
	getHealthStatus,
	getCronJobResults,
	getCronJobAnalytics,
	forceRestartAllCronJobs,
	getSystemCronJobInfo,
	getRankCronJobResults,
};
