const cron = require("node-cron");
const { logger } = require("../logger/logger");

let cronManagerInstance = null;

class CronManager {
	constructor() {
		this.activeTasks = new Map();
		this.taskRegistry = new Map();
		this.isInitialized = false;
	}

	static getInstance() {
		if (!cronManagerInstance) {
			cronManagerInstance = new CronManager();
		}
		return cronManagerInstance;
	}

	// Get total count of all cron tasks in the system
	getTotalSystemTasks() {
		return cron.getTasks().size;
	}

	// Destroy ALL existing cron tasks in the system
	destroyAllTasks() {
		logger.info("CronManager: Destroying all existing cron tasks...");

		// Get all tasks from node-cron
		const allTasks = cron.getTasks();
		logger.info(`Found ${allTasks.size} total system cron tasks`);

		// Destroy all tasks
		allTasks.forEach((task, key) => {
			try {
				task.destroy();
				logger.info(`Destroyed task: ${key}`);
			} catch (error) {
				logger.warn(`Failed to destroy task ${key}:`, error.message);
			}
		});

		// Clear our internal registry
		this.activeTasks.clear();
		this.taskRegistry.clear();

		logger.info("All cron tasks destroyed successfully!");
		return true;
	}

	// Register a new cron task with the manager
	registerTask(taskName, schedule, taskFunction, options = {}) {
		// If task already exists, destroy it first
		if (this.activeTasks.has(taskName)) {
			logger.info(
				`Task ${taskName} already exists, destroying old instance...`,
			);
			this.destroyTask(taskName);
		}

		try {
			const task = cron.schedule(schedule, taskFunction, {
				scheduled: true,
				timezone: "UTC",
				...options,
			});

			this.activeTasks.set(taskName, task);
			this.taskRegistry.set(taskName, {
				schedule,
				options,
				registeredAt: new Date(),
				status: "active",
			});

			logger.info(
				`Registered cron task: ${taskName} with schedule: ${schedule}`,
			);
			return task;
		} catch (error) {
			logger.error(`Failed to register task ${taskName}:`, error);
			return null;
		}
	}

	// Destroy a specific task
	destroyTask(taskName) {
		const task = this.activeTasks.get(taskName);
		if (task) {
			try {
				task.destroy();
				this.activeTasks.delete(taskName);

				// Update registry
				if (this.taskRegistry.has(taskName)) {
					const registry = this.taskRegistry.get(taskName);
					registry.status = "destroyed";
					registry.destroyedAt = new Date();
				}

				logger.info(`Destroyed task: ${taskName}`);
				return true;
			} catch (error) {
				logger.error(`Failed to destroy task ${taskName}:`, error);
				return false;
			}
		}
		return false;
	}

	// Destroy all our registered tasks
	destroyAllRegisteredTasks() {
		logger.info("Destroying all registered tasks...");

		this.activeTasks.forEach((task, taskName) => {
			this.destroyTask(taskName);
		});

		this.activeTasks.clear();
		logger.info("All registered tasks destroyed");
	}

	// Get status of all registered tasks
	getTaskStatus() {
		const systemTasks = this.getTotalSystemTasks();
		const ourTasks = this.activeTasks.size;

		const status = {
			totalSystemTasks: systemTasks,
			ourRegisteredTasks: ourTasks,
			registeredTaskNames: Array.from(this.activeTasks.keys()),
			hasDuplicates: systemTasks > ourTasks,
			recommendation: systemTasks > ourTasks ? "CLEANUP_NEEDED" : "HEALTHY",
			tasks: {},
		};

		this.taskRegistry.forEach((info, taskName) => {
			status.tasks[taskName] = {
				...info,
				isActive: this.activeTasks.has(taskName),
			};
		});

		return status;
	}

	// Force cleanup all tasks and reset
	forceReset() {
		logger.info("CronManager: Force resetting all cron tasks...");

		// First destroy all system tasks
		this.destroyAllTasks();

		// Wait a moment for cleanup
		return new Promise((resolve) => {
			setTimeout(() => {
				this.isInitialized = false;
				logger.info("CronManager reset complete");
				resolve(true);
			}, 1000);
		});
	}

	// Check if we should prevent starting new tasks (if duplicates detected)
	shouldPreventDuplicates() {
		const systemTasks = this.getTotalSystemTasks();
		const ourTasks = this.activeTasks.size;

		if (systemTasks > ourTasks && ourTasks > 0) {
			logger.warn(
				`Duplicate detection: ${systemTasks} system tasks vs ${ourTasks} our tasks`,
			);
			return true;
		}

		return false;
	}
}

module.exports = CronManager;
