const { cacheService } = require("../cache/cache.service");
const { logger } = require("../logger/logger");

class SchedulerService {
	constructor() {
		this.jobs = new Map();
		this.intervals = new Map();
	}

	register(name, handler, intervalMs = 5 * 60 * 1000, lockTtl = 5 * 60) {
		if (this.jobs.has(name)) {
			logger.warn("Job already registered", { job: name });
			return;
		}

		const runner = async () => {
			const lockKey = `scheduler-lock:${name}`;

			// Atomic lock acquisition
			const acquired = await cacheService.setNX(lockKey, true, lockTtl);
			if (!acquired) return;

			const startTime = Date.now();
			try {
				await handler();
				logger.debug("Scheduler job completed", {
					job: name,
					durationMs: Date.now() - startTime,
				});
			} catch (err) {
				logger.error("Scheduler job failed", {
					job: name,
					error: err.message,
					stack: err.stack,
				});
			} finally {
				await cacheService.delete(lockKey);
			}
		};

		// Delay first run to avoid race with interval
		const firstRunTimeout = setTimeout(runner, 1000);
		const intervalId = setInterval(runner, intervalMs);

		this.jobs.set(name, { intervalMs, handler });
		this.intervals.set(name, { intervalId, firstRunTimeout });

		logger.info("Scheduler job registered", { job: name, intervalMs });
	}

	stop(name) {
		const timers = this.intervals.get(name);
		if (timers) {
			clearInterval(timers.intervalId);
			clearTimeout(timers.firstRunTimeout);
			this.intervals.delete(name);
			this.jobs.delete(name);
			logger.info("Scheduler job stopped", { job: name });
		}
	}

	stopAll() {
		for (const name of this.intervals.keys()) {
			this.stop(name);
		}
		logger.info("All scheduler jobs stopped");
	}

	getStatus() {
		return Array.from(this.jobs.entries()).map(([name, config]) => ({
			name,
			intervalMs: config.intervalMs,
		}));
	}
}

const schedulerService = new SchedulerService();

module.exports = { schedulerService };
