const crypto = require("crypto");
const { rsaService } = require("./rsa.service");
const { nonceService } = require("./nonce.service");
const { ApiError } = require("../../utils/ApiError");

const hashPayload = (body) => {
	const normalized =
		typeof body === "string" ? body : JSON.stringify(body || {});
	return crypto.createHash("sha256").update(normalized).digest("hex");
};

const buildSigningString = ({ method, path, payloadHash, nonce, timestamp }) =>
	[method.toUpperCase(), path, payloadHash, nonce, timestamp].join(":");

const verifyRequestSignature = async (req) => {
	const signature = req.headers["x-signature"];
	const nonce = req.headers["x-nonce"];
	const timestamp = req.headers["x-timestamp"];

	if (!signature || !nonce || !timestamp) {
		throw new ApiError(400, "Missing signature headers");
	}

	const payloadHash = hashPayload(req.body);
	const signingString = buildSigningString({
		method: req.method,
		path: req.originalUrl,
		payloadHash,
		nonce,
		timestamp,
	});

	const isValid = rsaService.verify(signingString, signature);
	if (!isValid) {
		throw new ApiError(401, "Invalid signature");
	}

	const userId = req.user ? req.user._id : null;
	await nonceService.validateAndStore({
		nonce,
		userId,
		timestamp,
	});

	return { payloadHash, nonce, timestamp };
};

module.exports = {
	signatureService: { verifyRequestSignature, hashPayload, buildSigningString },
};
