const jwt = require("jsonwebtoken");
const { jwtConfig } = require("../config/jwt.config");
const { ApiError } = require("../utils/ApiError");
const { logger } = require("../core/logger/logger");
const { cacheService } = require("../core/cache/cache.service");

const authMiddleware = async (req, res, next) => {
	try {
		let token;

		// Check Authorization header first
		const authHeader = req.headers.authorization;
		if (authHeader) {
			const parts = authHeader.split(" ");
			if (parts.length !== 2 || parts[0] !== "Bearer")
				throw new ApiError(401, "Invalid authorization format");
			token = parts[1];
		}

		// Fallback to cookie if no Authorization header
		if (!token && req.cookies && req.cookies.auth_token) {
			token = req.cookies.auth_token;
		}

		// If still no token found
		if (!token) throw new ApiError(401, "Token missing");

		// Check if token is blacklisted (logged out)
		const isBlacklisted = await cacheService.exists(`token:blacklist:${token}`);
		if (isBlacklisted) {
			logger.warn("Blacklisted token used", {
				requestId: req.requestId,
				ip: req.clientIp,
			});
			throw new ApiError(401, "Token has been revoked");
		}

		// Check if token is still active (single-device enforcement)
		const isActive = await cacheService.exists(`token:active:${token}`);
		if (!isActive) {
			logger.warn("Inactive token used (logged in from another device)", {
				requestId: req.requestId,
				ip: req.clientIp,
			});
			throw new ApiError(401, "Session expired. Please login again.", {
				code: "SESSION_REPLACED",
				message: "You have been logged out because you logged in from another device.",
			});
		}

		// Verify with RSA public key (RS256)
		const payload = jwt.verify(token, jwtConfig.publicKey, {
			algorithms: [jwtConfig.algorithm],
			issuer: jwtConfig.issuer,
			audience: jwtConfig.audience,
		});

		req.user = {
			_id: payload._id,
			role: payload.role || "user",
			email: payload.email,
			username: payload.username,
			walletAddress: payload.walletAddress,
		};
		req.token = token; // Store token for logout

		next();
	} catch (err) {
		if (err.name === "TokenExpiredError") {
			logger.warn("Token expired", {
				requestId: req.requestId,
				ip: req.clientIp,
			});
			return next(new ApiError(401, "Token expired"));
		}
		if (err.name === "JsonWebTokenError") {
			logger.warn("Invalid token", {
				requestId: req.requestId,
				ip: req.clientIp,
			});
			return next(new ApiError(401, "Invalid token"));
		}
		if (err instanceof ApiError) return next(err);
		logger.error("Auth error", {
			error: err.message,
			requestId: req.requestId,
		});
		next(new ApiError(401, "Authentication failed"));
	}
};

module.exports = { authMiddleware };
