const { logger } = require("../core/logger/logger");
const { ApiError } = require("../utils/ApiError");

const errorMiddleware = (err, req, res, next) => {
	let statusCode =
		err instanceof ApiError
			? err.statusCode
			: err.status || err.statusCode || 500;
	if (statusCode < 100 || statusCode > 599) statusCode = 500;

	const errorResponse = {
		error: { message: err.message || "Internal Server Error", code: err.code },
		requestId: req.requestId,
		timestamp: new Date().toISOString(),
	};

	if (statusCode >= 400 && statusCode < 500 && err.details)
		errorResponse.error.details = err.details;

	const userId = req.user ? req.user._id : null;
	const ctx = {
		requestId: req.requestId,
		userId,
		method: req.method,
		path: req.originalUrl,
		statusCode,
	};

	if (statusCode >= 500)
		logger.error("Server error", { ...ctx, stack: err.stack });
	else if (statusCode >= 400) logger.warn("Client error", ctx);

	if (process.env.NODE_ENV === "production" && statusCode >= 500) {
		errorResponse.error.message = "Internal Server Error";
		delete errorResponse.error.details;
	} else if (process.env.NODE_ENV !== "production") {
		errorResponse.error.stack = err.stack;
	}

	res.status(statusCode).json(errorResponse);
};

module.exports = { errorMiddleware };
