const { ApiError } = require("../utils/ApiError");
const { logger } = require("../core/logger/logger");

// Role-based access control with hierarchy - admin(3) > support(2) > user(1)
// requireRole('user') => user, support, admin can access (everyone)
// requireRole('support') => support, admin can access (no user)
// requireRole('admin') => only admin can access (highest level only)

const ROLE_HIERARCHY = { admin: 3, support: 2, user: 1 };

const requireRole = (role = "user") => {
	if (!ROLE_HIERARCHY[role]) {
		throw new Error(
			`Invalid role "${role}". Must be: ${Object.keys(ROLE_HIERARCHY).join(
				", ",
			)}`,
		);
	}

	const minLevel = ROLE_HIERARCHY[role];

	return (req, res, next) => {
		if (!req.user) {
			logger.error("requireRole: No user context", {
				requestId: req.requestId,
			});
			return next(new ApiError(401, "Authentication required"));
		}

		if (!req.user.role || typeof req.user.role !== "string") {
			logger.warn("Role check failed: No role", {
				userId: req.user._id,
				requestId: req.requestId,
			});
			return next(new ApiError(403, "Access denied: No role assigned"));
		}

		const userRole = req.user.role.toLowerCase();
		const userLevel = ROLE_HIERARCHY[userRole];

		if (userLevel === undefined) {
			logger.warn("Role check failed: Invalid role", {
				userId: req.user._id,
				userRole,
				requestId: req.requestId,
			});
			return next(new ApiError(403, "Access denied: Invalid role"));
		}

		if (userLevel < minLevel) {
			logger.warn("Access denied", {
				userId: req.user._id,
				userRole,
				required: role,
				requestId: req.requestId,
			});
			return next(new ApiError(403, "Access denied: Insufficient permissions"));
		}

		if (minLevel >= ROLE_HIERARCHY.support) {
			logger.info("Privileged access", {
				userId: req.user._id,
				userRole,
				endpoint: req.originalUrl,
				requestId: req.requestId,
			});
		}

		next();
	};
};

module.exports = { requireRole, ROLE_HIERARCHY };
