const { authMiddleware } = require("./auth.middleware");
const { signatureMiddleware } = require("./signature.middleware");
const {
	rateLimitMiddleware,
} = require("../core/rate-limit/rateLimit.middleware");
const { requestIdMiddleware } = require("./requestId.middleware");
const { ipMiddleware } = require("./ip.middleware");
const { originMiddleware } = require("./origin.middleware");
const { sessionMiddleware } = require("../core/session/session.middleware");
const { requireRole } = require("./roles.middleware");

// Security middleware builder - Use flags to compose middleware stack
// Flags: P=IP, R=RateLimit, I=RequestId, S=Signature, O=Origin, N=Session, C=Context(all PION), A=Auth
// Usage: secure('CAR') => Context + Auth + RateLimit
// Usage: secure('CARS', { role: 'admin', rateLimitPolicy: 'sensitive' }) => Full security with admin access

const parseFlags = (flags = "") => {
	const f = new Set(flags.toUpperCase().split(""));
	const hasContext = f.has("C");
	return {
		requestId: f.has("I") || hasContext,
		ip: f.has("P") || hasContext,
		origin: f.has("O") || hasContext,
		session: f.has("N") || hasContext,
		auth: f.has("A"),
		rateLimit: f.has("R"),
		signature: f.has("S"),
	};
};

const secure = (flags = "", options = {}) => {
	const { rateLimitPolicy = "default", role } = options;
	const parsed = parseFlags(flags);
	const stack = [];

	if (parsed.requestId) stack.push(requestIdMiddleware);
	if (parsed.ip) stack.push(ipMiddleware);
	if (parsed.origin) stack.push(originMiddleware);
	if (parsed.session) stack.push(sessionMiddleware);
	if (parsed.auth) stack.push(authMiddleware);
	if (role) stack.push(requireRole(role));
	if (parsed.rateLimit) stack.push(rateLimitMiddleware(rateLimitPolicy));
	if (parsed.signature) stack.push(signatureMiddleware);

	return stack;
};

module.exports = {
	secure,
	requireRole,
};
