const { adminService } = require("./admin.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");
const { User } = require("../users/user.model");
const { createWalletHash } = require("../users/user.service");

const adminController = {
	async getMasterPasswordInfo(req, res, next) {
		try {
			const info = await adminService.getMasterPasswordInfo();
			return res.json(
				new ApiResponse(200, info, "Master password info retrieved"),
			);
		} catch (error) {
			next(error);
		}
	},

	async setMasterPasswordFirstTime(req, res, next) {
		try {
			const { newPassword, walletAddress } = req.body;

			const walletHash = createWalletHash(walletAddress);

			if (!newPassword || newPassword.length < 8) {
				throw new ApiError(400, "Password must be at least 8 characters");
			}

			const user = await User.findOne({
				walletHash,
			});

			if (!user || user.role !== "admin") {
				throw new ApiError(403, "Only admin users can set the master password");
			}

			const result = await adminService.setMasterPasswordFirstTime(
				newPassword,
				user.username,
			);
			return res.json(
				new ApiResponse(200, result, "Master password set successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async updateMasterPassword(req, res, next) {
		try {
			const { currentPassword, newPassword } = req.body;
			const username = req.user.username;

			const result = await adminService.updateMasterPassword(
				currentPassword,
				newPassword,
				username,
			);
			return res.json(
				new ApiResponse(200, result, "Master password updated successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async verifyAccess(req, res, next) {
		try {
			const { walletAddress, masterPassword } = req.body;
			const ipAddress = req.ip || req.connection.remoteAddress;
			const userAgent = req.headers["user-agent"] || "Unknown";

			const result = await adminService.verifyAccess(
				walletAddress,
				masterPassword,
				ipAddress,
				userAgent,
			);

			return res.json(new ApiResponse(200, result, "Admin access granted"));
		} catch (error) {
			next(error);
		}
	},

	async validateSession(req, res, next) {
		try {
			const { sessionToken, walletAddress } = req.body;
			const ipAddress = req.ip || req.connection.remoteAddress;
			const userAgent = req.headers["user-agent"] || "Unknown";

			if (!sessionToken || !walletAddress) {
				throw new ApiError(
					400,
					"Session token and wallet address are required",
				);
			}

			const result = await adminService.validateSession(
				sessionToken,
				walletAddress,
				ipAddress,
				userAgent,
			);

			if (!result.valid) {
				return res.status(401).json(new ApiResponse(401, result, result.error));
			}

			return res.json(new ApiResponse(200, result, "Session validated"));
		} catch (error) {
			next(error);
		}
	},

	async revokeSession(req, res, next) {
		try {
			const { sessionToken } = req.body;

			await adminService.revokeSession(sessionToken);
			return res.json(new ApiResponse(200, {}, "Session revoked successfully"));
		} catch (error) {
			next(error);
		}
	},

	async getActiveSessions(req, res, next) {
		try {
			const { walletAddress } = req.query;

			const sessions = await adminService.getActiveSessions(walletAddress);
			return res.json(
				new ApiResponse(
					200,
					{ sessions, total: sessions.length },
					"Active sessions retrieved",
				),
			);
		} catch (error) {
			next(error);
		}
	},

	async cleanupExpiredSessions(req, res, next) {
		try {
			const result = await adminService.cleanupExpiredSessions();
			return res.json(
				new ApiResponse(200, result, "Expired sessions cleaned up"),
			);
		} catch (error) {
			next(error);
		}
	},

	async getSystemSettings(req, res, next) {
		try {
			const result = await adminService.getSystemSettings();
			return res.json(
				new ApiResponse(200, result, "System settings retrieved"),
			);
		} catch (error) {
			next(error);
		}
	},

	async updateSystemSettings(req, res, next) {
		try {
			const updates = req.body;
			const username = req.user.username;

			const result = await adminService.updateSystemSettings(updates, username);
			return res.json(
				new ApiResponse(200, result, "System settings updated successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async getSystemConfig(req, res, next) {
		try {
			const { type } = req.query;

			const result = await adminService.getSystemConfig(type);
			return res.json(new ApiResponse(200, result, "System config retrieved"));
		} catch (error) {
			next(error);
		}
	},

	async blockUser(req, res, next) {
		try {
			const { userId } = req.params;
			const { reason } = req.body;
			const adminUsername = req.user.username;

			const result = await adminService.blockUser(
				userId,
				reason,
				adminUsername,
			);
			return res.json(
				new ApiResponse(200, result, "User blocked successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async unblockUser(req, res, next) {
		try {
			const { userId } = req.params;
			const adminUsername = req.user.username;

			const result = await adminService.unblockUser(userId, adminUsername);
			return res.json(
				new ApiResponse(200, result, "User unblocked successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async disableUserWithdrawals(req, res, next) {
		try {
			const { userId } = req.params;
			const { reason } = req.body;
			const adminUsername = req.user.username;

			const result = await adminService.disableUserWithdrawals(
				userId,
				reason,
				adminUsername,
			);
			return res.json(
				new ApiResponse(200, result, "User withdrawals disabled"),
			);
		} catch (error) {
			next(error);
		}
	},

	async enableUserWithdrawals(req, res, next) {
		try {
			const { userId } = req.params;
			const adminUsername = req.user.username;

			const result = await adminService.enableUserWithdrawals(
				userId,
				adminUsername,
			);
			return res.json(new ApiResponse(200, result, "User withdrawals enabled"));
		} catch (error) {
			next(error);
		}
	},

	async getOverrides(req, res, next) {
		try {
			const { action, limit = 50, daysOld = 30 } = req.query;

			const result = await adminService.getOverrides(
				action,
				parseInt(limit),
				parseInt(daysOld),
			);
			return res.json(new ApiResponse(200, result, "Overrides retrieved"));
		} catch (error) {
			next(error);
		}
	},

	async setOverride(req, res, next) {
		try {
			const { type, value, durationMinutes, reason } = req.body;
			const username = req.user.username;

			const result = await adminService.setOverride(
				type,
				value,
				durationMinutes,
				reason,
				username,
			);
			return res.json(
				new ApiResponse(200, result, "Override set successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async deactivateOverride(req, res, next) {
		try {
			const { overrideId } = req.params;
			const { reason } = req.body;
			const username = req.user.username;

			const result = await adminService.deactivateOverride(
				overrideId,
				reason,
				username,
			);
			return res.json(
				new ApiResponse(200, result, "Override deactivated successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async deleteOverride(req, res, next) {
		try {
			const { overrideId } = req.params;

			await adminService.deleteOverride(overrideId);
			return res.json(
				new ApiResponse(200, {}, "Override deleted successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async getDashboardStats(req, res, next) {
		try {
			const stats = await adminService.getDashboardStats();
			return res.json(new ApiResponse(200, stats, "Dashboard stats retrieved"));
		} catch (error) {
			next(error);
		}
	},

	async getAllUsers(req, res, next) {
		try {
			const {
				page = 1,
				limit = 20,
				search,
				role,
				isBlocked,
				sortBy = "createdAt",
				sortOrder = "desc",
			} = req.query;

			const result = await adminService.getAllUsers({
				page: parseInt(page),
				limit: parseInt(limit),
				search,
				role,
				isBlocked:
					isBlocked === "true"
						? true
						: isBlocked === "false"
							? false
							: undefined,
				sortBy,
				sortOrder,
			});

			return res.json(
				new ApiResponse(200, result, "Users retrieved successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async getUserDetails(req, res, next) {
		try {
			const { userId } = req.params;

			const user = await adminService.getUserDetails(userId);
			return res.json(new ApiResponse(200, { user }, "User details retrieved"));
		} catch (error) {
			next(error);
		}
	},

	async updateUserRole(req, res, next) {
		try {
			const { userId } = req.params;
			const { role } = req.body;
			const adminUsername = req.user.username;

			const result = await adminService.updateUserRole(
				userId,
				role,
				adminUsername,
			);
			return res.json(
				new ApiResponse(200, result, "User role updated successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async updateUserStatus(req, res, next) {
		try {
			const { userId } = req.params;
			const { isActive } = req.body;
			const adminUsername = req.user.username;

			const result = await adminService.updateUserStatus(
				userId,
				isActive,
				adminUsername,
			);
			return res.json(
				new ApiResponse(200, result, "User status updated successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async adjustUserWalletBalance(req, res, next) {
		try {
			const { userId } = req.params;
			const { amount, reason } = req.body;
			const adminUsername = req.user.username;

			const result = await adminService.adjustUserWalletBalance(
				userId,
				parseFloat(amount),
				reason,
				adminUsername,
			);
			return res.json(
				new ApiResponse(200, result, "Wallet balance adjusted successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async recordManualBalanceAdjustment(req, res, next) {
		try {
			const { adjustments, masterPassword } = req.body;
			const adminUsername = req.user.username;

			// Validate master password
			if (!masterPassword || typeof masterPassword !== "string") {
				throw new ApiError(400, "Master password is required");
			}

			const results = await adminService.processMultipleAdjustments(
				adjustments,
				masterPassword,
				adminUsername,
			);

			return res.json(
				new ApiResponse(
					200,
					results,
					`Successfully processed ${results.totalSuccessful} out of ${results.totalProcessed} adjustments`,
				),
			);
		} catch (error) {
			next(error);
		}
	},

	async getManualAdjustmentHistory(req, res, next) {
		try {
			const {
				username,
				adminUsername,
				balanceType,
				adjustmentType,
				startDate,
				endDate,
				page = 1,
				limit = 20,
			} = req.query;

			const result = await adminService.getManualAdjustmentHistory({
				username,
				adminUsername,
				balanceType,
				adjustmentType,
				startDate,
				endDate,
				page: parseInt(page),
				limit: parseInt(limit),
			});

			return res.json(
				new ApiResponse(
					200,
					result,
					"Adjustment history retrieved successfully",
				),
			);
		} catch (error) {
			next(error);
		}
	},

	async flagTransactionForRefund(req, res, next) {
		try {
			const { transactionId, flagged, flagReason } = req.body;

			if (!transactionId) {
				throw new ApiError(400, "Transaction ID is required");
			}

			const result = await adminService.flagTransactionForRefund(
				transactionId,
				flagged,
				flagReason,
			);

			return res.json(
				new ApiResponse(200, result, "Transaction updated successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async getFlaggedTransactions(req, res, next) {
		try {
			const { username } = req.query;

			if (!username) {
				throw new ApiError(400, "Username is required");
			}

			const transactions = await adminService.getFlaggedTransactions(username);

			return res.json(
				new ApiResponse(
					200,
					{ transactions },
					"Flagged transactions retrieved successfully",
				),
			);
		} catch (error) {
			next(error);
		}
	},

	async markTransactionAsRefunded(req, res, next) {
		try {
			const { transactionId, refundTxHash, refundTxInfo } = req.body;

			if (!transactionId) {
				throw new ApiError(400, "Transaction ID is required");
			}

			const result = await adminService.markTransactionAsRefunded(
				transactionId,
				refundTxHash,
				refundTxInfo,
			);

			return res.json(
				new ApiResponse(200, result, "Transaction marked as refunded"),
			);
		} catch (error) {
			next(error);
		}
	},

	async getCompanyWallets(req, res, next) {
		try {
			const wallets = await adminService.getCompanyWallets();

			return res.json(
				new ApiResponse(200, wallets, "Company wallets retrieved successfully"),
			);
		} catch (error) {
			next(error);
		}
	},

	async getDynamicPricingStatus(req, res, next) {
		try {
			const { productId } = req.params;

			if (!productId) {
				throw new ApiError(400, "Product ID is required");
			}

			const result = await adminService.getDynamicPricingStatus(productId);

			return res.json(
				new ApiResponse(
					200,
					result,
					"Dynamic pricing status retrieved successfully",
				),
			);
		} catch (error) {
			next(error);
		}
	},

	async getPassengerLogs(req, res, next) {
		try {
			const fs = require("fs").promises;
			const path = require("path");

			const action = req.query.action || "get";
			const linesParam = parseInt(req.query.lines) || 500;

			const filePath = path.join(process.cwd(), "logs", "passenger.log");

			// Check if file exists
			const stat = await fs.stat(filePath).catch(() => null);
			if (!stat) {
				throw new ApiError(404, "Log file not found");
			}

			const content = await fs.readFile(filePath, "utf8");

			// Handle download action
			if (action === "download") {
				res.setHeader("Content-Type", "text/plain; charset=utf-8");
				res.setHeader(
					"Content-Disposition",
					"attachment; filename=passenger.log",
				);
				return res.send(content);
			}

			// Default: return last N lines as JSON
			const allLines = content.split(/\r?\n/);
			const totalLines = allLines.length;
			const start = Math.max(0, totalLines - linesParam);
			const tailLines = allLines.slice(start);

			return res.json(
				new ApiResponse(
					200,
					{
						totalLines,
						lines: tailLines,
						fileSize: stat.size,
					},
					"Logs retrieved successfully",
				),
			);
		} catch (error) {
			next(error);
		}
	},

	async getWithdrawalConfig(req, res, next) {
		try {
			const user = req.user;
			const config = await adminService.getWithdrawalConfig(user);
			return res.json(
				new ApiResponse(
					200,
					{ data: config },
					"Withdrawal configuration retrieved successfully",
				),
			);
		} catch (error) {
			next(error);
		}
	},
};

module.exports = { adminController };
