const express = require('express');
const router = express.Router();
const { adminController } = require('./admin.controller');
const { secure } = require('../../middlewares/security.middleware');

// Public endpoint - get password info
router.get('/config/master-password', adminController.getMasterPasswordInfo);

// First time setup
router.post('/config/master-password/setup', adminController.setMasterPasswordFirstTime);

// Update master password
router.put('/config/master-password', ...secure('AR', { role: 'admin' }), adminController.updateMasterPassword);

// Verify admin access with master password
router.post('/session/verify', adminController.verifyAccess);

router.post('/session/validate', adminController.validateSession);

router.post('/session/revoke', ...secure('AR', { role: 'admin' }), adminController.revokeSession);

router.get('/sessions', ...secure('AR', { role: 'admin' }), adminController.getActiveSessions);

router.delete('/sessions/cleanup', ...secure('AR', { role: 'admin' }), adminController.cleanupExpiredSessions);

// System settings
router.get('/system/settings', ...secure('AR', { role: 'admin' }), adminController.getSystemSettings);
router.put('/system/settings', ...secure('AR', { role: 'admin' }), adminController.updateSystemSettings);

// Public config endpoint for maintenance checks
router.get('/system/config', adminController.getSystemConfig);

// User blocking and withdrawal controls
router.post('/users/:userId/block', ...secure('AR', { role: 'admin' }), adminController.blockUser);
router.post('/users/:userId/unblock', ...secure('AR', { role: 'admin' }), adminController.unblockUser);
router.post('/users/:userId/disable-withdrawals', ...secure('AR', { role: 'admin' }), adminController.disableUserWithdrawals);
router.post('/users/:userId/enable-withdrawals', ...secure('AR', { role: 'admin' }), adminController.enableUserWithdrawals);

// Overrides management
router.get('/overrides', ...secure('AR', { role: 'admin' }), adminController.getOverrides);
router.post('/overrides', ...secure('AR', { role: 'admin' }), adminController.setOverride);
router.put('/overrides/:overrideId/deactivate', ...secure('AR', { role: 'admin' }), adminController.deactivateOverride);
router.delete('/overrides/:overrideId', ...secure('AR', { role: 'admin' }), adminController.deleteOverride);

// Dashboard statistics
router.get('/dashboard/stats', ...secure('AR', { role: 'admin' }), adminController.getDashboardStats);

// User management - get all with filters
router.get('/users', ...secure('AR', { role: 'admin' }), adminController.getAllUsers);
router.get('/users/:userId', ...secure('AR', { role: 'admin' }), adminController.getUserDetails);
router.put('/users/:userId/role', ...secure('AR', { role: 'admin' }), adminController.updateUserRole);
router.put('/users/:userId/status', ...secure('AR', { role: 'admin' }), adminController.updateUserStatus);
router.post('/users/:userId/wallet/adjust', ...secure('AR', { role: 'admin' }), adminController.adjustUserWalletBalance);

// Manual balance adjustments
router.post('/balance-adjustments', ...secure('AR', { role: 'admin' }), adminController.recordManualBalanceAdjustment);
router.get('/balance-adjustments/history', ...secure('AR', { role: 'admin' }), adminController.getManualAdjustmentHistory);

// Company wallet addresses
router.get('/company-wallets', ...secure('AR'), adminController.getCompanyWallets);

// Dynamic pricing status
router.get('/products/:productId/pricing-status', ...secure('AR', { role: 'admin' }), adminController.getDynamicPricingStatus);

// Passenger logs (view and download)
router.get('/logs/passenger', ...secure('AR', { role: 'admin' }), adminController.getPassengerLogs);

// claim configuration
router.get('/claim/config', ...secure('AR'), adminController.getWithdrawalConfig);

module.exports = { adminRoutes: router };
