const { AdminConfig } = require("./adminConfig.model");
const { AdminOverride } = require("./adminOverride.model");
const { AdminSession } = require("./adminSession.model");
const { AdminSystemSettings } = require("./adminSystemSettings.model");
const { User } = require("../users/user.model");
const { WalletTransaction } = require("../wallets/walletTransaction.model");
const { Product } = require("../products/product.model");
const bcrypt = require("bcryptjs");
const crypto = require("crypto");
const { ApiError } = require("../../utils/ApiError");
const { createWalletHash } = require("../users/user.service");
const systemNotificationService = require("../../utils/system-notification.util");
const generateEmailFromTemplate = require("../../templates/generateEmailFromTemplate");
const {
	recordComprehensiveTransaction,
} = require("../purchase/services/transactionTracking.service");
const {
	getTokenValue,
	getTotalTaxEarned,
	getTotalUsers,
} = require("../config/globalConfig.service");
const { logger } = require("../../core/logger/logger");
const {
	verifyNTETransfer,
} = require("../purchase/services/blockchainVerification.service");
const UserPackage = require("../purchase/userPackage.model");
const { CLAIM_LIMITS } = require("../../utils/constants");
const { Withdrawal } = require("../withdrawals/withdrawal.model");

const adminService = {
	async getMasterPasswordInfo() {
		const config = await AdminConfig.findOne({ type: "master_password" });

		if (!config) {
			return {
				type: "master_password",
				note: "Not configured",
				lastUpdated: null,
				hasPassword: false,
				isFirstTime: true,
			};
		}

		return {
			type: config.type,
			note: config.note,
			lastUpdated: config.updatedAt,
			hasPassword: !!config.passwordHash,
			isFirstTime: false,
		};
	},

	async setMasterPasswordFirstTime(newPassword, setBy) {
		const existing = await AdminConfig.findOne({ type: "master_password" });

		if (existing && existing.passwordHash) {
			throw new ApiError(
				400,
				"Master password already exists. Use update endpoint instead.",
			);
		}

		const hashedPassword = await bcrypt.hash(newPassword, 12);

		if (existing) {
			existing.passwordHash = hashedPassword;
			existing.note = `Initially set by ${setBy} on ${new Date().toISOString()}`;
			await existing.save();
			return {
				message: "Master password set successfully",
				updatedAt: existing.updatedAt,
			};
		}

		const config = new AdminConfig({
			type: "master_password",
			passwordHash: hashedPassword,
			note: `Initially set by ${setBy} on ${new Date().toISOString()}`,
		});

		await config.save();

		return {
			message: "Master password set successfully",
			updatedAt: config.updatedAt,
		};
	},

	async updateMasterPassword(currentPassword, newPassword, updatedBy) {
		const config = await AdminConfig.findOne({ type: "master_password" });

		if (!config) {
			throw new ApiError(404, "Master password configuration not found");
		}

		const isValid = await bcrypt.compare(currentPassword, config.passwordHash);
		if (!isValid) {
			throw new ApiError(401, "Current password is incorrect");
		}

		const hashedPassword = await bcrypt.hash(newPassword, 12);

		config.passwordHash = hashedPassword;
		config.note = `Last updated by ${updatedBy} on ${new Date().toISOString()}`;
		await config.save();

		return {
			message: "Master password updated successfully",
			updatedAt: config.updatedAt,
		};
	},

	async verifyAccess(walletAddress, masterPassword, ipAddress, userAgent) {
		const walletHash = createWalletHash(walletAddress);
		const user = await User.findOne({ walletHash });

		if (!user) {
			throw new ApiError(
				404,
				"User not found. Please ensure you are logged in with an admin account.",
			);
		}

		if (user.role !== "admin") {
			throw new ApiError(
				403,
				"Insufficient privileges. Admin access required.",
			);
		}

		const config = await AdminConfig.findOne({ type: "master_password" });

		if (!config || !config.passwordHash) {
			throw new ApiError(500, "Admin configuration not found");
		}

		const isPasswordValid = await bcrypt.compare(
			masterPassword,
			config.passwordHash,
		);

		if (!isPasswordValid) {
			throw new ApiError(401, "Invalid master password");
		}

		const sessionToken = crypto.randomBytes(32).toString("hex");
		const expiresAt = new Date(Date.now() + 30 * 60 * 1000);

		const session = new AdminSession({
			sessionToken,
			walletAddress: walletAddress.toLowerCase(),
			ipAddress,
			userAgent,
			lastActivityAt: new Date(),
			expiresAt,
		});

		await session.save();

		// Notify admin user about new admin session (security email)
		try {
			const bodyContent = `
				<div style="color: #e1e4eb; line-height: 1.8;">
					<h2 style="color: #00e5ff; margin-bottom: 16px; font-size: 22px;">Admin Access Granted</h2>
					<p style="margin-bottom: 16px;">A new admin session was created for your account.</p>
					<div style="background: rgba(59, 130, 246, 0.08); border-left: 4px solid #3b82f6; padding: 16px; border-radius: 8px; margin-bottom: 18px;">
						<table style="width: 100%; border-collapse: collapse; font-size: 14px;">
							<tr>
								<td style="padding: 6px 0; color: #9aa3b2;">Username:</td>
								<td style="padding: 6px 0; color: #ffffff; text-align: right; font-weight: 600;">${user.username}</td>
							</tr>
							<tr>
								<td style="padding: 6px 0; color: #9aa3b2;">Wallet Address:</td>
								<td style="padding: 6px 0; color: #ffffff; text-align: right; font-family: monospace; font-size: 11px;">${walletAddress}</td>
							</tr>
							<tr>
								<td style="padding: 6px 0; color: #9aa3b2;">IP Address:</td>
								<td style="padding: 6px 0; color: #ffffff; text-align: right; font-family: monospace; font-size: 11px;">${ipAddress || "Unknown"}</td>
							</tr>
							<tr>
								<td style="padding: 6px 0; color: #9aa3b2;">User Agent:</td>
								<td style="padding: 6px 0; color: #ffffff; text-align: right; font-size: 11px;">${userAgent || "Unknown"}</td>
							</tr>
							<tr>
								<td style="padding: 6px 0; color: #9aa3b2;">Session Expires:</td>
								<td style="padding: 6px 0; color: #ffffff; text-align: right;">${expiresAt.toLocaleString()}</td>
							</tr>
						</table>
					</div>
					<p style="margin-top: 8px; color: #9aa3b2; font-size: 13px;">If you did not initiate this action, please revoke this session and update the master password immediately.</p>
				</div>
			`;

			const emailHtml = generateEmailFromTemplate(bodyContent);

			await systemNotificationService.sendToUser({
				username: user.username,
				email: user.email,
				subject: "Admin Access Granted - NodeMeta",
				title: "New Admin Session Created",
				body: `A new admin session was created from IP ${ipAddress || "Unknown IP"}. It will expire at ${expiresAt.toLocaleString()}.`,
				html: emailHtml,
				pushPayload: {
					icon: "/icons/icon-192x192.png",
					badge: "/icons/badge-72x72.png",
					data: {
						url: "/dashboard/admin",
						ipAddress: ipAddress || null,
						userAgent: userAgent || null,
					},
				},
				sendEmail: true,
				sendPush: true,
			});
		} catch (error) {
			logger.error("Failed to send admin access notification:", {
				errorMessage: error.message,
				errorStack: error.stack,
				walletAddress,
				ipAddress,
				userAgent,
			});
		}

		return {
			success: true,
			message: "Admin access granted",
			sessionToken,
			expiresIn: 1800,
			expiresAt,
			user: {
				username: user.username,
				walletAddress: user.walletHash,
				role: user.role,
			},
		};
	},

	async validateSession(sessionToken, walletAddress, ipAddress, userAgent) {
		const now = new Date();
		const INACTIVITY_TIMEOUT = 30 * 60 * 1000;

		const session = await AdminSession.findOne({
			sessionToken,
			walletAddress: walletAddress.toLowerCase(),
		});

		if (!session) {
			return {
				valid: false,
				error: "Session not found",
				reason: "invalid_session",
			};
		}

		if (session.expiresAt < now) {
			await AdminSession.deleteOne({ sessionToken });
			return {
				valid: false,
				error: "Session expired",
				reason: "expired",
			};
		}

		const timeSinceLastActivity =
			now.getTime() - session.lastActivityAt.getTime();

		if (
			process.env.NODE_ENV !== "development" &&
			timeSinceLastActivity > INACTIVITY_TIMEOUT
		) {
			await AdminSession.deleteOne({ sessionToken });
			return {
				valid: false,
				error: "Session expired due to inactivity",
				reason: "inactive",
				inactiveFor: Math.floor(timeSinceLastActivity / 60000),
			};
		}

		session.lastActivityAt = now;
		session.ipAddress = ipAddress;
		session.userAgent = userAgent;
		await session.save();

		const remainingTime = session.expiresAt.getTime() - now.getTime();

		const user = await User.findOne({
			walletHash: session.walletAddress,
		}).select("username walletHash role");

		return {
			valid: true,
			remainingTime: Math.floor(remainingTime / 1000),
			expiresAt: session.expiresAt,
			lastActivityAt: now,
			session: {
				walletAddress: session.walletAddress,
				expiresAt: session.expiresAt,
				lastActivityAt: now,
			},
			user: user
				? {
						username: user.username,
						walletAddress: user.walletHash,
						role: user.role,
					}
				: null,
		};
	},

	async revokeSession(sessionToken) {
		const result = await AdminSession.deleteOne({ sessionToken });

		if (result.deletedCount === 0) {
			throw new ApiError(404, "Session not found");
		}

		return { message: "Session revoked successfully" };
	},

	async getActiveSessions(walletAddress) {
		const query = { expiresAt: { $gt: new Date() } };

		if (walletAddress) {
			query.walletAddress = walletAddress.toLowerCase();
		}

		const sessions = await AdminSession.find(query)
			.select(
				"-_id sessionToken walletAddress ipAddress userAgent lastActivityAt expiresAt createdAt",
			)
			.sort({ lastActivityAt: -1 });

		return sessions;
	},

	async cleanupExpiredSessions() {
		const result = await AdminSession.deleteMany({
			expiresAt: { $lte: new Date() },
		});

		return {
			message: "Expired sessions cleaned up",
			deletedCount: result.deletedCount,
		};
	},

	async getSystemSettings() {
		const responseData = { success: true };

		let settings = await AdminSystemSettings.findOne({ type: "system_config" });

		if (!settings) {
			settings = new AdminSystemSettings({
				type: "system_config",
				withdrawalsEnabled: true,
				maintenanceMode: false,
				dashboardMaintenanceMode: false,
				userRegistrationEnabled: true,
				purchaseEnabled: true,
				lastUpdatedBy: "SYSTEM",
			});
			await settings.save();
		}

		responseData.settings = {
			withdrawalsEnabled: settings.withdrawalsEnabled,
			withdrawalsMessage: settings.withdrawalsMessage,
			maintenanceMode: settings.maintenanceMode,
			maintenanceMessage: settings.maintenanceMessage,
			dashboardMaintenanceMode: settings.dashboardMaintenanceMode,
			dashboardMaintenanceMessage: settings.dashboardMaintenanceMessage,
			userRegistrationEnabled: settings.userRegistrationEnabled,
			registrationMessage: settings.registrationMessage,
			purchaseEnabled: settings.purchaseEnabled,
			purchaseMessage: settings.purchaseMessage,
			lastUpdatedBy: settings.lastUpdatedBy,
			lastUpdatedAt: settings.lastUpdatedAt,
			cronJobInProgress: settings.cronJobInProgress,
			cronJobStartedAt: settings.cronJobStartedAt,
			lastCronJobCompleted: settings.lastCronJobCompleted,
			lastCronJobCompletedAt: settings.lastCronJobCompletedAt,
		};

		return responseData;
	},

	async updateSystemSettings(updates, updatedBy) {
		let settings = await AdminSystemSettings.findOne({ type: "system_config" });

		if (!settings) {
			settings = new AdminSystemSettings({ type: "system_config" });
		}

		const allowedFields = [
			"withdrawalsEnabled",
			"withdrawalsMessage",
			"maintenanceMode",
			"maintenanceMessage",
			"dashboardMaintenanceMode",
			"dashboardMaintenanceMessage",
			"userRegistrationEnabled",
			"registrationMessage",
			"purchaseEnabled",
			"purchaseMessage",
		];

		allowedFields.forEach((field) => {
			if (updates[field] !== undefined) {
				settings[field] = updates[field];
			}
		});

		settings.lastUpdatedBy = updatedBy;
		settings.lastUpdatedAt = new Date();

		await settings.save();

		return {
			message: "System settings updated successfully",
			settings: {
				withdrawalsEnabled: settings.withdrawalsEnabled,
				withdrawalsMessage: settings.withdrawalsMessage,
				maintenanceMode: settings.maintenanceMode,
				maintenanceMessage: settings.maintenanceMessage,
				dashboardMaintenanceMode: settings.dashboardMaintenanceMode,
				dashboardMaintenanceMessage: settings.dashboardMaintenanceMessage,
				userRegistrationEnabled: settings.userRegistrationEnabled,
				registrationMessage: settings.registrationMessage,
				purchaseEnabled: settings.purchaseEnabled,
				purchaseMessage: settings.purchaseMessage,
				lastUpdatedBy: settings.lastUpdatedBy,
				lastUpdatedAt: settings.lastUpdatedAt,
			},
		};
	},

	// Handles automatic disabling/enabling of withdrawals during critical cron jobs
	async disableWithdrawalsForCronJob(jobType, adminUsername = "SYSTEM") {
		try {
			let settings = await AdminSystemSettings.findOne({
				type: "system_config",
			});

			if (!settings) {
				settings = new AdminSystemSettings({
					type: "system_config",
					withdrawalsEnabled: true,
					maintenanceMode: false,
					dashboardMaintenanceMode: false,
					userRegistrationEnabled: true,
					purchaseEnabled: true,
					lastUpdatedBy: "SYSTEM",
				});
			}

			const CRITICAL_CRON_JOBS = {
				BONUS_DISTRIBUTION: "Daily Bonus Distribution",
				RANK_SALARY: "Monthly Rank Salary Distribution",
				MONTHLY_BONUS: "Monthly Bonus Distribution",
			};

			const WITHDRAWAL_MESSAGES = {
				BONUS_DISTRIBUTION:
					"Withdrawals are temporarily disabled while daily bonuses are being distributed.",
				RANK_SALARY:
					"Withdrawals are temporarily disabled while monthly rank salaries are being distributed.",
				MONTHLY_BONUS:
					"Withdrawals are temporarily disabled while monthly bonuses are being distributed.",
			};

			const message =
				WITHDRAWAL_MESSAGES[jobType] ||
				"Withdrawals are temporarily disabled for system maintenance.";

			settings.withdrawalsEnabled = false;
			settings.withdrawalsMessage = message;
			settings.lastUpdatedBy = adminUsername;
			settings.lastUpdatedAt = new Date();
			settings.cronJobInProgress = jobType;
			settings.cronJobStartedAt = new Date();

			await settings.save();

			logger.info(
				`Withdrawals disabled for ${CRITICAL_CRON_JOBS[jobType] || jobType}`,
			);
			return {
				success: true,
				message: `Withdrawals disabled for ${
					CRITICAL_CRON_JOBS[jobType] || jobType
				}`,
				jobType,
				disabledAt: settings.lastUpdatedAt,
			};
		} catch (error) {
			logger.error(`Error disabling withdrawals for ${jobType}:`, error);
			throw error;
		}
	},

	async enableWithdrawalsAfterCronJob(jobType, adminUsername = "SYSTEM") {
		try {
			let settings = await AdminSystemSettings.findOne({
				type: "system_config",
			});

			if (!settings) {
				settings = new AdminSystemSettings({
					type: "system_config",
					withdrawalsEnabled: true,
					maintenanceMode: false,
					dashboardMaintenanceMode: false,
					userRegistrationEnabled: true,
					purchaseEnabled: true,
					lastUpdatedBy: "SYSTEM",
				});
			}

			settings.withdrawalsEnabled = true;
			settings.withdrawalsMessage = "";
			settings.lastUpdatedBy = adminUsername;
			settings.lastUpdatedAt = new Date();
			settings.lastCronJobCompleted = jobType;
			settings.lastCronJobCompletedAt = new Date();

			// Remove cron job progress indicators
			settings.cronJobInProgress = null;
			settings.cronJobStartedAt = null;

			await settings.save();

			const CRITICAL_CRON_JOBS = {
				BONUS_DISTRIBUTION: "Daily Bonus Distribution",
				RANK_SALARY: "Monthly Rank Salary Distribution",
				MONTHLY_BONUS: "Monthly Bonus Distribution",
			};

			logger.info(
				`Withdrawals re-enabled after ${
					CRITICAL_CRON_JOBS[jobType] || jobType
				} completion`,
			);
			return {
				success: true,
				message: `Withdrawals re-enabled after ${
					CRITICAL_CRON_JOBS[jobType] || jobType
				} completion`,
				jobType,
				enabledAt: settings.lastUpdatedAt,
			};
		} catch (error) {
			logger.error(`Error re-enabling withdrawals after ${jobType}:`, error);
			throw error;
		}
	},

	async isWithdrawalDisabledForCronJob() {
		try {
			const settings = await AdminSystemSettings.findOne({
				type: "system_config",
			});

			return {
				isDisabled:
					settings?.withdrawalsEnabled === false && settings?.cronJobInProgress,
				cronJobInProgress: settings?.cronJobInProgress || null,
				message: settings?.withdrawalsMessage || "",
				startedAt: settings?.cronJobStartedAt || null,
			};
		} catch (error) {
			logger.error("Error checking withdrawal status:", error);
			return {
				isDisabled: false,
				cronJobInProgress: null,
				message: "",
				startedAt: null,
			};
		}
	},

	async getSystemConfig(type) {
		const settings = await AdminSystemSettings.findOne({
			type: "system_config",
		});

		if (!settings) {
			throw new ApiError(404, "System settings not found");
		}

		const responseData = { success: true };

		if (!type) {
			responseData.configs = {
				withdrawals: {
					type: "withdrawals",
					enabled: settings.withdrawalsEnabled,
					status: settings.withdrawalsEnabled ? "enabled" : "disabled",
					message: settings.withdrawalsMessage || "",
					statusText: settings.withdrawalsEnabled
						? "All users can currently withdraw funds"
						: "All withdrawals are currently blocked system-wide",
				},
				maintenance: {
					type: "maintenance",
					enabled: settings.maintenanceMode,
					status: settings.maintenanceMode ? "active" : "inactive",
					message: settings.maintenanceMessage || "",
					statusText: settings.maintenanceMode
						? "System is currently in maintenance mode"
						: "System is running normally",
				},
				dashboardMaintenance: {
					type: "dashboard-maintenance",
					enabled: settings.dashboardMaintenanceMode,
					status: settings.dashboardMaintenanceMode ? "active" : "inactive",
					message: settings.dashboardMaintenanceMessage || "",
					statusText: settings.dashboardMaintenanceMode
						? "Dashboard is currently in maintenance mode (admin excluded)"
						: "Dashboard is accessible to all users",
				},
				registration: {
					type: "registration",
					enabled: settings.userRegistrationEnabled,
					status: settings.userRegistrationEnabled ? "open" : "closed",
					message: settings.registrationMessage || "",
					statusText: settings.userRegistrationEnabled
						? "New users can register and create accounts"
						: "New user registration is currently disabled",
				},
				purchase: {
					type: "purchase",
					enabled: settings.purchaseEnabled,
					status: settings.purchaseEnabled ? "enabled" : "disabled",
					message: settings.purchaseMessage || "",
					statusText: settings.purchaseEnabled
						? "Users can currently purchase packages"
						: "Package purchases are currently disabled",
				},
			};
			responseData.lastUpdatedBy = settings.lastUpdatedBy;
			responseData.lastUpdatedAt = settings.lastUpdatedAt;
			return responseData;
		}

		switch (type) {
			case "withdrawals":
				responseData.config = {
					type: "withdrawals",
					enabled: settings.withdrawalsEnabled,
					status: settings.withdrawalsEnabled ? "enabled" : "disabled",
					message: settings.withdrawalsMessage || "",
					statusText: settings.withdrawalsEnabled
						? "All users can currently withdraw funds"
						: "All withdrawals are currently blocked system-wide",
				};
				break;

			case "maintenance":
				responseData.config = {
					type: "maintenance",
					enabled: settings.maintenanceMode,
					status: settings.maintenanceMode ? "active" : "inactive",
					message: settings.maintenanceMessage || "",
					statusText: settings.maintenanceMode
						? "System is currently in maintenance mode"
						: "System is running normally",
				};
				break;

			case "dashboard-maintenance":
				responseData.config = {
					type: "dashboard-maintenance",
					enabled: settings.dashboardMaintenanceMode,
					status: settings.dashboardMaintenanceMode ? "active" : "inactive",
					message: settings.dashboardMaintenanceMessage || "",
					statusText: settings.dashboardMaintenanceMode
						? "Dashboard is currently in maintenance mode (admin excluded)"
						: "Dashboard is accessible to all users",
				};
				break;

			case "registration":
				responseData.config = {
					type: "registration",
					enabled: settings.userRegistrationEnabled,
					status: settings.userRegistrationEnabled ? "open" : "closed",
					message: settings.registrationMessage || "",
					statusText: settings.userRegistrationEnabled
						? "New users can register and create accounts"
						: "New user registration is currently disabled",
				};
				break;

			case "purchase":
				responseData.config = {
					type: "purchase",
					enabled: settings.purchaseEnabled,
					status: settings.purchaseEnabled ? "enabled" : "disabled",
					message: settings.purchaseMessage || "",
					statusText: settings.purchaseEnabled
						? "Users can currently purchase packages"
						: "Package purchases are currently disabled",
				};
				break;

			default:
				throw new ApiError(
					400,
					"Invalid config type. Use: withdrawals, maintenance, dashboard-maintenance, registration, or purchase",
				);
		}

		responseData.lastUpdatedBy = settings.lastUpdatedBy;
		responseData.lastUpdatedAt = settings.lastUpdatedAt;
		return responseData;
	},

	async blockUser(userId, reason, blockedBy) {
		const user = await User.findById(userId);

		if (!user) {
			throw new ApiError(404, "User not found");
		}

		if (user.isBlocked) {
			throw new ApiError(400, "User is already blocked");
		}

		user.isBlocked = true;
		user.blockedAt = new Date();
		user.blockedBy = blockedBy;
		user.blockReason = reason || "No reason provided";

		await user.save();

		return {
			message: "User blocked successfully",
			user: {
				id: user._id,
				username: user.username,
				email: user.email,
				isBlocked: user.isBlocked,
				blockedAt: user.blockedAt,
				blockedBy: user.blockedBy,
				blockReason: user.blockReason,
			},
		};
	},

	async unblockUser(userId, unblockedBy) {
		const user = await User.findById(userId);

		if (!user) {
			throw new ApiError(404, "User not found");
		}

		if (!user.isBlocked) {
			throw new ApiError(400, "User is not blocked");
		}

		user.isBlocked = false;
		user.blockedAt = undefined;
		user.blockedBy = undefined;
		user.blockReason = undefined;

		await user.save();

		return {
			message: "User unblocked successfully",
			user: {
				id: user._id,
				username: user.username,
				email: user.email,
				isBlocked: user.isBlocked,
				unblockedBy,
			},
		};
	},

	async disableUserWithdrawals(userId, reason, disabledBy) {
		const user = await User.findById(userId);

		if (!user) {
			throw new ApiError(404, "User not found");
		}

		if (user.withdrawalsDisabled) {
			throw new ApiError(400, "User withdrawals are already disabled");
		}

		user.withdrawalsDisabled = true;
		user.withdrawalsDisabledAt = new Date();
		user.withdrawalsDisabledBy = disabledBy;
		user.withdrawalDisableReason = reason || "No reason provided";

		await user.save();

		return {
			message: "User withdrawals disabled successfully",
			user: {
				id: user._id,
				username: user.username,
				email: user.email,
				withdrawalsDisabled: user.withdrawalsDisabled,
				withdrawalsDisabledAt: user.withdrawalsDisabledAt,
				withdrawalsDisabledBy: user.withdrawalsDisabledBy,
				withdrawalDisableReason: user.withdrawalDisableReason,
			},
		};
	},

	async enableUserWithdrawals(userId, enabledBy) {
		const user = await User.findById(userId);

		if (!user) {
			throw new ApiError(404, "User not found");
		}

		if (!user.withdrawalsDisabled) {
			throw new ApiError(400, "User withdrawals are not disabled");
		}

		user.withdrawalsDisabled = false;
		user.withdrawalsDisabledAt = undefined;
		user.withdrawalsDisabledBy = undefined;
		user.withdrawalDisableReason = undefined;

		await user.save();

		return {
			message: "User withdrawals enabled successfully",
			user: {
				id: user._id,
				username: user.username,
				email: user.email,
				withdrawalsDisabled: user.withdrawalsDisabled,
				enabledBy,
			},
		};
	},

	async getOverrides(action = "get", limit = 50, daysOld = 30) {
		if (action === "get") {
			const activeOverrides = await AdminOverride.find({
				expiresAt: { $gt: new Date() },
				isActive: true,
			});

			// Get current values from global config
			const [currentTokenValue, currentTotalTax, currentTotalUsers] =
				await Promise.all([
					getTokenValue(),
					getTotalTaxEarned(),
					getTotalUsers(),
				]);

			const processedOverrides = {
				tokenValue: {
					currentValue: currentTokenValue,
					override: null,
				},
				totalUsers: {
					currentValue: currentTotalUsers,
					override: null,
				},
				totalTax: {
					currentValue: currentTotalTax,
					override: null,
				},
			};

			activeOverrides.forEach((override) => {
				const overrideData = {
					id: override._id,
					value: override.value,
					setBy: override.setBy,
					setAt: override.setAt,
					expiresAt: override.expiresAt,
					durationMinutes: override.durationMinutes,
					reason: override.reason,
				};

				if (
					override.type === "TOKEN_VALUE" &&
					!processedOverrides.tokenValue.override
				) {
					processedOverrides.tokenValue.override = overrideData;
				} else if (
					override.type === "TOTAL_USERS" &&
					!processedOverrides.totalUsers.override
				) {
					processedOverrides.totalUsers.override = overrideData;
				} else if (
					override.type === "TOTAL_TAX" &&
					!processedOverrides.totalTax.override
				) {
					processedOverrides.totalTax.override = overrideData;
				}
			});

			return {
				success: true,
				overrides: processedOverrides,
			};
		}

		if (action === "cleanup") {
			const cutoffDate = new Date();
			cutoffDate.setDate(cutoffDate.getDate() - daysOld);

			const result = await AdminOverride.deleteMany({
				expiresAt: { $lt: cutoffDate },
			});

			return {
				success: true,
				message: `Cleaned up overrides older than ${daysOld} days`,
				deletedCount: result.deletedCount,
			};
		}

		if (action === "history") {
			const history = await AdminOverride.find({})
				.sort({ setAt: -1 })
				.limit(limit);

			return {
				success: true,
				history: history,
			};
		}

		throw new ApiError(400, "Invalid action");
	},

	async setOverride(type, value, durationMinutes, reason, setBy) {
		const validTypes = ["TOKEN_VALUE", "TOTAL_USERS", "TOTAL_TAX"];

		if (!validTypes.includes(type)) {
			throw new ApiError(
				400,
				`Invalid override type. Must be one of: ${validTypes.join(", ")}`,
			);
		}

		if (!value || value === null) {
			throw new ApiError(400, "Override value is required");
		}

		if (!durationMinutes || durationMinutes <= 0) {
			throw new ApiError(400, "Duration must be greater than 0 minutes");
		}

		const maxDuration = 30 * 24 * 60;
		if (durationMinutes > maxDuration) {
			throw new ApiError(400, "Duration cannot exceed 30 days");
		}

		if (!reason || reason.trim() === "") {
			throw new ApiError(400, "Reason is required");
		}

		await AdminOverride.updateMany(
			{ type, isActive: true },
			{
				$set: {
					isActive: false,
					deactivatedAt: new Date(),
					deactivatedBy: setBy,
					deactivationReason: "Replaced by new override",
				},
			},
		);

		const expiresAt = new Date(Date.now() + durationMinutes * 60 * 1000);

		const override = new AdminOverride({
			type,
			value,
			setBy,
			setAt: new Date(),
			expiresAt,
			durationMinutes,
			reason,
			isActive: true,
			deactivatedAt: new Date(0),
			deactivatedBy: "",
		});

		await override.save();

		return {
			success: true,
			message: "Override set successfully",
			override: {
				id: override._id,
				type: override.type,
				value: override.value,
				setBy: override.setBy,
				setAt: override.setAt,
				expiresAt: override.expiresAt,
				durationMinutes: override.durationMinutes,
				reason: override.reason,
				isActive: override.isActive,
			},
		};
	},

	async deactivateOverride(overrideId, reason, deactivatedBy) {
		const override = await AdminOverride.findById(overrideId);

		if (!override) {
			throw new ApiError(404, "Override not found");
		}

		if (!override.isActive) {
			throw new ApiError(400, "Override is already deactivated");
		}

		override.isActive = false;
		override.deactivatedAt = new Date();
		override.deactivatedBy = deactivatedBy;
		override.deactivationReason = reason || "Manual deactivation";

		await override.save();

		return {
			success: true,
			message: "Override deactivated successfully",
			override: {
				id: override._id,
				type: override.type,
				isActive: override.isActive,
				deactivatedAt: override.deactivatedAt,
				deactivatedBy: override.deactivatedBy,
				deactivationReason: override.deactivationReason,
			},
		};
	},

	async deleteOverride(overrideId) {
		const result = await AdminOverride.findByIdAndDelete(overrideId);

		if (!result) {
			throw new ApiError(404, "Override not found");
		}

		return {
			success: true,
			message: "Override deleted successfully",
		};
	},

	async getDashboardStats() {
		const totalUsers = await User.countDocuments();
		const activeUsers = await User.countDocuments({ isActive: true });
		const blockedUsers = await User.countDocuments({ isBlocked: true });
		const adminUsers = await User.countDocuments({ role: "admin" });

		const recentUsers = await User.find()
			.sort({ createdAt: -1 })
			.limit(5)
			.select("username email fullName createdAt role");

		const activeOverridesCount = await AdminOverride.countDocuments({
			isActive: true,
			expiresAt: { $gt: new Date() },
		});

		const activeSessionsCount = await AdminSession.countDocuments({
			expiresAt: { $gt: new Date() },
		});

		return {
			users: {
				total: totalUsers,
				active: activeUsers,
				blocked: blockedUsers,
				admins: adminUsers,
			},
			recent: recentUsers,
			system: {
				activeOverrides: activeOverridesCount,
				activeSessions: activeSessionsCount,
			},
		};
	},

	async getAllUsers(options) {
		const { page, limit, search, role, isBlocked, sortBy, sortOrder } = options;

		const query = {};

		if (search) {
			query.$or = [
				{ username: { $regex: search, $options: "i" } },
				{ email: { $regex: search, $options: "i" } },
				{ fullName: { $regex: search, $options: "i" } },
			];
		}

		if (role) {
			query.role = role;
		}

		if (isBlocked !== undefined) {
			query.isBlocked = isBlocked;
		}

		const sort = {};
		sort[sortBy] = sortOrder === "asc" ? 1 : -1;

		const skip = (page - 1) * limit;

		const users = await User.find(query)
			.select("-password -twoFactorSecretHash -twoFactorSecretTemp")
			.skip(skip)
			.limit(limit)
			.sort(sort);

		const total = await User.countDocuments(query);

		// Gather comprehensive statistics for the filtered results
		const sevenDaysAgo = new Date();
		sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);
		const thirtyDaysAgo = new Date();
		thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);

		const [
			activeUsersCount,
			inactiveUsersCount,
			blockedUsersCount,
			withdrawalsDisabledCount,
			twoFactorEnabledCount,
			emailVerifiedCount,
			recentRegistrations7Days,
			recentRegistrations30Days,
			roleDistribution,
			walletBalanceStats,
		] = await Promise.all([
			User.countDocuments({ ...query, isActive: true }),
			User.countDocuments({ ...query, isActive: false }),
			User.countDocuments({ ...query, isBlocked: true }),
			User.countDocuments({ ...query, withdrawalsDisabled: true }),
			User.countDocuments({ ...query, twoFactorEnabled: true }),
			User.countDocuments({ ...query, emailVerified: true }),
			User.countDocuments({
				...query,
				createdAt: { $gte: sevenDaysAgo },
			}),
			User.countDocuments({
				...query,
				createdAt: { $gte: thirtyDaysAgo },
			}),
			User.aggregate([
				{ $match: query },
				{ $group: { _id: "$role", count: { $sum: 1 } } },
			]),
			User.aggregate([
				{ $match: query },
				{
					$group: {
						_id: null,
						totalBalance: { $sum: "$walletBalance" },
						averageBalance: { $avg: "$walletBalance" },
						minBalance: { $min: "$walletBalance" },
						maxBalance: { $max: "$walletBalance" },
					},
				},
			]),
		]);

		const roleStats = roleDistribution.reduce((acc, item) => {
			acc[item._id] = item.count;
			return acc;
		}, {});

		const balanceStats = walletBalanceStats[0] || {
			totalBalance: 0,
			averageBalance: 0,
			minBalance: 0,
			maxBalance: 0,
		};

		return {
			users,
			pagination: {
				page,
				limit,
				total,
				totalPages: Math.ceil(total / limit),
			},
			stats: {
				total,
				active: activeUsersCount,
				inactive: inactiveUsersCount,
				blocked: blockedUsersCount,
				withdrawalsDisabled: withdrawalsDisabledCount,
				twoFactorEnabled: twoFactorEnabledCount,
				emailVerified: emailVerifiedCount,
				recentRegistrations: {
					last7Days: recentRegistrations7Days,
					last30Days: recentRegistrations30Days,
				},
				roleDistribution: roleStats,
				walletBalance: {
					total: balanceStats.totalBalance,
					average: balanceStats.averageBalance,
					min: balanceStats.minBalance,
					max: balanceStats.maxBalance,
				},
			},
		};
	},

	async getUserDetails(userId) {
		const user = await User.findById(userId).select(
			"-password -twoFactorSecretHash -twoFactorSecretTemp",
		);

		if (!user) {
			throw new ApiError(404, "User not found");
		}

		return user;
	},

	async updateUserRole(userId, newRole, updatedBy) {
		const validRoles = ["user", "admin", "support"];

		if (!validRoles.includes(newRole)) {
			throw new ApiError(
				400,
				`Invalid role. Must be one of: ${validRoles.join(", ")}`,
			);
		}

		const user = await User.findById(userId);

		if (!user) {
			throw new ApiError(404, "User not found");
		}

		const oldRole = user.role;
		user.role = newRole;

		await user.save();

		return {
			message: "User role updated successfully",
			user: {
				id: user._id,
				username: user.username,
				email: user.email,
				oldRole,
				newRole: user.role,
				updatedBy,
			},
		};
	},

	async updateUserStatus(userId, isActive, updatedBy) {
		if (typeof isActive !== "boolean") {
			throw new ApiError(400, "isActive must be a boolean value");
		}

		const user = await User.findById(userId);

		if (!user) {
			throw new ApiError(404, "User not found");
		}

		const oldStatus = user.isActive;
		user.isActive = isActive;

		await user.save();

		return {
			message: "User status updated successfully",
			user: {
				id: user._id,
				username: user.username,
				email: user.email,
				oldStatus,
				newStatus: user.isActive,
				updatedBy,
			},
		};
	},

	async adjustUserWalletBalance(userId, amount, reason, updatedBy) {
		const user = await User.findById(userId);

		if (!user) {
			throw new ApiError(404, "User not found");
		}

		const oldBalance = user.walletBalance;
		user.walletBalance += amount;

		if (user.walletBalance < 0) {
			throw new ApiError(400, "Wallet balance cannot be negative");
		}

		user.walletBalanceUpdatedAt = new Date();
		user.walletBalanceUpdatedBy = `${updatedBy} - ${reason}`;

		await user.save();

		return {
			message: "Wallet balance adjusted successfully",
			user: {
				id: user._id,
				username: user.username,
				email: user.email,
				oldBalance,
				newBalance: user.walletBalance,
				adjustment: amount,
				reason,
				updatedBy,
			},
		};
	},

	async recordManualBalanceAdjustment(options) {
		const {
			username,
			userId,
			amountNTE,
			amountUSD,
			adjustmentType,
			balanceType,
			balanceCategory,
			reason,
			adminUsername,
			referenceId,
			walletBalanceBefore,
		} = options;

		const isDeduction = adjustmentType === "deduction";
		const absoluteAmount = Math.abs(amountNTE);
		const absoluteAmountUSD = Math.abs(amountUSD);

		const actionVerb = isDeduction ? "deducted from" : "added to";
		const description = `Manual balance adjustment: ${absoluteAmount} NTE ${actionVerb} ${username}'s ${balanceType} balance by admin ${adminUsername}. Reason: ${reason}`;

		try {
			// Fetch product information using balanceType (partial match)
			const product = await Product.findOne({
				name: { $regex: balanceType, $options: "i" },
			});
			let productType = balanceCategory || "unknown";

			if (product) {
				productType = product.packageType;
			}

			const txResult = await recordComprehensiveTransaction({
				username,
				userId,
				transactionType: isDeduction ? "balance_deduction" : "balance_addition",
				bonusType: "manual_balance_adjustment",
				amountNTE: absoluteAmount,
				amountUSD: absoluteAmountUSD,
				description,
				metadata: {
					packageType: productType,
					packageName: balanceType,
					triggerType: "admin_action",
				},
				walletBalanceBefore,
				isDeduction,
				skipWalletUpdate: false,
			});

			// Send notification email to user about manual balance adjustment
			try {
				const user = await User.findById(userId).select("username email walletBalance");
				if (user) {
					const adjustmentSign = isDeduction ? "-" : "+";
					const finalBalance =
						txResult.walletBalanceAfter !== undefined
							? txResult.walletBalanceAfter
							: (user.walletBalance || 0);

					const bodyContent = `
						<div style="color: #e1e4eb; line-height: 1.8;">
							<h2 style="color: #00e5ff; margin-bottom: 16px; font-size: 22px;">Wallet Balance ${
								isDeduction ? "Deducted" : "Credited"
							}</h2>
							<p style="margin-bottom: 16px;">Your wallet balance has been ${
								isDeduction ? "reduced" : "increased"
							} by an administrator.</p>
							<div style="background: rgba(59, 130, 246, 0.08); border-left: 4px solid #3b82f6; padding: 16px; border-radius: 8px; margin-bottom: 18px;">
								<table style="width: 100%; border-collapse: collapse; font-size: 14px;">
									<tr>
										<td style="padding: 6px 0; color: #9aa3b2;">Username:</td>
										<td style="padding: 6px 0; color: #ffffff; text-align: right; font-weight: 600;">${user.username}</td>
									</tr>
									<tr>
										<td style="padding: 6px 0; color: #9aa3b2;">Adjustment:</td>
										<td style="padding: 6px 0; color: #00e5ff; text-align: right; font-weight: 600;">${
											adjustmentSign + absoluteAmount.toFixed(6)
										} NTE</td>
									</tr>
									<tr>
										<td style="padding: 6px 0; color: #9aa3b2;">Balance Type:</td>
										<td style="padding: 6px 0; color: #ffffff; text-align: right; text-transform: capitalize;">${balanceType}</td>
									</tr>
									<tr>
										<td style="padding: 6px 0; color: #9aa3b2;">Balance Before:</td>
										<td style="padding: 6px 0; color: #ffffff; text-align: right;">${walletBalanceBefore.toFixed(6)} NTE</td>
									</tr>
									<tr>
										<td style="padding: 6px 0; color: #9aa3b2;">Balance After:</td>
										<td style="padding: 6px 0; color: #ffffff; text-align: right; font-weight: 600;">${finalBalance.toFixed(6)} NTE</td>
									</tr>
									<tr>
										<td style="padding: 6px 0; color: #9aa3b2;">Reason:</td>
										<td style="padding: 6px 0; color: #ffffff; text-align: right;">${reason}</td>
									</tr>
									<tr>
										<td style="padding: 6px 0; color: #9aa3b2;">Adjusted By:</td>
										<td style="padding: 6px 0; color: #ffffff; text-align: right;">${adminUsername}</td>
									</tr>
								</table>
							</div>
							<p style="margin-top: 8px; color: #9aa3b2; font-size: 13px;">If you did not expect this change, please contact support immediately.</p>
						</div>
					`;

					const emailHtml = generateEmailFromTemplate(bodyContent);

					await systemNotificationService.sendToUser({
						username: user.username,
						email: user.email,
						subject: "Wallet Balance Updated - NodeMeta",
						title: "Wallet Balance Adjusted",
						body: `Your wallet balance was ${
							isDeduction ? "reduced" : "increased"
						} by ${adjustmentSign}${absoluteAmount.toFixed(6)} NTE. New balance: ${finalBalance.toFixed(6)} NTE.`,
						html: emailHtml,
						pushPayload: {
							icon: "/icons/icon-192x192.png",
							badge: "/icons/badge-72x72.png",
							data: {
								url: "/dashboard/wallet",
								adjustmentType: isDeduction ? "deduction" : "addition",
								balanceType,
								amountNTE: adjustmentSign + absoluteAmount.toFixed(6),
							},
						},
						sendEmail: true,
						sendPush: true,
					});
				}
			} catch (notificationError) {
				logger.error("Failed to send manual balance adjustment notification:", {
					errorMessage: notificationError.message,
					errorStack: notificationError.stack,
					username,
					adminUsername,
				});
			}

			// Notify admins about the manual balance adjustment
			try {
				const adminBodyContent = `
					<div style="color: #e1e4eb; line-height: 1.8;">
						<h2 style="color: #00e5ff; margin-bottom: 16px; font-size: 22px;">Manual Balance Adjustment</h2>
						<p style="margin-bottom: 16px;">An administrator has performed a manual balance adjustment for a user.</p>
						<div style="background: rgba(59, 130, 246, 0.08); border-left: 4px solid #3b82f6; padding: 16px; border-radius: 8px; margin-bottom: 18px;">
							<table style="width: 100%; border-collapse: collapse; font-size: 14px;">
								<tr>
									<td style="padding: 6px 0; color: #9aa3b2;">Username:</td>
									<td style="padding: 6px 0; color: #ffffff; text-align: right; font-weight: 600;">${username}</td>
								</tr>
								<tr>
									<td style="padding: 6px 0; color: #9aa3b2;">Adjustment Type:</td>
									<td style="padding: 6px 0; color: #ffffff; text-align: right; text-transform: capitalize;">${
										isDeduction ? "deduction" : "addition"
									}</td>
								</tr>
								<tr>
									<td style="padding: 6px 0; color: #9aa3b2;">Amount:</td>
									<td style="padding: 6px 0; color: #00e5ff; text-align: right; font-weight: 600;">${
										(isDeduction ? "-" : "+") + absoluteAmount.toFixed(6)
									} NTE</td>
								</tr>
								<tr>
									<td style="padding: 6px 0; color: #9aa3b2;">Balance Type:</td>
									<td style="padding: 6px 0; color: #ffffff; text-align: right; text-transform: capitalize;">${balanceType}</td>
								</tr>
								<tr>
									<td style="padding: 6px 0; color: #9aa3b2;">Balance Before:</td>
									<td style="padding: 6px 0; color: #ffffff; text-align: right;">${walletBalanceBefore.toFixed(6)} NTE</td>
								</tr>
								<tr>
									<td style="padding: 6px 0; color: #9aa3b2;">Reason:</td>
									<td style="padding: 6px 0; color: #ffffff; text-align: right;">${reason}</td>
								</tr>
								<tr>
									<td style="padding: 6px 0; color: #9aa3b2;">Adjusted By:</td>
									<td style="padding: 6px 0; color: #ffffff; text-align: right;">${adminUsername}</td>
								</tr>
							</table>
						</div>
						<p style="margin-top: 8px; color: #9aa3b2; font-size: 13px;">This email is for audit and traceability of manual admin actions.</p>
					</div>
				`;

				const adminEmailHtml = generateEmailFromTemplate(adminBodyContent);

				await systemNotificationService.sendToAdmin({
					subject: `Manual Balance Adjustment: ${
						isDeduction ? "-" : "+"
					}${absoluteAmount.toFixed(2)} NTE - ${username}`,
					title: "Manual Balance Adjustment",
					body: `Admin ${adminUsername} performed a ${
						isDeduction ? "deduction" : "credit"
					} of ${
						(isDeduction ? "-" : "+") + absoluteAmount.toFixed(6)
					} NTE on ${username}'s ${balanceType} balance.`,
					html: adminEmailHtml,
					pushPayload: {
						icon: "/icons/icon-192x192.png",
						badge: "/icons/badge-72x72.png",
						data: {
							url: "/admin/users",
							username,
							adjustmentType: isDeduction ? "deduction" : "addition",
							balanceType,
							amountNTE: (isDeduction ? "-" : "+") + absoluteAmount.toFixed(6),
						},
					},
					sendEmail: true,
					sendPush: true,
				});
			} catch (adminNotificationError) {
				logger.error("Failed to send admin manual adjustment notification:", {
					errorMessage: adminNotificationError.message,
					errorStack: adminNotificationError.stack,
					username,
					adminUsername,
				});
			}

			return txResult;
		} catch (error) {
			throw new ApiError(500, `Failed to record adjustment: ${error.message}`);
		}
	},

	async getManualAdjustmentHistory(filters) {
		const {
			username,
			balanceType,
			adjustmentType,
			startDate,
			endDate,
			page = 1,
			limit = 20,
		} = filters;

		const query = {
			subType: "manual_balance_adjustment",
		};

		if (username) {
			query.username = username.trim();
		}

		if (adjustmentType) {
			query.type =
				adjustmentType === "deduction"
					? "balance_deduction"
					: "balance_addition";
		}

		if (balanceType) {
			query["metadata.packageName"] = balanceType;
		}

		if (startDate || endDate) {
			query.createdAt = {};
			if (startDate) {
				query.createdAt.$gte = new Date(startDate);
			}
			if (endDate) {
				const end = new Date(endDate);
				end.setHours(23, 59, 59, 999);
				query.createdAt.$lte = end;
			}
		}

		const totalCount = await WalletTransaction.countDocuments(query);
		const totalPages = Math.ceil(totalCount / limit);

		const adjustments = await WalletTransaction.find(query)
			.sort({ createdAt: -1 })
			.skip((page - 1) * limit)
			.limit(limit)
			.lean();

		const aggregation = await WalletTransaction.aggregate([
			{ $match: query },
			{
				$group: {
					_id: null,
					totalAdditions: {
						$sum: {
							$cond: [{ $eq: ["$type", "balance_addition"] }, "$amountNTE", 0],
						},
					},
					totalDeductions: {
						$sum: {
							$cond: [{ $eq: ["$type", "balance_deduction"] }, "$amountNTE", 0],
						},
					},
					totalAdditionsUSD: {
						$sum: {
							$cond: [{ $eq: ["$type", "balance_addition"] }, "$amountUSD", 0],
						},
					},
					totalDeductionsUSD: {
						$sum: {
							$cond: [{ $eq: ["$type", "balance_deduction"] }, "$amountUSD", 0],
						},
					},
					count: { $sum: 1 },
				},
			},
		]);

		const statistics = aggregation[0] || {
			totalAdditions: 0,
			totalDeductions: 0,
			totalAdditionsUSD: 0,
			totalDeductionsUSD: 0,
			count: 0,
		};

		return {
			success: true,
			data: adjustments,
			pagination: { page, limit, totalCount, totalPages },
			statistics,
		};
	},

	async processMultipleAdjustments(adjustments, masterPassword, adminUsername) {
		const config = await AdminConfig.findOne({ type: "master_password" });

		if (!config || !config.passwordHash) {
			throw new ApiError(500, "Admin configuration not found");
		}

		const isPasswordValid = await bcrypt.compare(
			masterPassword.trim(),
			config.passwordHash,
		);

		if (!isPasswordValid) {
			throw new ApiError(401, "Invalid admin master password");
		}

		if (!Array.isArray(adjustments) || adjustments.length === 0) {
			throw new ApiError(
				400,
				"Adjustments array is required and must not be empty",
			);
		}

		for (const adj of adjustments) {
			if (!adj.username || typeof adj.username !== "string") {
				throw new ApiError(400, "Each adjustment must have a valid username");
			}

			if (typeof adj.amountNTE !== "number" || adj.amountNTE === 0) {
				throw new ApiError(400, `Invalid amount for user ${adj.username}`);
			}

			if (!["addition", "deduction"].includes(adj.adjustmentType)) {
				throw new ApiError(
					400,
					`Invalid adjustment type for user ${adj.username}`,
				);
			}

			if (
				!["core", "elite", "stake", "meta_pulse", "others"].includes(
					adj.balanceType,
				)
			) {
				throw new ApiError(
					400,
					`Invalid balance type for user ${adj.username}`,
				);
			}

			if (!adj.reason || adj.reason.trim().length < 10) {
				throw new ApiError(
					400,
					`Reason required for ${adj.username} (min 10 chars)`,
				);
			}
		}

		const results = {
			success: true,
			successful: [],
			failed: [],
			totalProcessed: 0,
			totalSuccessful: 0,
			totalFailed: 0,
		};

		const tokenValue = await getTokenValue();

		for (const adjustment of adjustments) {
			try {
				const {
					username,
					amountNTE,
					adjustmentType,
					balanceType,
					balanceCategory,
					reason,
					referenceId,
				} = adjustment;

				const user = await User.findOne({ username: username.trim() });
				if (!user) {
					results.failed.push({
						username: username.trim(),
						error: "USER_NOT_FOUND",
					});
					continue;
				}

				const amountUSD = Math.abs(amountNTE) * tokenValue;
				const walletBalanceBefore = user.walletBalance || 0;

				const result = await this.recordManualBalanceAdjustment({
					username: username.trim(),
					userId: user._id,
					amountNTE:
						adjustmentType === "deduction"
							? -Math.abs(amountNTE)
							: Math.abs(amountNTE),
					amountUSD:
						adjustmentType === "deduction"
							? -Math.abs(amountUSD)
							: Math.abs(amountUSD),
					adjustmentType,
					balanceType,
					balanceCategory,
					reason: reason.trim(),
					adminUsername,
					referenceId: referenceId || null,
					walletBalanceBefore,
				});

				results.successful.push({
					username: username.trim(),
					amountNTE: Math.abs(amountNTE),
					amountUSD: Math.abs(amountUSD),
					adjustmentType,
					balanceType,
					balanceBefore: walletBalanceBefore,
					balanceAfter: result.walletBalanceAfter || user.walletBalance,
					transactionIds: {
						distribution: result.distributionId,
						bonusTransaction: result.bonusTransactionId,
						walletTransaction: result.walletTransactionId,
					},
					traceId: result.traceId,
				});
			} catch (error) {
				results.failed.push({
					username: adjustment.username,
					error: "PROCESSING_ERROR",
					message: error.message,
				});
			}

			results.totalProcessed++;
		}

		results.totalSuccessful = results.successful.length;
		results.totalFailed = results.failed.length;

		return results;
	},

	async flagTransactionForRefund(transactionId, flagged, flagReason) {
		const updateData = {};

		if (flagged !== undefined) {
			updateData.flaggedForRefund = flagged;
			updateData.refundStatus = flagged ? "pending" : "none";
			updateData.flaggedAt = flagged ? new Date() : null;
		}

		if (flagReason !== undefined) {
			updateData.flagReason = flagReason;
		}

		if (flagged === false) {
			updateData.refundStatus = "none";
			updateData.flagReason = "";
		}

		if (Object.keys(updateData).length === 0) {
			throw new ApiError(400, "No update data provided");
		}

		const result = await WalletTransaction.updateOne(
			{ _id: transactionId },
			{ $set: updateData },
		);

		if (result.modifiedCount === 0) {
			throw new ApiError(404, "Transaction not found or already flagged");
		}

		const updatedTransaction = await WalletTransaction.findById(transactionId);

		logger.info("Transaction flagged for refund", {
			transactionId,
			flagged,
			flagReason,
		});

		// Notify user about refund flag status change
		try {
			if (updatedTransaction && updatedTransaction.userId) {
				const user = await User.findById(updatedTransaction.userId).select(
					"username email",
				);

				if (user) {
					const isNowFlagged = flagged === true;
					const amountNTE = Math.abs(updatedTransaction.amountNTE || 0);

					const bodyContent = `
						<div style="color: #e1e4eb; line-height: 1.8;">
							<h2 style="color: ${
								isNowFlagged ? "#f97316" : "#22c55e"
							}; margin-bottom: 16px; font-size: 22px;">Transaction ${
								isNowFlagged ? "Flagged for Review" : "Flag Updated"
							}</h2>
							<p style="margin-bottom: 16px;">Your transaction for ${amountNTE.toFixed(
								6,
							)} NTE has been ${
								isNowFlagged ? "flagged for refund review" : "updated"
							} by the NodeMeta team.</p>
							<div style="background: rgba(59, 130, 246, 0.08); border-left: 4px solid #3b82f6; padding: 16px; border-radius: 8px; margin-bottom: 18px;">
								<table style="width: 100%; border-collapse: collapse; font-size: 14px;">
									<tr>
										<td style="padding: 6px 0; color: #9aa3b2;">Transaction ID:</td>
										<td style="padding: 6px 0; color: #ffffff; text-align: right; font-family: monospace; font-size: 11px;">${
											updatedTransaction._id.toString()
										}</td>
									</tr>
									<tr>
										<td style="padding: 6px 0; color: #9aa3b2;">Type:</td>
										<td style="padding: 6px 0; color: #ffffff; text-align: right; text-transform: capitalize;">${
											updatedTransaction.type
										}</td>
									</tr>
									<tr>
										<td style="padding: 6px 0; color: #9aa3b2;">Current Refund Status:</td>
										<td style="padding: 6px 0; color: #ffffff; text-align: right;">${
											updatedTransaction.refundStatus || "none"
										}</td>
									</tr>
									${
										flagReason
											? `
									<tr>
										<td style="padding: 6px 0; color: #9aa3b2;">Reason:</td>
										<td style="padding: 6px 0; color: #ffffff; text-align: right;">${flagReason}</td>
									</tr>
							`
								: ""
							}
							</table>
						</div>
						<p style="margin-top: 8px; color: #9aa3b2; font-size: 13px;">You will receive another notification once the refund review is completed.</p>
					</div>
					`;

					const emailHtml = generateEmailFromTemplate(bodyContent);

					await systemNotificationService.sendToUser({
						username: user.username,
						email: user.email,
						subject: isNowFlagged
							? "Transaction Flagged for Refund Review - NodeMeta"
							: "Transaction Refund Flag Updated - NodeMeta",
						title: isNowFlagged
							? "Transaction Under Review"
							: "Refund Flag Updated",
						body: isNowFlagged
							? `Your transaction for ${amountNTE.toFixed(
									6,
								)} NTE has been flagged for refund review.`
							: `The refund flag for your transaction (${amountNTE.toFixed(
									6,
								)} NTE) has been updated.`,
						html: emailHtml,
						pushPayload: {
							icon: "/icons/icon-192x192.png",
							badge: "/icons/badge-72x72.png",
							data: {
								url: "/dashboard/transactions",
								transactionId: updatedTransaction._id.toString(),
								refundStatus: updatedTransaction.refundStatus || "none",
							},
						},
						sendEmail: true,
						sendPush: true,
					});
				}
			}
		} catch (notificationError) {
			logger.error("Failed to send refund flag notification:", {
				transactionId,
				flagged,
				flagReason,
				errorMessage: notificationError.message,
				errorStack: notificationError.stack,
			});
		}

		return {
			success: true,
			data: updatedTransaction,
		};
	},

	async getFlaggedTransactions(username) {
		const transactions = await WalletTransaction.find({
			username,
			flaggedForRefund: true,
		})
			.sort({ createdAt: -1 })
			.lean();

		logger.info(
			`Retrieved ${transactions.length} flagged transactions for ${username}`,
		);

		return transactions;
	},

	async markTransactionAsRefunded(transactionId, refundTxHash, refundTxInfo) {
		const originalTransaction = await WalletTransaction.findById(transactionId);

		if (!originalTransaction) {
			throw new ApiError(404, "Transaction not found");
		}

		if (!refundTxHash || refundTxHash.trim() === "") {
			throw new ApiError(400, "Refund transaction hash is required");
		}

		// Get user details
		const user = await User.findById(originalTransaction.userId);

		if (!user) {
			throw new ApiError(404, "User not found for this transaction");
		}

		// Extract necessary details for verification
		const toAddress =
			originalTransaction.metadata?.toAddress || user.walletHash;
		const fromAddress = process.env.COMPANY_WALLET_ADDRESS;
		const amountNTE = Math.abs(originalTransaction.amountNTE);

		if (!toAddress) {
			throw new ApiError(
				400,
				"Cannot determine recipient address for refund verification",
			);
		}

		// Get current token price for USD calculation (price may have changed since original transaction)
		const currentTokenValue = await getTokenValue();
		const currentAmountUSD = amountNTE * currentTokenValue;

		// Verify the refund transaction on blockchain
		logger.info("Verifying refund transaction:", {
			transactionHash: refundTxHash,
			fromAddress,
			toAddress,
			expectedAmount: amountNTE,
			currentTokenPrice: currentTokenValue,
			calculatedUSD: currentAmountUSD,
			originalUSD: Math.abs(originalTransaction.amountUSD),
		});

		const verification = await verifyNTETransfer(
			refundTxHash,
			fromAddress,
			toAddress,
			amountNTE,
			true,
		);

		if (!verification.isValid) {
			logger.error("Refund transaction verification failed:", {
				transactionHash: refundTxHash,
				error: verification.error,
				details: verification,
			});
			throw new ApiError(
				400,
				`Refund transaction verification failed: ${verification.error}`,
			);
		}

		logger.info("Refund transaction verified successfully:", {
			transactionHash: refundTxHash,
			amount: verification.amount,
			blockNumber: verification.blockNumber,
			confirmations: verification.confirmations,
		});

		const walletBalanceBefore = user.walletBalance || 0;

		const updateData = {
			refundStatus: "completed",
			refundedAt: new Date(),
			refundTxHash,
		};

		if (refundTxInfo) {
			updateData.refundTxInfo = refundTxInfo;
		}

		const result = await WalletTransaction.updateOne(
			{ _id: transactionId },
			{ $set: updateData },
		);

		if (result.modifiedCount === 0) {
			throw new ApiError(404, "Transaction not found");
		}

		// Log the refund transaction with verified blockchain details
		await this.logRefundTransaction({
			username: originalTransaction.username,
			userId: originalTransaction.userId.toString(),
			toAddress,
			amountNTE,
			amountUSD: currentAmountUSD,
			transactionHash: refundTxHash,
			blockNumber: verification.blockNumber,
			gasUsed: verification.gasUsed || "0",
			gasPrice: "0",
			sourceBalanceInfo: {
				sourceBalanceType:
					originalTransaction.metadata?.sourceBalanceType || "others",
				packageName:
					originalTransaction.metadata?.packageName || "Other Earnings",
				packageType: originalTransaction.metadata?.packageType || "other",
				availableBalance: originalTransaction.metadata?.availableBalance || 0,
				isProportional: originalTransaction.metadata?.isProportional || false,
			},
			metadata: {
				walletBalanceBefore,
				fromAddress,
				refundTxInfo,
				originalTransactionId: transactionId.toString(),
				originalAmountUSD: Math.abs(originalTransaction.amountUSD),
				currentTokenPrice: currentTokenValue,
				currentAmountUSD: currentAmountUSD,
				blockchainVerification: {
					verified: true,
					confirmations: verification.confirmations,
					verifiedAmount: verification.amount,
				},
			},
		});

		logger.info("Transaction marked as refunded", {
			transactionId,
			refundTxHash,
			verifiedAmount: verification.amount,
			blockNumber: verification.blockNumber,
		});

		// Notify user about completed refund
		try {
			const amountNTE = Math.abs(originalTransaction.amountNTE || 0);
			const bodyContent = `
				<div style="color: #e1e4eb; line-height: 1.8;">
					<h2 style="color: #10b981; margin-bottom: 16px; font-size: 22px;">Refund Completed</h2>
					<p style="margin-bottom: 16px;">Your refund has been processed successfully and sent back to your wallet.</p>
					<div style="background: rgba(16, 185, 129, 0.12); border-left: 4px solid #10b981; padding: 16px; border-radius: 8px; margin-bottom: 18px;">
						<table style="width: 100%; border-collapse: collapse; font-size: 14px;">
							<tr>
								<td style="padding: 6px 0; color: #9aa3b2;">Amount Refunded:</td>
								<td style="padding: 6px 0; color: #00e5ff; text-align: right; font-weight: 600;">${amountNTE.toFixed(
									6,
								)} NTE</td>
							</tr>
							<tr>
								<td style="padding: 6px 0; color: #9aa3b2;">Transaction Hash:</td>
								<td style="padding: 6px 0; color: #ffffff; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${refundTxHash}</td>
							</tr>
							<tr>
								<td style="padding: 6px 0; color: #9aa3b2;">Block Number:</td>
								<td style="padding: 6px 0; color: #ffffff; text-align: right;">#${
									verification.blockNumber || "Pending"
								}</td>
							</tr>
							<tr>
								<td style="padding: 6px 0; color: #9aa3b2;">Original Transaction:</td>
								<td style="padding: 6px 0; color: #ffffff; text-align: right; font-family: monospace; font-size: 11px;">${
									originalTransaction._id.toString()
								}</td>
							</tr>
						</table>
					</div>
					<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 12px 16px; border-radius: 8px; margin-bottom: 18px;">
						<p style="margin: 0; color: #e1e4eb; font-size: 13px;">You can verify this refund on the blockchain explorer using the transaction hash above.</p>
					</div>
					<p style="margin-top: 8px; color: #9aa3b2; font-size: 13px;">If you have any questions about this refund, please contact support with the transaction hash.</p>
				</div>
			`;

			const emailHtml = generateEmailFromTemplate(bodyContent);

			await systemNotificationService.sendToUser({
				username: user.username,
				email: user.email,
				subject: "Refund Completed - NodeMeta",
				title: "Refund Processed Successfully",
				body: `Your refund of ${amountNTE.toFixed(
						6,
					)} NTE has been processed. TX: ${refundTxHash}.`,
				html: emailHtml,
				pushPayload: {
					icon: "/icons/icon-192x192.png",
					badge: "/icons/badge-72x72.png",
					data: {
						url: "/dashboard/transactions",
						transactionHash: refundTxHash,
						originalTransactionId: transactionId.toString(),
					},
				},
				sendEmail: true,
				sendPush: true,
			});
		} catch (notificationError) {
			logger.error("Failed to send refund completed notification:", {
				transactionId,
				refundTxHash,
				errorMessage: notificationError.message,
				errorStack: notificationError.stack,
			});
		}

		return {
			success: true,
			message: "Transaction marked as refunded",
			verification: {
				verified: true,
				amount: verification.amount,
				blockNumber: verification.blockNumber,
				confirmations: verification.confirmations,
			},
		};
	},

	async logRefundTransaction({
		username,
		userId,
		toAddress,
		amountNTE,
		amountUSD,
		transactionHash,
		blockNumber,
		gasUsed,
		gasPrice,
		sourceBalanceInfo,
		metadata,
	}) {
		try {
			const result = await recordComprehensiveTransaction({
				username,
				userId,
				transactionType: "withdrawal",
				bonusType: "refund",
				amountNTE: -Math.abs(amountNTE),
				amountUSD: -Math.abs(amountUSD),
				walletAddress: toAddress,
				transactionHash,
				blockchainDetails: {
					blockNumber,
					gasUsed,
					gasPrice,
				},
				sourceBalanceInfo,
				metadata: {
					...metadata,
					isRefund: true,
				},
			});

			return { success: true, result };
		} catch (error) {
			logger.error("Error logging refund transaction:", error);
			return { success: false, error: error.message };
		}
	},

	async getCompanyWallets() {
		try {
			const products = await Product.aggregate([
				{
					$match: {
						$or: [
							{ name: { $regex: "core", $options: "i" } },
							{ name: { $regex: "elite", $options: "i" } },
							{ name: { $regex: "stak", $options: "i" } },
							{ name: { $regex: "meta", $options: "i" } },
							{ name: { $regex: "pulse", $options: "i" } },
						],
					},
				},
				{
					$project: {
						name: { $toLower: "$name" },
						reserveWalletAddress: 1,
					},
				},
			]);

			const COMPANY_WALLET = {
				core: null,
				elite: null,
				meta_pulse: null,
				others: null,
			};

			products.forEach((wallet) => {
				if (wallet.name && wallet.reserveWalletAddress) {
					const category = wallet.name.includes("core")
						? "core"
						: wallet.name.includes("elite")
							? "elite"
							: wallet.name.includes("meta") || wallet.name.includes("pulse")
								? "meta_pulse"
								: "others";

					COMPANY_WALLET[category] = wallet.reserveWalletAddress;
				}
			});

			return {
				success: true,
				data: COMPANY_WALLET,
			};
		} catch (error) {
			logger.error("Error fetching company wallets:", error);
			throw new ApiError(500, "Failed to fetch company wallets");
		}
	},

	async getDynamicPricingStatus(productId) {
		try {
			const product = await Product.findById(productId);

			if (!product) {
				throw new ApiError(404, "Product not found");
			}

			const soldCount = await UserPackage.countDocuments({
				packageId: productId,
			});

			const isDynamicPricingEnabled =
				product.price?.startUSD > 0 &&
				(!product.price?.fixedUSD || product.price.fixedUSD <= 0) &&
				product.price?.countForPriceIncrease > 0 &&
				product.price?.increaseSalePriceUSD > 0;

			return {
				productId: product._id,
				productName: product.name,
				isDynamicPricingEnabled,
				currentPrice:
					product.dynamicPricing?.currentPriceUSD ||
					product.price?.startUSD ||
					0,
				basePrice:
					product.dynamicPricing?.basePriceUSD || product.price?.startUSD || 0,
				soldCount,
				totalPriceIncreases: product.dynamicPricing?.priceIncreases || 0,
				countForPriceIncrease: product.price?.countForPriceIncrease || 0,
				increaseSalePriceUSD: product.price?.increaseSalePriceUSD || 0,
				lastPriceUpdate: product.dynamicPricing?.lastPriceUpdate,
				priceHistory: product.dynamicPricing?.priceHistory || [],
				nextPriceIncreaseAt: isDynamicPricingEnabled
					? Math.ceil(soldCount / product.price.countForPriceIncrease) *
						product.price.countForPriceIncrease
					: null,
			};
		} catch (error) {
			logger.error("Error fetching dynamic pricing status:", error);
			throw error instanceof ApiError
				? error
				: new ApiError(500, "Failed to fetch dynamic pricing status");
		}
	},

	async getWithdrawalConfig(user) {
		try {
			//get last calim time
			const lastClaim = await Withdrawal.findOne({ username: user.username })
				.sort({ createdAt: -1 })
				.lean();

			const userInfo = await User.findOne({ username: user.username }).lean();

			// Get current NTE price from blockchain
			const currentNTEPrice = await getTokenValue();

			if (!currentNTEPrice || currentNTEPrice <= 0) {
				throw new ApiError(503, "Unable to fetch current token price");
			}

			// Withdrawal limits
			const MAX_WITHDRAWAL_USD = CLAIM_LIMITS.MAX_CLAIM_PER_TRANSACTION_USD;
			const MIN_WITHDRAWAL_NTE = CLAIM_LIMITS.MIN_AMOUNT_NTE;

			// Calculate max NTE based on current price
			const maxNTEAllowed = Math.floor(MAX_WITHDRAWAL_USD / currentNTEPrice);
			const minNTEAllowed = Math.floor(MIN_WITHDRAWAL_NTE * currentNTEPrice);

			return {
				limits: {
					maxWithdrawalUSD: MAX_WITHDRAWAL_USD,
					minWithdrawalNTE: MIN_WITHDRAWAL_NTE,
					maxNTEAllowed: maxNTEAllowed,
					minNTEAllowed: minNTEAllowed,
				},
				pricing: {
					currentNTEPrice: currentNTEPrice,
					lastUpdated: new Date().toISOString(),
				},
				restrictions: {
					cooldownSeconds: CLAIM_LIMITS.COOLDOWN_MINUTES_USER * 60,
					lastClaimAt: lastClaim ? lastClaim.createdAt : null,
					waitingPeriodSeconds: lastClaim
						? Math.max(
								0,
								CLAIM_LIMITS.COOLDOWN_MINUTES_USER * 60 -
									(Math.floor(
										(Date.now() - new Date(lastClaim.createdAt)) / 1000,
									) || 0),
							)
						: 0,
					havePermission:
						!userInfo.withdrawalsDisabled &&
						userInfo.isActive &&
						!userInfo.isBlocked &&
						!user.twoFactorEnabled,
				},
			};
		} catch (error) {
			logger.error("Error fetching withdrawal config:", error);
			throw error instanceof ApiError
				? error
				: new ApiError(500, "Failed to retrieve withdrawal configuration");
		}
	},
};

module.exports = { adminService };
