const { AdminOverride } = require("./adminOverride.model");
const { logger } = require("../../core/logger/logger");
const { cacheService } = require("../../core/cache/cache.service");

const CACHE_KEY = "admin:overrides";
const CACHE_TTL = 60; // 1 minute in seconds

async function getAdminOverrides() {
	try {
		// Check cache first
		const cached = await cacheService.get(CACHE_KEY);
		if (cached) {
			return cached;
		}

		logger.debug("Fetching admin overrides from database");

		// Get active overrides that haven't expired
		const activeOverrides = await AdminOverride.find({
			expiresAt: { $gt: new Date() },
			isActive: true,
		})
			.sort({ setAt: -1 })
			.lean();

		const processedOverrides = {
			tokenValue: null,
			totalUsers: null,
			totalTax: null,
		};

		// Process overrides (take the most recent one for each type)
		activeOverrides.forEach((override) => {
			if (override.type === "TOKEN_VALUE" && !processedOverrides.tokenValue) {
				processedOverrides.tokenValue = {
					value: override.value,
					setBy: override.setBy,
					setAt: override.setAt,
					expiresAt: override.expiresAt,
					reason: override.reason,
				};
			} else if (
				override.type === "TOTAL_USERS" &&
				!processedOverrides.totalUsers
			) {
				processedOverrides.totalUsers = {
					value: override.value,
					setBy: override.setBy,
					setAt: override.setAt,
					expiresAt: override.expiresAt,
					reason: override.reason,
				};
			} else if (
				override.type === "TOTAL_TAX" &&
				!processedOverrides.totalTax
			) {
				processedOverrides.totalTax = {
					value: override.value,
					setBy: override.setBy,
					setAt: override.setAt,
					expiresAt: override.expiresAt,
					reason: override.reason,
				};
			}
		});

		// Update cache
		await cacheService.set(CACHE_KEY, processedOverrides, CACHE_TTL);

		return processedOverrides;
	} catch (error) {
		logger.error("Error getting admin overrides:", {
			message: error.message,
			stack: error.stack,
			name: error.name
		});

		// Return empty overrides as fallback
		return {
			tokenValue: null,
			totalUsers: null,
			totalTax: null,
		};
	}
}

// Create a new admin override
async function createAdminOverride({
	type,
	value,
	setBy,
	durationMinutes,
	reason,
}) {
	try {
		const now = new Date();
		const expiresAt = new Date(now.getTime() + durationMinutes * 60 * 1000);

		const override = new AdminOverride({
			type,
			value,
			setBy,
			setAt: now,
			expiresAt,
			durationMinutes,
			reason,
			isActive: true,
		});

		await override.save();

		// Clear cache to force refresh
		await cacheService.delete(CACHE_KEY);

		logger.info(`Admin override created: ${type} = ${value} by ${setBy}`);

		return override;
	} catch (error) {
		logger.error("Error creating admin override:", error);
		throw error;
	}
}

// Deactivate an admin override
async function deactivateAdminOverride(overrideId) {
	try {
		const override = await AdminOverride.findByIdAndUpdate(
			overrideId,
			{ isActive: false, deactivatedAt: new Date() },
			{ new: true },
		);

		if (!override) {
			throw new Error("Admin override not found");
		}

		// Clear cache to force refresh
		await cacheService.delete(CACHE_KEY);

		logger.info(`Admin override deactivated: ${override.type}`);

		return override;
	} catch (error) {
		logger.error("Error deactivating admin override:", error);
		throw error;
	}
}

// Get all admin overrides
async function getAllAdminOverrides({
	limit = 50,
	skip = 0,
	type = null,
} = {}) {
	try {
		const query = type ? { type } : {};

		const overrides = await AdminOverride.find(query)
			.sort({ setAt: -1 })
			.limit(limit)
			.skip(skip)
			.lean();

		const total = await AdminOverride.countDocuments(query);

		return {
			overrides,
			total,
			limit,
			skip,
		};
	} catch (error) {
		logger.error("Error getting all admin overrides:", error);
		throw error;
	}
}

// Clear expired overrides
async function clearExpiredOverrides() {
	try {
		const result = await AdminOverride.deleteMany({
			expiresAt: { $lt: new Date() },
		});

		if (result.deletedCount > 0) {
			logger.info(`Cleared ${result.deletedCount} expired admin overrides`);
			await cacheService.delete(CACHE_KEY);
		}

		return result.deletedCount;
	} catch (error) {
		logger.error("Error clearing expired overrides:", error);
		throw error;
	}
}

// Clear cache
async function clearCache() {
	await cacheService.delete(CACHE_KEY);
	logger.info("Admin override cache cleared");
}

module.exports = {
	getAdminOverrides,
	createAdminOverride,
	deactivateAdminOverride,
	getAllAdminOverrides,
	clearExpiredOverrides,
	clearCache,
};
