const tokenStatsService = require("./tokenStats.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");

class TokenStatsController {
	/**
	 * Get all tokens with pagination and filtering
	 */
	async getAllTokens(req, res, next) {
		try {
			const {
				page = 1,
				limit = 20,
				search = "",
				status = "all",
			} = req.query;

			const result = await tokenStatsService.getAllTokens({
				page: parseInt(page),
				limit: parseInt(limit),
				search,
				status,
			});

			return res
				.status(200)
				.json(new ApiResponse(200, result, "Tokens retrieved successfully"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Invalidate a specific token or all tokens for a user
	 */
	async invalidateToken(req, res, next) {
		try {
			const { token, tokenId, username } = req.body;

			if (!token && !tokenId && !username) {
				throw new ApiError(
					400,
					"Either token, tokenId, or username is required",
				);
			}

			let result;

			if (username) {
				// Invalidate all tokens for user
				result = await tokenStatsService.invalidateUserTokens(username);
			} else if (tokenId) {
				// Invalidate specific token by tokenId
				result = await tokenStatsService.invalidateTokenById(tokenId);
			} else if (token) {
				// Invalidate specific token by token string
				result = await tokenStatsService.invalidateToken(token);
			}

			return res
				.status(200)
				.json(new ApiResponse(200, result, result.message));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Cleanup expired tokens
	 */
	async cleanupExpiredTokens(req, res, next) {
		try {
			const result = await tokenStatsService.cleanupExpiredTokens();

			return res
				.status(200)
				.json(new ApiResponse(200, result, result.message));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Get token cache statistics
	 */
	async getCacheStats(req, res, next) {
		try {
			const stats = await tokenStatsService.getCacheStats();

			return res
				.status(200)
				.json(new ApiResponse(200, stats, "Cache stats retrieved"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Force logout specific user(s) - Admin only
	 */
	async forceLogoutUser(req, res, next) {
		try {
			const { username, usernames } = req.body;

			if (!username && !usernames) {
				throw new ApiError(400, "Username or usernames array is required");
			}

			let result;

			if (usernames && Array.isArray(usernames)) {
				// Force logout multiple users
				result = await tokenStatsService.forceLogoutUsers(usernames);
			} else if (username) {
				// Force logout single user
				result = await tokenStatsService.invalidateUserTokens(username);
			}

			return res
				.status(200)
				.json(new ApiResponse(200, result, result.message));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Force logout ALL users - Admin only (nuclear option)
	 */
	async forceLogoutAllUsers(req, res, next) {
		try {
			const { confirm } = req.body;

			if (confirm !== "FORCE_LOGOUT_ALL") {
				throw new ApiError(
					400,
					'Please confirm by sending { "confirm": "FORCE_LOGOUT_ALL" }',
				);
			}

			const result = await tokenStatsService.forceLogoutAllUsers();

			return res
				.status(200)
				.json(new ApiResponse(200, result, result.message));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Get all active sessions for a specific user
	 */
	async getUserSessions(req, res, next) {
		try {
			const { username } = req.params;

			if (!username) {
				throw new ApiError(400, "Username is required");
			}

			const result = await tokenStatsService.getUserSessions(username);

			return res
				.status(200)
				.json(
					new ApiResponse(200, result, `Sessions retrieved for ${username}`),
				);
		} catch (error) {
			next(error);
		}
	}
}

module.exports = new TokenStatsController();
