const { ApiError } = require("../../utils/ApiError");
const { cacheService } = require("../../core/cache/cache.service");

const BLACKLIST_TTL = 24 * 60 * 60; // 24 hours in seconds

class TokenStatsService {
	/**
	 * Get all active tokens from cache
	 */
	async #getAllActiveTokens() {
		const tokens = [];
		const cacheStore = cacheService.store;

		if (!cacheStore || !(cacheStore instanceof Map)) {
			return tokens;
		}

		for (const [key, record] of cacheStore.entries()) {
			if (key.startsWith("token:active:")) {
				try {
					const token = key.replace("token:active:", "");
					const tokenData = record.value;

					if (!tokenData) {
						continue;
					}

					// Check if blacklisted
					const isBlacklisted = await cacheService.exists(
						`token:blacklist:${token}`,
					);

					// Check if expired
					const isExpired = record.expiresAt && record.expiresAt <= Date.now();

					tokens.push({
						token,
						tokenId: tokenData.tokenId || "unknown",
						userId: tokenData.userId || "unknown",
						username: tokenData.username || "unknown",
						walletAddress: tokenData.walletAddress || null,
						createdAt: tokenData.createdAt || Date.now(),
						expiresAt: record.expiresAt || null,
						ip: tokenData.ip || "unknown",
						userAgent: tokenData.userAgent || "unknown",
						isExpired,
						isBlacklisted,
					});
				} catch (error) {}
			}
		}

		return tokens;
	}

	/**
	 * Get all tokens with filtering and pagination
	 */
	async getAllTokens({ page = 1, limit = 20, search = "", status = "all" }) {
		let tokens = await this.#getAllActiveTokens();

		// Apply status filter
		if (status !== "all") {
			tokens = tokens.filter((token) => {
				if (status === "active") {
					return !token.isBlacklisted && !token.isExpired;
				} else if (status === "blacklisted") {
					return token.isBlacklisted;
				} else if (status === "expired") {
					return token.isExpired && !token.isBlacklisted;
				}
				return true;
			});
		}

		// Apply search filter
		if (search) {
			const searchLower = search.toLowerCase();
			tokens = tokens.filter(
				(token) =>
					token.username?.toLowerCase().includes(searchLower) ||
					token.walletAddress?.toLowerCase().includes(searchLower) ||
					token.ip?.toLowerCase().includes(searchLower) ||
					token.tokenId?.toLowerCase().includes(searchLower) ||
					token.userId?.toLowerCase().includes(searchLower),
			);
		}

		// Sort by createdAt (most recent first)
		tokens.sort((a, b) => b.createdAt - a.createdAt);

		// Calculate stats
		const allTokens = await this.#getAllActiveTokens();
		const stats = {
			activeTokens: allTokens.filter((t) => !t.isBlacklisted && !t.isExpired)
				.length,
			blacklistedTokens: allTokens.filter((t) => t.isBlacklisted).length,
			expiredTokens: allTokens.filter((t) => t.isExpired && !t.isBlacklisted)
				.length,
			totalTokens: allTokens.length,
		};

		// Pagination
		const total = tokens.length;
		const totalPages = Math.ceil(total / limit);
		const start = (page - 1) * limit;
		const end = start + limit;
		const paginatedTokens = tokens.slice(start, end);

		return {
			tokens: paginatedTokens,
			pagination: {
				page,
				limit,
				total,
				totalPages,
			},
			stats,
		};
	}

	/**
	 * Invalidate a specific token by token string
	 */
	async invalidateToken(token) {
		const tokenData = await cacheService.get(`token:active:${token}`);

		if (!tokenData) {
			throw new ApiError(404, "Token not found");
		}

		// Remove from active tracking
		await cacheService.delete(`token:active:${token}`);
		await cacheService.delete(`token:${tokenData.tokenId}`);

		// Add to blacklist
		await cacheService.set(`token:blacklist:${token}`, true, BLACKLIST_TTL);

		return { success: true, message: "Token invalidated successfully" };
	}

	/**
	 * Invalidate a specific token by tokenId
	 */
	async invalidateTokenById(tokenId) {
		const tokens = await this.#getAllActiveTokens();
		const tokenToInvalidate = tokens.find((t) => t.tokenId === tokenId);

		if (!tokenToInvalidate) {
			throw new ApiError(404, "Token not found");
		}

		return await this.invalidateToken(tokenToInvalidate.token);
	}

	/**
	 * Invalidate all tokens for a user
	 */
	async invalidateUserTokens(username) {
		const tokens = await this.#getAllActiveTokens();
		let invalidatedCount = 0;

		for (const tokenData of tokens) {
			if (tokenData.username === username) {
				await this.invalidateToken(tokenData.token);
				invalidatedCount++;
			}
		}

		return {
			success: true,
			message: `Invalidated ${invalidatedCount} token(s) for user ${username}`,
			count: invalidatedCount,
		};
	}

	/**
	 * Force logout multiple users (admin only)
	 */
	async forceLogoutUsers(usernames) {
		const results = [];
		let totalInvalidated = 0;

		for (const username of usernames) {
			try {
				const result = await this.invalidateUserTokens(username);
				results.push({ username, ...result });
				totalInvalidated += result.count;
			} catch (error) {
				results.push({
					username,
					success: false,
					message: error.message,
					count: 0,
				});
			}
		}

		return {
			success: true,
			message: `Force logged out ${usernames.length} user(s), invalidated ${totalInvalidated} token(s)`,
			totalInvalidated,
			results,
		};
	}

	/**
	 * Force logout ALL users (admin only - nuclear option)
	 */
	async forceLogoutAllUsers() {
		const tokens = await this.#getAllActiveTokens();
		let invalidatedCount = 0;
		const usernames = new Set();

		for (const tokenData of tokens) {
			if (!tokenData.isBlacklisted && !tokenData.isExpired) {
				await this.invalidateToken(tokenData.token);
				invalidatedCount++;
				usernames.add(tokenData.username);
			}
		}

		return {
			success: true,
			message: `Force logged out all users: ${usernames.size} users, invalidated ${invalidatedCount} token(s)`,
			usersAffected: usernames.size,
			tokensInvalidated: invalidatedCount,
		};
	}

	/**
	 * Get all active sessions for a specific user
	 */
	async getUserSessions(username) {
		const tokens = await this.#getAllActiveTokens();
		const userTokens = tokens.filter((t) => t.username === username);

		const activeSessions = userTokens.filter(
			(t) => !t.isBlacklisted && !t.isExpired,
		);

		return {
			username,
			totalSessions: userTokens.length,
			activeSessions: activeSessions.length,
			sessions: userTokens.map((t) => ({
				tokenId: t.tokenId,
				ip: t.ip,
				userAgent: t.userAgent,
				createdAt: t.createdAt,
				expiresAt: t.expiresAt,
				isActive: !t.isBlacklisted && !t.isExpired,
				isExpired: t.isExpired,
				isBlacklisted: t.isBlacklisted,
			})),
		};
	}

	/**
	 * Cleanup expired tokens
	 */
	async cleanupExpiredTokens() {
		const tokens = await this.#getAllActiveTokens();
		let cleanedCount = 0;

		for (const tokenData of tokens) {
			if (tokenData.isExpired) {
				await cacheService.delete(`token:active:${tokenData.token}`);
				await cacheService.delete(`token:${tokenData.tokenId}`);
				cleanedCount++;
			}
		}

		const stats = await this.getCacheStats();

		return {
			success: true,
			message: `Cleaned up ${cleanedCount} expired token(s)`,
			cleanedCount,
			stats,
		};
	}

	/**
	 * Get cache statistics
	 */
	async getCacheStats() {
		const tokens = await this.#getAllActiveTokens();

		const stats = {
			activeTokens: tokens.filter((t) => !t.isBlacklisted && !t.isExpired)
				.length,
			blacklistedTokens: tokens.filter((t) => t.isBlacklisted).length,
			expiredTokens: tokens.filter((t) => t.isExpired && !t.isBlacklisted)
				.length,
			totalTokens: tokens.length,
			totalCachedItems: cacheService.getStats().size,
		};

		return stats;
	}
}

module.exports = new TokenStatsService();
