const { blogService } = require("./blog.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");

const blogController = {
	// Get all blogs with search, filters and pagination
	async listBlogs(req, res, next) {
		try {
			const {
				page = 1,
				limit = 20,
				search = "",
				category = "",
				sortBy = "createdAt",
				sortOrder = "desc",
				admin = false,
			} = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				search,
				category,
				sortBy,
				sortOrder,
				includeInactive: admin === "true" && req.user?.role === "admin",
			};

			const result = await blogService.listBlogs(options);
			res.json(new ApiResponse(200, result, "Blogs retrieved successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Fetch a blog using its slug
	async getBlogBySlug(req, res, next) {
		try {
			const { slug } = req.params;
			const admin = req.query.admin === "true";

			const includeInactive = admin && req.user?.role === "admin";
			const blog = await blogService.getBlogBySlug(slug, includeInactive);

			if (!blog) {
				throw new ApiError(404, "Blog not found");
			}

			res.json(new ApiResponse(200, { blog }, "Blog retrieved successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Create a new blog post - admin only
	async createBlog(req, res, next) {
		try {
			const blogData = {
				title: req.body.title,
				subtitle: req.body.subtitle || "",
				description: req.body.description || "",
				content: req.body.content,
				category: req.body.category,
				author: req.body.author,
				date: req.body.date,
				featured: req.body.featured === "true",
				slug: req.body.slug || "",
			};

			const files = req.files || [];
			const uploadConfig = req.uploadConfig;
			const blog = await blogService.createBlog(blogData, files, uploadConfig);

			res
				.status(201)
				.json(new ApiResponse(201, { blog }, "Blog created successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Update blog post - admin only
	async updateBlog(req, res, next) {
		try {
			const { id } = req.params;
			const blogData = {
				title: req.body.title,
				subtitle: req.body.subtitle || "",
				description: req.body.description || "",
				content: req.body.content,
				category: req.body.category,
				author: req.body.author,
				date: req.body.date,
				featured: req.body.featured === "true",
				slug: req.body.slug || "",
			};

			const files = req.files || [];
			const existingImage = req.body.existingImage;
			const existingAttachments = req.body.existingAttachments
				? JSON.parse(req.body.existingAttachments)
				: [];
			const uploadConfig = req.uploadConfig;

			const blog = await blogService.updateBlog(
				id,
				blogData,
				files,
				existingImage,
				existingAttachments,
				uploadConfig,
			);

			res.json(new ApiResponse(200, { blog }, "Blog updated successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Delete a blog post
	async deleteBlog(req, res, next) {
		try {
			const { id } = req.params;
			await blogService.deleteBlog(id);

			res.json(
				new ApiResponse(200, { deletedId: id }, "Blog deleted successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Activate or deactivate a blog
	async toggleBlogStatus(req, res, next) {
		try {
			const { id } = req.params;
			const { active } = req.body;

			if (typeof active !== "boolean") {
				throw new ApiError(400, "Active field must be a boolean");
			}

			const blog = await blogService.toggleBlogStatus(id, active);

			res.json(
				new ApiResponse(
					200,
					{ blog },
					`Blog ${active ? "activated" : "deactivated"} successfully`,
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Handle view and like actions on blogs
	async handleAction(req, res, next) {
		try {
			const { blogId, action, userId } = req.body;

			if (!blogId || !action) {
				throw new ApiError(400, "blogId and action are required");
			}

			if (!["view", "like"].includes(action)) {
				throw new ApiError(400, 'Invalid action. Must be "view" or "like"');
			}

			if (action === "like" && !userId) {
				throw new ApiError(400, "userId is required for like action");
			}

			const result = await blogService.handleBlogAction(blogId, action, userId);

			res.json(
				new ApiResponse(200, result, `Blog ${action} recorded successfully`),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get all comments for a blog post
	async getComments(req, res, next) {
		try {
			const { id } = req.params;

			const includeHidden = req.user?.role === "admin";
			const comments = await blogService.getComments(id, includeHidden);

			res.json(
				new ApiResponse(200, { comments }, "Comments retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Add a comment to a blog
	async createComment(req, res, next) {
		try {
			const { id } = req.params;
			const { text, author, userId, parentId } = req.body;

			if (!text || !author) {
				throw new ApiError(400, "text and author are required");
			}

			// Clean and validate parentId
			const cleanParentId = parentId?.trim();

			const commentData = {
				blogId: id,
				text: text.trim(),
				author: author.trim(),
				userId: userId?.trim() || `anonymous_${Date.now()}`,
				parentId: cleanParentId || null,
			};

			const comment = await blogService.createComment(commentData);

			res.status(201).json(
				new ApiResponse(
					201,
					{
						comment,
						isReply: !!cleanParentId,
						parentId: cleanParentId,
					},
					"Comment created successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Toggle comment visibility (hide/show) - admin only
	async toggleCommentVisibility(req, res, next) {
		try {
			const { commentId } = req.params;
			const { hidden } = req.body;

			if (typeof hidden !== "boolean") {
				throw new ApiError(400, "hidden field must be a boolean");
			}

			const comment = await blogService.toggleCommentVisibility(
				commentId,
				hidden,
			);

			res.json(
				new ApiResponse(
					200,
					{ comment },
					`Comment ${hidden ? "hidden" : "shown"} successfully`,
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Delete a comment - admin only
	async deleteComment(req, res, next) {
		try {
			const { commentId } = req.params;

			await blogService.deleteComment(commentId);

			res.json(
				new ApiResponse(
					200,
					{ deletedId: commentId },
					"Comment deleted successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},
};

module.exports = { blogController };
