const express = require("express");
const { blogController } = require("./blog.controller");
const { secure } = require("../../middlewares/security.middleware");
const { uploadManager } = require("../../utils/upload.util");

const router = express.Router();

// Middleware to set where files should be uploaded
const setUploadConfig = (folder, prefix) => (req, res, next) => {
	req.uploadConfig = { folder, prefix };
	next();
};

// Configure upload for blog image (max 1 file, 5MB)
const uploadBlogImage = uploadManager.array("image", 1);

// Public routes anyone can access
router.get("/", blogController.listBlogs);
router.get(
	"/all-blogs",
	...secure("AR", { role: "admin" }),
	blogController.listBlogs,
);
router.get("/slug/:slug", blogController.getBlogBySlug);
router.get("/:id/comments", blogController.getComments);
router.get(
	"/:id/comments/all",
	...secure("AR", { role: "admin" }),
	blogController.getComments,
);
router.post("/actions", blogController.handleAction);
router.post("/:id/comments", blogController.createComment);

// Admin only routes
router.post(
	"/",
	...secure("AR", { role: "admin" }),
	setUploadConfig("blogs", "blog"),
	uploadBlogImage,
	blogController.createBlog,
);
router.put(
	"/:id",
	...secure("AR", { role: "admin" }),
	setUploadConfig("blogs", "blog"),
	uploadBlogImage,
	blogController.updateBlog,
);
router.delete(
	"/:id",
	...secure("AR", { role: "admin" }),
	blogController.deleteBlog,
);
router.patch(
	"/:id/status",
	...secure("AR", { role: "admin" }),
	blogController.toggleBlogStatus,
);

// Comment management routes (admin only)
router.patch(
	"/comments/:commentId/visibility",
	...secure("AR", { role: "admin" }),
	blogController.toggleCommentVisibility,
);
router.delete(
	"/comments/:commentId",
	...secure("AR", { role: "admin" }),
	blogController.deleteComment,
);

module.exports = { blogRoutes: router };
