const { Blog } = require("./blog.model");
const { Comment } = require("./comment.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const { uploadManager } = require("../../utils/upload.util");

const SERVER_URL = process.env.SERVER_URL || "http://localhost:5000";

// Helper to prepend server URL to image path
const getFullImageUrl = (imagePath) => {
	if (!imagePath) return "";
	if (imagePath.startsWith("http://") || imagePath.startsWith("https://")) {
		return imagePath;
	}
	return `${SERVER_URL}${imagePath}`;
};

// Calculate reading time for blog content
const calculateReadTime = (content) => {
	const wordsPerMinute = 200;
	const textContent = content.replace(/<[^>]*>/g, "");
	const wordCount = textContent
		.split(/\s+/)
		.filter((word) => word.length > 0).length;
	const readTime = Math.ceil(wordCount / wordsPerMinute);
	return `${readTime} min read`;
};

// Turn title into URL-friendly slug
const generateSlug = (title) => {
	return title
		.toLowerCase()
		.replace(/[^a-z0-9 -]/g, "")
		.replace(/\s+/g, "-")
		.replace(/-+/g, "-")
		.trim()
		.replace(/^-+|-+$/g, "");
};

// Make sure slug is unique by adding counter if needed
const ensureUniqueSlug = async (baseSlug, excludeId = null) => {
	let slug = baseSlug;
	let counter = 1;

	while (true) {
		const filter = { slug };
		if (excludeId) {
			filter._id = { $ne: excludeId };
		}

		const existingBlog = await Blog.findOne(filter);
		if (!existingBlog) {
			break;
		}

		slug = `${baseSlug}-${counter}`;
		counter++;
	}

	return slug;
};

const blogService = {
	// Get blogs list with search and pagination
	async listBlogs(options) {
		const {
			page = 1,
			limit = 20,
			search = "",
			category = "",
			sortBy = "createdAt",
			sortOrder = "desc",
			includeInactive = false,
		} = options;

		const skip = (page - 1) * limit;
		const filter = {};

		if (!includeInactive) {
			filter.active = true;
		}

		if (search) {
			filter.$or = [
				{ title: { $regex: search, $options: "i" } },
				{ description: { $regex: search, $options: "i" } },
				{ content: { $regex: search, $options: "i" } },
				{ slug: { $regex: search, $options: "i" } },
			];
		}

		if (category) {
			filter.category = category;
		}

		const sortOptions = {};
		sortOptions[sortBy] = sortOrder === "asc" ? 1 : -1;

		const [blogs, totalBlogs] = await Promise.all([
			Blog.find(filter).sort(sortOptions).skip(skip).limit(limit).lean(),
			Blog.countDocuments(filter),
		]);

		// Add server URL to image paths
		const blogsWithFullUrls = blogs.map((blog) => ({
			...blog,
			image: getFullImageUrl(blog.image),
		}));

		return {
			blogs: blogsWithFullUrls,
			pagination: {
				page,
				limit,
				total: totalBlogs,
				totalPages: Math.ceil(totalBlogs / limit),
			},
		};
	},

	// Find blog by slug
	async getBlogBySlug(slug, includeInactive = false) {
		const filter = { slug };
		if (!includeInactive) {
			filter.active = true;
		}

		const blog = await Blog.findOne(filter).lean();
		if (blog) {
			blog.image = getFullImageUrl(blog.image);
		}
		return blog;
	},

	// Find blog by ID
	async getBlogById(id, includeInactive = false) {
		const filter = { _id: id };
		if (!includeInactive) {
			filter.active = true;
		}

		const blog = await Blog.findOne(filter).lean();
		if (!blog) {
			throw new ApiError(404, "Blog not found");
		}

		blog.image = getFullImageUrl(blog.image);
		return blog;
	},

	// Create a new blog post
	async createBlog(blogData, files = [], uploadConfig) {
		const {
			title,
			subtitle,
			description,
			content,
			category,
			author,
			date,
			featured,
			slug,
		} = blogData;

		if (!title || !content || !category || !author) {
			throw new ApiError(
				400,
				"Missing required fields: title, content, category, author",
			);
		}

		// Extract single file from files array
		const imageFile = files[0];

		if (!imageFile) {
			throw new ApiError(400, "Image is required");
		}

		// Upload the blog image
		const imageUrl = await uploadManager.uploadImage(
			imageFile,
			uploadConfig.folder,
			uploadConfig.prefix,
		);

		// Create unique slug
		let finalSlug = slug || generateSlug(title);
		finalSlug = generateSlug(finalSlug);
		finalSlug = await ensureUniqueSlug(finalSlug);

		// Figure out how long it takes to read
		const readTime = calculateReadTime(content);

		const blog = new Blog({
			title,
			subtitle,
			description,
			content,
			category,
			author,
			date: date || new Date().toISOString().split("T")[0],
			featured: featured || false,
			image: imageUrl,
			slug: finalSlug,
			readTime,
			active: true,
			views: 0,
			likes: 0,
			likedBy: [],
		});

		await blog.save();

		logger.info("Blog created", { blogId: blog._id, slug: blog.slug });

		const blogObject = blog.toObject();
		blogObject.image = getFullImageUrl(blogObject.image);
		return blogObject;
	},

	// Update blog post
	async updateBlog(
		id,
		blogData,
		files = [],
		existingImage,
		existingAttachments = [],
		uploadConfig,
	) {
		const blog = await Blog.findById(id);
		if (!blog) {
			throw new ApiError(404, "Blog not found");
		}

		const {
			title,
			subtitle,
			description,
			content,
			category,
			author,
			date,
			featured,
			slug,
		} = blogData;

		if (!title || !content || !category || !author) {
			throw new ApiError(
				400,
				"Missing required fields: title, content, category, author",
			);
		}

		// Extract single file from files array
		const imageFile = files[0];

		// Deal with image updates
		let imageUrl = blog.image;
		if (imageFile) {
			// Replace old image with new one
			imageUrl = await uploadManager.replaceImage(
				imageFile,
				blog.image,
				uploadConfig.folder,
				uploadConfig.prefix,
			);
		} else if (!existingImage) {
			// If no existing image and no new file, delete old image
			await uploadManager.deleteImage(blog.image);
			imageUrl = "";
		}

		// Update slug if needed
		let finalSlug = slug || title;
		finalSlug = generateSlug(finalSlug);
		finalSlug = await ensureUniqueSlug(finalSlug, id);

		// Recalculate read time if content changed
		const readTime = content ? calculateReadTime(content) : blog.readTime;

		// Update all the fields
		blog.title = title;
		blog.subtitle = subtitle;
		blog.description = description;
		blog.content = content;
		blog.category = category;
		blog.author = author;
		blog.date = date || blog.date;
		blog.featured = featured;
		blog.image = imageUrl;
		blog.slug = finalSlug;
		blog.readTime = readTime;

		await blog.save();

		logger.info("Blog updated", { blogId: blog._id, slug: blog.slug });

		const blogObject = blog.toObject();
		blogObject.image = getFullImageUrl(blogObject.image);
		return blogObject;
	},

	// Delete a blog post
	async deleteBlog(id) {
		const blog = await Blog.findById(id);
		if (!blog) {
			throw new ApiError(404, "Blog not found");
		}

		// Delete the blog's image
		if (blog.image) {
			await uploadManager.deleteImage(blog.image);
		}

		// Delete all comments too
		await Comment.deleteMany({ blogId: id.toString() });

		await Blog.findByIdAndDelete(id);

		logger.info("Blog deleted", { blogId: id });
	},

	// Turn blog on or off
	async toggleBlogStatus(id, active) {
		const blog = await Blog.findById(id);
		if (!blog) {
			throw new ApiError(404, "Blog not found");
		}

		blog.active = active;
		await blog.save();

		logger.info("Blog status toggled", { blogId: blog._id, active });

		const blogObject = blog.toObject();
		blogObject.image = getFullImageUrl(blogObject.image);
		return blogObject;
	},

	// Track views and likes on blogs
	async handleBlogAction(blogId, action, userId) {
		const blog = await Blog.findById(blogId);
		if (!blog) {
			throw new ApiError(404, "Blog not found");
		}

		let hasLiked = false;

		if (action === "view") {
			blog.views += 1;
		} else if (action === "like") {
			const likedBy = blog.likedBy || [];
			const currentlyLiked = likedBy.includes(userId);

			if (currentlyLiked) {
				// User is un-liking
				blog.likes = Math.max(0, blog.likes - 1);
				blog.likedBy = likedBy.filter((id) => id !== userId);
				hasLiked = false;
			} else {
				// User is liking
				blog.likes += 1;
				blog.likedBy.push(userId);
				hasLiked = true;
			}
		}

		await blog.save();

		logger.info("Blog action recorded", { blogId, action, userId });

		return {
			views: blog.views,
			likes: blog.likes,
			hasLiked: action === "like" ? hasLiked : blog.likedBy?.includes(userId),
		};
	},

	// Fetch comments with nested replies
	async getComments(blogId, includeHidden = false) {
		const filter = { blogId };
		if (!includeHidden) {
			filter.hidden = { $ne: true };
		}

		const comments = await Comment.find(filter).sort({ createdAt: -1 }).lean();

		// Build the comment tree with replies
		const commentsMap = new Map();
		const topLevelComments = [];

		comments.forEach((comment) => {
			commentsMap.set(comment._id.toString(), {
				...comment,
				replies: [],
			});
		});

		comments.forEach((comment) => {
			if (comment.parentId) {
				const parentId =
					typeof comment.parentId === "object"
						? comment.parentId.toString()
						: comment.parentId;
				const parent = commentsMap.get(parentId);
				if (parent) {
					parent.replies.push(commentsMap.get(comment._id.toString()));
				} else {
					// If parent not found, treat as top-level
					topLevelComments.push(commentsMap.get(comment._id.toString()));
				}
			} else {
				topLevelComments.push(commentsMap.get(comment._id.toString()));
			}
		});

		return topLevelComments;
	},

	// Add a new comment
	async createComment(commentData) {
		const { blogId, text, author, userId, parentId } = commentData;

		// Make sure the blog exists
		const blog = await Blog.findById(blogId);
		if (!blog) {
			throw new ApiError(404, "Blog not found");
		}

		// Check if parent comment exists when replying
		if (parentId) {
			const parentComment = await Comment.findById(parentId);
			if (!parentComment) {
				throw new ApiError(404, "Parent comment not found");
			}
			if (parentComment.blogId.toString() !== blogId.toString()) {
				throw new ApiError(400, "Parent comment does not belong to this blog");
			}
		}

		const comment = new Comment({
			blogId,
			text,
			author,
			userId: userId || "anonymous",
			parentId: parentId || null,
			hidden: false,
		});

		await comment.save();

		logger.info("Comment created", {
			commentId: comment._id,
			blogId,
			userId,
			parentId: comment.parentId,
			isReply: !!parentId,
		});

		return {
			...comment.toObject(),
			replies: [],
		};
	},

	// Toggle comment visibility (hide/show)
	async toggleCommentVisibility(commentId, hidden) {
		const comment = await Comment.findById(commentId);
		if (!comment) {
			throw new ApiError(404, 'Comment not found');
		}

		comment.hidden = hidden;
		await comment.save();

		logger.info('Comment visibility toggled', { commentId, hidden });

		return comment.toObject();
	},

	// Delete a comment and its replies
	async deleteComment(commentId) {
		const comment = await Comment.findById(commentId);
		if (!comment) {
			throw new ApiError(404, 'Comment not found');
		}

		// Delete all replies to this comment
		await Comment.deleteMany({ parentId: commentId });

		// Delete the comment itself
		await Comment.findByIdAndDelete(commentId);

		logger.info('Comment deleted', { commentId });
	},
};

module.exports = { blogService };
