const mongoose = require("mongoose");
const { BonusDistribution } = require("./bonusDistribution.model");
const { BonusTransaction } = require("./bonusTransaction.model");
const { User } = require("../users/user.model");
const { Product } = require("../products/product.model");
const {
	getTokenValue,
	getTotalTaxEarned,
	isUsersWithActivePackagesCacheValid,
	updateUsersWithActivePackagesCache,
	getTotalUsersForPackageSync,
	hasActiveUnlockedPackageSync,
} = require("../config/globalConfig.service");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const { generateTraceId } = require("../purchase/services/userPackage.service");
const {
	recordComprehensiveTransaction,
	recordReferralBonus,
	recordBonusClaim,
} = require("../purchase/services/transactionTracking.service");
const { getUserUpline } = require("../users/user.service");
const { WalletTransaction } = require("../wallets/walletTransaction.model");
const {
	calculateBonusPerUserNTE,
	calculateStakingBonusNTE,
	calculateFlexibleStakingBonusNTE,
	calculateDailyReferralBonusNTE,
	calculateTargetFillBonusNTE,
	formatNTEPrecision,
} = require("../../utils/productCalculations");
const UserPackage = require("../purchase/userPackage.model");

const getTodayDateString = () => {
	const today = new Date();
	return today.toISOString().split("T")[0];
};

// Check and update package expiration status
const checkAndUpdatePackageExpiration = async (userPackage) => {
	try {
		const currentDate = new Date();
		const endDate = new Date(userPackage.endDate);

		if (currentDate > endDate && userPackage.status === "active") {
			const product = await Product.findById(userPackage.packageId).lean();

			const finalStatus =
				product?.packageType === "locked" ? "completed" : "expired";

			const updatedPackage = await UserPackage.findByIdAndUpdate(
				userPackage._id,
				{
					status: finalStatus,
					updatedAt: new Date(),
					expiredAt: currentDate,
				},
				{ new: true },
			).lean();

			logger.info("Package marked as expired", {
				packageId: userPackage._id.toString(),
				username: userPackage.username,
				endDate: userPackage.endDate,
				oldStatus: userPackage.status,
				newStatus: finalStatus,
				packageType: product?.packageType,
			});

			return updatedPackage || userPackage;
		}

		return userPackage;
	} catch (error) {
		logger.error("Error checking package expiration:", error);
		return userPackage;
	}
};

// Bulk update for all expired packages
async function updateAllExpiredPackages() {
	try {
		logger.info(
			"PACKAGE EXPIRATION | Job started | Checking for expired packages",
		);

		const currentDate = new Date();

		// Find all active packages that have expired
		const expiredPackages = await UserPackage.find({
			status: "active",
			endDate: { $lt: currentDate },
		}).lean();

		logger.info(
			`PACKAGE EXPIRATION | Found ${expiredPackages.length} packages to expire | Processing updates`,
		);

		let updatedCount = 0;
		const results = [];

		for (const pkg of expiredPackages) {
			try {
				// Get product type to determine final status
				const product = await Product.findById(pkg.packageId).lean();
				const finalStatus =
					product?.packageType === "locked" ? "completed" : "expired";

				// Update package status
				await UserPackage.findByIdAndUpdate(pkg._id, {
					$set: {
						status: finalStatus,
						updatedAt: currentDate,
						expiredAt: currentDate,
					},
				});

				logger.info(
					`PACKAGE EXPIRATION | DB WRITE: Package ${
						pkg._id
					} updated from 'active' to '${finalStatus}' | User: ${
						pkg.username
					} | Package: ${product?.name || "Unknown"} | EndDate: ${pkg.endDate}`,
				);

				updatedCount++;
				results.push({
					packageId: pkg._id.toString(),
					username: pkg.username,
					oldStatus: pkg.status,
					newStatus: finalStatus,
					endDate: pkg.endDate,
				});
			} catch (error) {
				logger.error(
					`PACKAGE EXPIRATION | ERROR: Failed to update package ${pkg._id} | User: ${pkg.username} | Error: ${error.message}`,
				);
			}
		}

		logger.info(
			`PACKAGE EXPIRATION | Job completed: ${updatedCount}/$${
				expiredPackages.length
			} packages updated | ${expiredPackages.length - updatedCount} errors`,
		);

		return {
			success: true,
			updatedCount,
			totalExpired: expiredPackages.length,
			results,
		};
	} catch (error) {
		logger.error("PACKAGE EXPIRATION | Job failed:", error.message);
		return {
			success: false,
			error: error.message,
			updatedCount: 0,
		};
	}
}

const bonusService = {
	// Check if bonuses have already been distributed for a specific user and date
	async hasBonusesBeenDistributed(username, dateString, userPackageId = null) {
		try {
			// First, check if daily_summary exists
			const dailySummary = await BonusDistribution.findOne({
				username,
				distributionDate: dateString,
				bonusType: "daily_summary",
				status: "completed",
			}).lean();

			if (dailySummary) {
				logger.info(
					`Daily bonuses already distributed for user: ${username}, date: ${dateString}`,
				);
				return true;
			}

			// If checking specific package, verify it's not in the distribution summary
			if (userPackageId) {
				// Check for any bonus transaction (excluding failed ones) for this package today
				const packageCheck = await BonusTransaction.findOne({
					username,
					createdAt: {
						$gte: new Date(dateString + "T00:00:00.000Z"),
						$lt: new Date(dateString + "T23:59:59.999Z"),
					},
					"metadata.userPackageId": userPackageId.toString(),
					status: "completed",
					// Only check for package-based bonuses (not referral bonuses)
					bonusType: {
						$in: ["daily_bonus", "staking_bonus", "flexible_staking_bonus"],
					},
				}).lean();

				if (packageCheck) {
					logger.info(
						`Package ${userPackageId} already processed for user: ${username}, date: ${dateString}`,
					);
					return true;
				}
			}

			return false;
		} catch (error) {
			logger.error("Error checking bonus distribution status:", error);
			return false;
		}
	},
	async distributeMonthlyBonusToUser(username, force = false) {
		try {
			// Check if already distributed (always check to prevent duplicates)
			if (await this.hasMonthlyBonusBeenDistributed(username)) {
				logger.info(
					`MONTHLY BONUS DISTRIBUTION | SKIP: User ${username} already received monthly bonus`,
				);
				return {
					success: true,
					message: "Monthly bonus already distributed for this month",
					alreadyDistributed: true,
					monthlyBonusNTE: 0,
					monthlyBonusUSD: 0,
				};
			}

			// Get user's packages with monthly bonus (held for at least one month)
			const userPackagesWithMonthlyBonus = await UserPackage.aggregate([
				{
					$lookup: {
						from: "products",
						localField: "packageId",
						foreignField: "_id",
						as: "product",
					},
				},
				{ $unwind: "$product" },
				{
					$match: {
						username,
						status: "active",
						"product.hasMonthlyBonus": true,
						purchasedAt: {
							$lte: new Date(Date.now() - 30 * 24 * 60 * 60 * 1000),
						}, // At least 30 days old
					},
				},
			]);

			if (
				!userPackagesWithMonthlyBonus ||
				userPackagesWithMonthlyBonus.length === 0
			) {
				return {
					success: false,
					message: "No eligible packages for monthly bonus",
					monthlyBonusNTE: 0,
					monthlyBonusUSD: 0,
					eligiblePackages: [],
				};
			}

			// Get total tax earned and count of eligible users (packages held for at least one month)
			const totalTaxEarned = (await getTotalTaxEarned()) || 0;
			const totalEligibleUsers = await UserPackage.aggregate([
				{
					$lookup: {
						from: "products",
						localField: "packageId",
						foreignField: "_id",
						as: "product",
					},
				},
				{ $unwind: "$product" },
				{
					$match: {
						status: "active",
						"product.hasMonthlyBonus": true,
						purchasedAt: {
							$lte: new Date(Date.now() - 30 * 24 * 60 * 60 * 1000),
						}, // At least 30 days old
					},
				},
				{ $group: { _id: "$username" } },
				{ $count: "totalUsers" },
			]).then((result) => result[0]?.totalUsers || 1);

			const monthlyBonusNTE = totalTaxEarned / totalEligibleUsers;

			if (isNaN(monthlyBonusNTE) || monthlyBonusNTE <= 0) {
				return {
					success: false,
					message: "No monthly bonus calculated",
					monthlyBonusNTE: 0,
					monthlyBonusUSD: 0,
					eligiblePackages: [],
				};
			}

			// Get user details
			const user = await User.findOne({ username });
			if (!user) {
				return {
					success: false,
					message: "User not found",
					monthlyBonusNTE: 0,
					monthlyBonusUSD: 0,
				};
			}

			const currentMonth = new Date().toISOString().substring(0, 7); // YYYY-MM format
			const tokenValue = await getTokenValue();

			// Create monthly distribution record with package tracking
			const monthlyDistributionId = new mongoose.Types.ObjectId();

			try {
				const traceId = generateTraceId("monthly_bonus", username, new Date());

				// Distribute bonus proportionally among eligible packages
				const bonusPerPackage =
					monthlyBonusNTE / userPackagesWithMonthlyBonus.length;
				let totalDistributed = 0;

				for (const eligiblePackage of userPackagesWithMonthlyBonus) {
					const result = await this.recordBonusTransaction({
						username,
						userId: user._id,
						bonusType: "monthly_bonus",
						amountNTE: bonusPerPackage,
						amountUSD: bonusPerPackage * tokenValue,
						description: `Monthly tax bonus from ${eligiblePackage.product.name} package (${currentMonth})`,
						metadata: {
							userPackageId: eligiblePackage._id.toString(),
							packageId: eligiblePackage.packageId.toString(),
							packageName: eligiblePackage.product.name,
							packageType: eligiblePackage.product.packageType,
							monthKey: currentMonth,
							bonusOrigin: "monthly_tax_distribution",
							distributionId: monthlyDistributionId,
							traceId,
							calculationDetails: {
								formula: "Total Tax Earned / Number of Eligible Users",
								totalTaxEarned,
								totalEligibleUsers,
								bonusPerUser: monthlyBonusNTE,
								bonusPerPackage: bonusPerPackage,
							},
						},
						walletBalanceBefore: user.walletBalance || 0,
						distributionId: monthlyDistributionId,
						userPackageId: eligiblePackage._id.toString(),
						packageId: eligiblePackage.packageId,
						packageName: eligiblePackage.product.name,
						packageType: eligiblePackage.product.packageType,
						session: null,
					});

					if (result.success) {
						totalDistributed += bonusPerPackage;
					}
				}

				// Record main monthly distribution summary
				await this.logBonusDistribution({
					_id: monthlyDistributionId,
					username,
					distributionDate: getTodayDateString(),
					bonusType: "monthly_bonus",
					amountNTE: totalDistributed,
					amountUSD: totalDistributed * tokenValue,
					metadata: {
						traceId,
						monthKey: currentMonth,
						calculationMethod: "NTE-First",
						tokenValue: tokenValue,
						triggerType: force ? "manual" : "automatic",
						bonusOrigin: "monthly_tax_distribution",
						eligiblePackagesCount: userPackagesWithMonthlyBonus.length,
						calculation: {
							formula: "Total Tax Earned / Number of Eligible Users",
							totalTaxEarned,
							totalEligibleUsers,
							bonusPerUser: monthlyBonusNTE,
						},
						eligiblePackages: userPackagesWithMonthlyBonus.map((pkg) => ({
							userPackageId: pkg._id.toString(),
							packageId: pkg.packageId.toString(),
							packageName: pkg.product.name,
							packageType: pkg.product.packageType,
						})),
					},
				});

				logger.info(
					`MONTHLY BONUS DISTRIBUTION | DB WRITE: User ${username} monthly bonus distributed | Bonus: ${
						formatNTEPrecision(totalDistributed).display
					} | Packages: ${
						userPackagesWithMonthlyBonus.length
					} | Month: ${currentMonth}`,
				);

				return {
					success: true,
					username,
					monthKey: currentMonth,
					monthlyBonusNTE: totalDistributed,
					monthlyBonusUSD: totalDistributed * tokenValue,
					eligiblePackagesCount: userPackagesWithMonthlyBonus.length,
					eligiblePackages: userPackagesWithMonthlyBonus.map((pkg) => ({
						userPackageId: pkg._id.toString(),
						packageId: pkg.packageId.toString(),
						packageName: pkg.product.name,
						packageType: pkg.product.packageType,
					})),
					distributionId: monthlyDistributionId.toString(),
					message: `Monthly bonus distributed successfully across ${userPackagesWithMonthlyBonus.length} eligible packages`,
				};
			} catch (error) {
				logger.error(`Error distributing monthly bonus to ${username}:`, error);
				return {
					success: false,
					message: error.message || "Error distributing monthly bonus",
					error: error.message,
					monthlyBonusNTE: 0,
					monthlyBonusUSD: 0,
				};
			}
		} catch (error) {
			logger.error(
				`Error in distributeMonthlyBonusToUser for ${username}:`,
				error,
			);
			return {
				success: false,
				message: "Error processing monthly bonus distribution",
				error: error.message,
				monthlyBonusNTE: 0,
				monthlyBonusUSD: 0,
			};
		}
	},

	async logBonusDistribution(distributionData) {
		try {
			const traceId = generateTraceId(
				distributionData.bonusType || "bonus_distribution",
				distributionData.username,
			);

			if (distributionData.bonusType === "monthly_bonus") {
				// Use find and update/create to prevent duplicates for monthly bonuses
				const monthKey = distributionData.metadata.monthKey;
				const filter = {
					username: distributionData.username,
					bonusType: "monthly_bonus",
					"metadata.monthKey": monthKey,
				};
				const existing = await BonusDistribution.findOne(filter).lean();
				if (existing) {
					const { _id, ...dataToUpdate } = distributionData;
					await BonusDistribution.updateOne(filter, {
						$set: {
							...dataToUpdate,
							loggedAt: new Date(),
							status: "completed",
							metadata: {
								...distributionData.metadata,
								traceId: traceId,
							},
						},
					});
					return await BonusDistribution.findOne(filter).lean();
				} else {
					const doc = new BonusDistribution({
						...distributionData,
						loggedAt: new Date(),
						status: "completed",
						_id: new mongoose.Types.ObjectId(),
						metadata: {
							...distributionData.metadata,
							traceId: traceId,
						},
					});
					await doc.save();
					return doc.toObject();
				}
			} else {
				const { _id, ...dataToInsert } = distributionData;
				const bonusDistribution = new BonusDistribution({
					...dataToInsert,
					loggedAt: new Date(),
					status: "completed",
					_id: new mongoose.Types.ObjectId(),
					metadata: {
						...distributionData.metadata,
						traceId: traceId,
					},
				});

				await bonusDistribution.save();

				return bonusDistribution.toObject();
			}
		} catch (error) {
			logger.error("Error logging bonus distribution:", error);
			throw error;
		}
	},

	async recordBonusTransaction(options) {
		try {
			const {
				username,
				userId,
				bonusType,
				amountNTE,
				amountUSD,
				description,
				metadata = {},
				walletBalanceBefore,
				distributionId,
				userPackageId,
				packageId,
				packageName,
				packageType,
				session,
			} = options;

			const traceId = generateTraceId(bonusType, username);
			const metadataUpdated = {
				triggerType: "automatic",
				...metadata,
				userPackageId: userPackageId ? userPackageId.toString() : null,
				packageId: packageId ? packageId.toString() : null,
				packageName: packageName || null,
				packageType: packageType || null,
				bonusOrigin: metadata.bonusOrigin || "package_based",
			};
			return await recordComprehensiveTransaction({
				username,
				userId,
				transactionType: "bonus_payment",
				bonusType,
				amountNTE,
				amountUSD,
				description,
				metadata: metadataUpdated,
				walletBalanceBefore,
				isDeduction: false,
				skipWalletUpdate: false,
				traceId,
				session,
			});
		} catch (error) {
			logger.error("Error recording bonus transaction:", {
				username: options.username,
				bonusType: options.bonusType,
				error: error.message,
			});
			throw error;
		}
	},

	async distributeBonuses(username) {
		try {
			const dateString = getTodayDateString();
			logger.info("Starting daily bonus distribution", {
				username,
				dateString,
			});

			// Make sure the cache is valid, update if not
			if (!(await isUsersWithActivePackagesCacheValid())) {
				logger.warn(
					`USER BONUS DISTRIBUTION | Cache invalid for user ${username}, updating quickly...`,
				);

				try {
					await updateUsersWithActivePackagesCache();
					logger.info(
						`USER BONUS DISTRIBUTION | Cache updated for user ${username}`,
					);
				} catch (cacheUpdateError) {
					logger.warn(
						`USER BONUS DISTRIBUTION | Cache update failed for user ${username}, proceeding:`,
						cacheUpdateError.message,
					);
				}
			}

			// If daily summary bonus already distributed, skip
			if (await this.hasBonusesBeenDistributed(username, dateString)) {
				logger.info("Bonuses already distributed today", {
					username,
					dateString,
				});
				return {
					success: false,
					message: "Bonuses already distributed today",
					alreadyDistributed: true,
				};
			}

			// Get all active packages for this user
			const userPackagesList = await UserPackage.find({
				username,
				status: "active",
			}).lean();

			if (!userPackagesList || userPackagesList.length === 0) {
				logger.info("No active packages for user", { username });
				return {
					success: false,
					message: "No active packages found",
				};
			}

			// Only keep packages that have started and are not expired
			const activePackages = [];

			for (const pkg of userPackagesList) {
				const updatedPkg = await checkAndUpdatePackageExpiration(pkg);
				if (updatedPkg.status === "active") {
					// Check if package period has started
					const currentDate = new Date();
					const startDate = new Date(updatedPkg.startDate);

					if (currentDate >= startDate) {
						activePackages.push(updatedPkg);
					}
				}
			}

			if (activePackages.length === 0) {
				logger.info("No active packages in distribution period found", {
					username,
				});
				return {
					success: false,
					message: "No active packages in distribution period found",
				};
			}

			// Get upline referrers for referral bonuses
			const uplineReferrers = await getUserUpline(username);

			// Get the token value for calculations
			const tokenValue = await getTokenValue();

			let totalBonusesDistributed = 0;
			let totalReferralBonusesDistributed = 0;
			let totalTransactions = 0;
			const distributionSummary = [];

			// Process each package for bonus distribution
			for (const userPackage of activePackages) {
				try {
					// Get the product details
					const product = await Product.findById(userPackage.packageId).lean();
					if (!product) {
						logger.warn(`Product not found for package ${userPackage._id}`);
						continue;
					}

					logger.info(
						`PACKAGE BONUS | Processing package ${product.name} for user ${username} | Type: ${product.packageType}`,
						{
							username,
							packageId: userPackage._id,
							packageName: product.name,
							packageType: product.packageType,
						},
					);

					// Check if this specific package already received bonuses today
					if (
						await this.hasBonusesBeenDistributed(
							username,
							dateString,
							userPackage._id.toString(),
						)
					) {
						logger.info(
							`PACKAGE BONUS | Skipping package ${product.name} - already distributed today`,
							{
								username,
								packageId: userPackage._id,
								packageName: product.name,
							},
						);
						continue;
					}

					// Create package-specific distribution ID
					const packageDistributionId = new mongoose.Types.ObjectId();

					// Use the new package-based bonus distribution function
					const packageResult = await this.distributePackageBasedBonuses(
						username,
						userPackage,
						product,
						uplineReferrers,
						packageDistributionId,
					);

					if (packageResult.success) {
						totalBonusesDistributed += packageResult.totalBonusNTE || 0;
						totalReferralBonusesDistributed +=
							packageResult.referralBonusesTotalNTE || 0;
						totalTransactions += packageResult.transactionsProcessed || 0;

						// Add only bonusTransaction IDs to distribution summary
						if (
							Array.isArray(packageResult.bonusTransactionIds) &&
							packageResult.bonusTransactionIds.length > 0
						) {
							distributionSummary.push(...packageResult.bonusTransactionIds);
						}

						logger.info(
							`PACKAGE BONUS | Distributed: User ${username} received ${packageResult.totalBonusNTE} from ${product.name}`,
							{
								username,
								packageId: userPackage._id,
								packageName: product.name,
								bonusNTE: packageResult.totalBonusNTE,
								bonusesCount: packageResult.bonusesCount,
								transactionsCount: packageResult.transactionsProcessed,
							},
						);
					} else {
						logger.error(
							`Failed to process package ${userPackage._id}:`,
							packageResult.error,
						);
					}
				} catch (packageError) {
					logger.error(`Error processing package ${userPackage._id}:`, {
						message: packageError?.message,
						stack: packageError?.stack,
						error: packageError,
					});
				}
			}

			// Log final summary and get distribution ID
			const traceId = generateTraceId("daily_summary", username);
			const combinedTotalNTE =
				totalBonusesDistributed + totalReferralBonusesDistributed;
			const distributionRecord = await this.logBonusDistribution({
				username,
				distributionDate: dateString,
				bonusType: "daily_summary",
				amountNTE: combinedTotalNTE,
				amountUSD: combinedTotalNTE * tokenValue,
				metadata: {
					traceId,
					totalTransactions,
					userBonusesNTE: totalBonusesDistributed,
					referralBonusesNTE: totalReferralBonusesDistributed,
					packagesProcessed: distributionSummary.length,
					referrersBenefited: uplineReferrers.length,
					distributionSummary,
					calculationMethod: "NTE-First",
					tokenValue,
					triggerType: "automatic",
					relatedTransactions: totalTransactions > 0 ? true : false,
				},
			});

			logger.info("Daily bonus distribution completed", {
				username,
				totalBonusesDistributed,
				totalTransactions,
				packagesProcessed: distributionSummary.length,
				referrersBenefited: uplineReferrers.length,
			});

			// Store distribution ID in the result for reference
			return {
				success: true,
				data: {
					username,
					distributionDate: dateString,
					distributionId: distributionRecord._id?.toString(),
					totalBonusesDistributed: combinedTotalNTE,
					userBonusesDistributed: totalBonusesDistributed,
					referralBonusesDistributed: totalReferralBonusesDistributed,
					totalTransactions,
					packagesProcessed: distributionSummary.length,
					referrersBenefited: uplineReferrers.length,
					distributionSummary,
				},
				message: "Bonus distribution completed successfully",
			};
		} catch (error) {
			logger.error(`Error in bonus distribution for ${username}:`, error);

			// Log the error
			try {
				const traceId = generateTraceId("bonus_distribution_error", username);
				await this.logBonusDistribution({
					username,
					distributionDate: getTodayDateString(),
					bonusType: "error",
					amountNTE: 0,
					amountUSD: 0,
					metadata: {
						error: error.message,
						stack: error.stack,
						calculationMethod: "NTE-First",
						traceId,
					},
				});
			} catch (logError) {
				logger.error("Failed to log error:", logError);
			}

			return {
				success: false,
				message: "Critical error in bonus distribution",
				error: error.message,
			};
		}
	},

	async distributeBonusesToAllUsers() {
		try {
			const dateString = getTodayDateString();

			logger.info(
				"DAILY BONUS DISTRIBUTION | Starting daily bonus distribution for all users",
				{
					dateString,
				},
			);

			// LOCK CHECK: Prevent multiple simultaneous distributions
			const lockKey = `daily_distribution_lock_${dateString}`;
			const existingLock = await BonusDistribution.findOne({
				username: "SYSTEM",
				distributionDate: dateString,
				bonusType: "daily_system_summary",
				status: { $in: ["in_progress", "completed"] },
			}).lean();

			if (existingLock) {
				if (existingLock.status === "in_progress") {
					// Check if lock is stale (older than 2 hours)
					const lockAge =
						Date.now() - new Date(existingLock.loggedAt).getTime();
					if (lockAge < 2 * 60 * 60 * 1000) {
						logger.warn(
							"DAILY BONUS DISTRIBUTION | Another distribution is already in progress",
							{ lockAge: Math.round(lockAge / 1000) + "s" },
						);
						return {
							success: false,
							message: "Distribution already in progress",
							alreadyDistributed: true,
						};
					}
					logger.warn(
						"DAILY BONUS DISTRIBUTION | Stale lock detected, proceeding",
					);
				} else {
					logger.info(
						"DAILY BONUS DISTRIBUTION | Distribution already completed for today",
					);
					return {
						success: true,
						message: "Distribution already completed",
						alreadyDistributed: true,
					};
				}
			}

			// Create lock record with in_progress status
			const lockId = new mongoose.Types.ObjectId();
			await BonusDistribution.create({
				_id: lockId,
				username: "SYSTEM",
				distributionDate: dateString,
				bonusType: "daily_system_summary",
				amountNTE: 0,
				amountUSD: 0,
				status: "in_progress",
				loggedAt: new Date(),
				metadata: {
					lockKey,
					startedAt: new Date(),
					triggerType: "automatic",
					traceId: generateTraceId("daily_system_lock", "SYSTEM"),
				},
			});

			logger.info(
				"DAILY BONUS DISTRIBUTION | Lock acquired, proceeding with distribution",
			);

			if (!(await isUsersWithActivePackagesCacheValid())) {
				logger.info(
					"DAILY BONUS DISTRIBUTION | Cache is invalid or expired, updating before processing...",
				);

				try {
					await updateUsersWithActivePackagesCache();
					logger.info(
						"DAILY BONUS DISTRIBUTION | Cache updated successfully, proceeding with distribution",
					);
				} catch (cacheUpdateError) {
					logger.error(
						"DAILY BONUS DISTRIBUTION | Failed to update cache, proceeding anyway:",
						cacheUpdateError.message,
					);
				}
			} else {
				logger.info(
					"DAILY BONUS DISTRIBUTION | Cache is valid, proceeding with distribution",
				);
			}

			const usersWithActivePackages = await UserPackage.distinct("username", {
				status: "active",
			});

			const results = {
				totalUsers: usersWithActivePackages.length,
				successful: 0,
				failed: 0,
				alreadyDistributed: 0,
				totalBonusesDistributed: 0,
				totalTransactions: 0,
				errors: [],
				startTime: new Date(),
				endTime: null,
			};

			logger.info(
				`DAILY BONUS DISTRIBUTION | Processing ${usersWithActivePackages.length} users with active packages`,
			);

			for (let i = 0; i < usersWithActivePackages.length; i++) {
				const username = usersWithActivePackages[i];

				try {
					const userResult = await this.distributeBonuses(username);

					if (userResult.success) {
						results.successful++;
						results.totalBonusesDistributed +=
							userResult.data?.totalBonusesDistributed || 0;
						results.totalTransactions +=
							userResult.data?.totalTransactions || 0;
					} else if (userResult.alreadyDistributed) {
						results.alreadyDistributed++;
					} else {
						results.failed++;
						results.errors.push({
							username,
							error: userResult.message || userResult.error,
						});
					}
				} catch (userError) {
					logger.error(`Error processing user ${username}:`, userError);
					results.failed++;
					results.errors.push({
						username,
						error: userError.message,
					});
				}
			}

			results.endTime = new Date();
			const duration = (results.endTime - results.startTime) / 1000;
			const tokenValue = await getTokenValue();

			// VERIFICATION STEP: Check for missed packages
			logger.info(
				"DAILY BONUS DISTRIBUTION | Running verification for missed packages",
				{ dateString },
			);
			const missedPackagesResults = await this.verifyBonusDistribution(
				dateString,
			);

			if (missedPackagesResults.missedPackagesProcessed > 0) {
				logger.info(
					"DAILY BONUS DISTRIBUTION | Verification recovered missed packages",
					{
						recovered: missedPackagesResults.missedPackagesProcessed,
						usersAffected: missedPackagesResults.usersAffected,
					},
				);

				results.totalBonusesDistributed +=
					missedPackagesResults.additionalBonusesNTE || 0;
				results.totalTransactions +=
					missedPackagesResults.additionalTransactions || 0;
				results.recovered = {
					missedPackagesProcessed:
						missedPackagesResults.missedPackagesProcessed,
					usersAffected: missedPackagesResults.usersAffected,
					additionalBonusesNTE: missedPackagesResults.additionalBonusesNTE || 0,
					additionalTransactions:
						missedPackagesResults.additionalTransactions || 0,
				};
			} else {
				logger.info(
					"DAILY BONUS DISTRIBUTION | Verification complete: No missed packages found",
				);
			}

			// Update lock record to completed status with final totals
			const traceId = generateTraceId("daily_system_summary", "SYSTEM");
			await BonusDistribution.updateOne(
				{
					username: "SYSTEM",
					distributionDate: dateString,
					bonusType: "daily_system_summary",
					status: "in_progress",
				},
				{
					$set: {
						status: "completed",
						amountNTE: results.totalBonusesDistributed,
						amountUSD: results.totalBonusesDistributed * tokenValue,
						metadata: {
							...results,
							calculationMethod: "NTE-First",
							tokenValue,
							distributionDuration: duration,
							triggerType: "automatic",
							traceId,
							verificationRun: true,
							completedAt: new Date(),
						},
					},
				},
			);

			logger.info("Daily bonus distribution completed", results);

			return results;
		} catch (error) {
			logger.error(
				"DAILY BONUS DISTRIBUTION | Error in daily bonus distribution:",
				error.message,
			);

			// Release lock on error by updating status to failed
			try {
				const dateString = getTodayDateString();
				await BonusDistribution.updateOne(
					{
						username: "SYSTEM",
						distributionDate: dateString,
						bonusType: "daily_system_summary",
						status: "in_progress",
					},
					{
						$set: {
							status: "failed",
							metadata: {
								error: error.message,
								failedAt: new Date(),
								triggerType: "automatic",
							},
						},
					},
				);
			} catch (lockError) {
				logger.error(
					"DAILY BONUS DISTRIBUTION | Failed to release lock:",
					lockError.message,
				);
			}

			return {
				success: false,
				error: error.message,
				totalUsers: 0,
				successful: 0,
				failed: 0,
				alreadyDistributed: 0,
			};
		}
	},

	async getBonusDistributionHistory(username, limit = 10, page = 1) {
		try {
			const skip = (page - 1) * limit;

			const [distributions, total] = await Promise.all([
				BonusDistribution.find({
					$or: [
						{ username },
						{ "metadata.referrerUsername": username },
						{ "metadata.purchaserUsername": username },
					],
				})
					.sort({ loggedAt: -1 })
					.skip(skip)
					.limit(limit)
					.lean(),
				BonusDistribution.countDocuments({
					$or: [
						{ username },
						{ "metadata.referrerUsername": username },
						{ "metadata.purchaserUsername": username },
					],
				}),
			]);

			return {
				success: true,
				data: {
					distributions,
					pagination: {
						total,
						page,
						limit,
						totalPages: Math.ceil(total / limit),
						hasNext: page < Math.ceil(total / limit),
						hasPrev: page > 1,
					},
				},
			};
		} catch (error) {
			logger.error("Error getting bonus distribution history:", error);
			throw new ApiError(500, "Failed to retrieve bonus distribution history");
		}
	},

	async getDetailedBonusTransactions(filters = {}) {
		try {
			const {
				username,
				bonusType,
				startDate,
				endDate,
				minAmount,
				maxAmount,
				page = 1,
				limit = 20,
				sortBy = "processedAt",
				sortOrder = "desc",
			} = filters;

			const query = {};

			if (username) {
				query.username = username;
			}

			if (bonusType) {
				query.bonusType = bonusType;
			}

			// Date range filter
			if (startDate || endDate) {
				query.processedAt = {};
				if (startDate) {
					query.processedAt.$gte = new Date(startDate);
				}
				if (endDate) {
					const end = new Date(endDate);
					end.setHours(23, 59, 59, 999);
					query.processedAt.$lte = end;
				}
			}

			// Amount range filter
			if (minAmount || maxAmount) {
				query.amountNTE = {};
				if (minAmount) {
					query.amountNTE.$gte = parseFloat(minAmount);
				}
				if (maxAmount) {
					query.amountNTE.$lte = parseFloat(maxAmount);
				}
			}

			const sort = {};
			sort[sortBy] = sortOrder === "asc" ? 1 : -1;

			const skip = (page - 1) * limit;

			const [transactions, total] = await Promise.all([
				BonusTransaction.find(query).sort(sort).skip(skip).limit(limit).lean(),
				BonusTransaction.countDocuments(query),
			]);

			return {
				success: true,
				data: {
					transactions,
					pagination: {
						total,
						page,
						limit,
						totalPages: Math.ceil(total / limit),
						hasNext: page < Math.ceil(total / limit),
						hasPrev: page > 1,
					},
				},
			};
		} catch (error) {
			logger.error("Error fetching detailed bonus transactions:", error);
			throw new ApiError(500, "Failed to retrieve bonus transactions");
		}
	},

	async getTransactionDetails(transactionId) {
		try {
			let transaction = await BonusTransaction.findById(transactionId).lean();

			let source = "bonusTransactions";

			// If not found in bonus transactions, check wallet transactions
			if (!transaction) {
				transaction = await WalletTransaction.findById(transactionId).lean();
				source = "walletTransactions";
			}

			if (!transaction) {
				throw new ApiError(404, "Transaction not found");
			}

			// Get transaction context
			const context = await this.getTransactionContext(transaction, source);

			return {
				success: true,
				data: {
					...transaction,
					source,
					context,
				},
				message: "Transaction details retrieved successfully",
			};
		} catch (error) {
			logger.error("Error getting transaction details:", error);
			throw error;
		}
	},

	async getTransactionContext(transaction, source) {
		try {
			const context = {
				relatedTransactions: [],
				productDetails: null,
				userDetails: null,
			};

			// Get product details if packageId exists
			if (transaction.packageId) {
				try {
					const packageId = transaction.packageId;
					const product = await Product.findById(packageId).lean();

					if (product) {
						context.productDetails = {
							name: product.name,
							packageType: product.packageType,
							price: product.price,
						};
					}
				} catch (productError) {
					logger.warn("Error getting product details:", productError);
				}
			}

			// Get basic user details
			if (transaction.username) {
				try {
					const user = await User.findOne(
						{ username: transaction.username },
						{ fullName: 1, email: 1, walletBalance: 1 },
					).lean();

					if (user) {
						context.userDetails = {
							username: transaction.username,
							fullName: user.fullName,
							email: user.email,
							currentWalletBalance: user.walletBalance,
						};
					}
				} catch (userError) {
					logger.warn("Error getting user details:", userError);
				}
			}

			// Get related transactions if it's a referral bonus
			if (
				transaction.bonusType?.includes("referral") &&
				transaction.purchaserUsername
			) {
				try {
					const relatedTxs = await BonusTransaction.find({
						"metadata.purchaserUsername":
							transaction["metadata.purchaserUsername"] ||
							transaction.purchaserUsername,
						bonusType: transaction.bonusType,
						distributionDate:
							transaction.distributionDate ||
							transaction.referenceDate ||
							transaction["metadata.distributionDate"],
					})
						.limit(5)
						.lean();

					context.relatedTransactions = relatedTxs.map((tx) => ({
						_id: tx._id.toString(),
						username: tx.username,
						amountNTE: tx.amountNTE,
						bonusType: tx.bonusType,
						createdAt: tx.createdAt,
					}));
				} catch (relatedError) {
					logger.warn("Error getting related transactions:", relatedError);
				}
			}

			return context;
		} catch (error) {
			logger.error("Error getting transaction context:", error);
			return {};
		}
	},

	async hasMonthlyBonusBeenDistributed(username) {
		try {
			const now = new Date();
			const monthKey = `${now.getFullYear()}-${String(
				now.getMonth() + 1,
			).padStart(2, "0")}`;

			const existingDistribution = await BonusDistribution.findOne({
				username,
				bonusType: "monthly_bonus",
				"metadata.monthKey": monthKey,
				status: "completed",
			}).lean();

			return !!existingDistribution;
		} catch (error) {
			logger.error("Error checking monthly bonus distribution status:", error);
			return false;
		}
	},

	async calculateMonthlyBonus(username) {
		try {
			const dateString = getTodayDateString();
			const now = new Date();
			const monthKey = `${now.getFullYear()}-${String(
				now.getMonth() + 1,
			).padStart(2, "0")}`;

			logger.info(
				"MONTHLY BONUS DISTRIBUTION | Job started | Processing all eligible users",
				{ monthKey },
			);

			// Get all users who have active packages with monthly bonus
			const usersWithMonthlyBonusPackages = await UserPackage.aggregate([
				{
					$lookup: {
						from: "products",
						localField: "packageId",
						foreignField: "_id",
						as: "product",
					},
				},
				{ $unwind: "$product" },
				{ $match: { status: "active", "product.hasMonthlyBonus": true } },
				{ $group: { _id: "$username", packages: { $push: "$$ROOT" } } },
			]);

			if (
				!usersWithMonthlyBonusPackages ||
				usersWithMonthlyBonusPackages.length === 0
			) {
				logger.info("MONTHLY BONUS DISTRIBUTION | No eligible users found");
				return {
					success: true,
					totalEligible: 0,
					successful: 0,
					failed: 0,
					alreadyDistributed: 0,
					totalBonusesDistributed: 0,
					message: "No users eligible for monthly bonus",
				};
			}

			logger.info(
				`MONTHLY BONUS DISTRIBUTION | Processing ${usersWithMonthlyBonusPackages.length} eligible users`,
			);

			const results = {
				totalEligible: usersWithMonthlyBonusPackages.length,
				successful: 0,
				failed: 0,
				alreadyDistributed: 0,
				totalBonusesDistributed: 0,
				errors: [],
				distributions: [],
			};

			for (const userGroup of usersWithMonthlyBonusPackages) {
				const username = userGroup._id;

				try {
					// Check if already distributed
					if (!force && (await this.hasMonthlyBonusBeenDistributed(username))) {
						results.alreadyDistributed++;
						results.distributions.push({
							username,
							status: "already_distributed",
							reason: "Already distributed for current month",
						});
						continue;
					}

					// Get user details
					const user = await User.findOne({ username }).lean();
					if (!user) {
						results.failed++;
						results.errors.push({
							username,
							error: "User not found",
							timestamp: new Date(),
						});
						continue;
					}

					// Distribute monthly bonus to the user
					const monthlyDistributionResult =
						await this.distributeMonthlyBonusToUser(username, force);

					if (monthlyDistributionResult.success) {
						results.successful++;
						results.totalBonusesDistributed +=
							monthlyDistributionResult.monthlyBonusNTE || 0;
						results.distributions.push({
							username,
							bonusUSD: monthlyDistributionResult.monthlyBonusUSD,
							bonusNTE: monthlyDistributionResult.monthlyBonusNTE,
							packages: monthlyDistributionResult.eligiblePackagesCount,
							status: "success",
							distributionId: monthlyDistributionResult.distributionId,
						});
						logger.info(
							`Distributed monthly bonus ${
								monthlyDistributionResult.monthlyBonusUSD || 0
							} USD to ${username} across ${
								monthlyDistributionResult.eligiblePackagesCount || 0
							} packages`,
						);
					} else {
						results.failed++;
						results.errors.push({
							username,
							error: monthlyDistributionResult.message || "Distribution failed",
							timestamp: new Date(),
						});
						results.distributions.push({
							username,
							status: "error",
							reason: monthlyDistributionResult.message,
						});
					}
				} catch (error) {
					results.failed++;
					results.errors.push({
						username,
						error: error.message,
						timestamp: new Date(),
					});
					results.distributions.push({
						username,
						status: "error",
						reason: error.message,
					});
					logger.error(
						`❌ Error distributing monthly bonus to ${username}:`,
						error.message,
					);
				}
			}

			const tokenValue = await getTokenValue();
			const traceId = generateTraceId("monthly_system_summary", "SYSTEM");

			// Log system summary
			await this.logBonusDistribution({
				username: "SYSTEM",
				distributionDate: dateString,
				bonusType: "monthly_system_summary",
				amountNTE: results.totalBonusesDistributed,
				amountUSD: results.totalBonusesDistributed * tokenValue,
				metadata: {
					monthKey,
					...results,
					tokenValue,
					distributionType: "monthly_bonus",
					triggerType: force ? "manual" : "automatic",
					traceId,
				},
			});

			logger.info(
				`MONTHLY BONUS DISTRIBUTION | Job completed: ${results.successful}/${
					results.totalEligible
				} users processed | ${results.totalBonusesDistributed.toFixed(
					4,
				)} NTE distributed | ${results.failed} errors | Month: ${monthKey}`,
			);

			return {
				success: true,
				...results,
				message: `Distributed monthly bonuses to ${results.successful} out of ${results.totalEligible} eligible users`,
			};
		} catch (error) {
			logger.error("Error distributing monthly bonuses to all users:", error);
			return {
				success: false,
				error: error.message,
				totalEligible: 0,
				successful: 0,
				failed: 0,
			};
		}
	},

	async distributeMonthlyBonusesToAllUsers(force = false) {
		try {
			const now = new Date();
			const monthKey = `${now.getFullYear()}-${String(
				now.getMonth() + 1,
			).padStart(2, "0")}`;
			const monthName = now.toLocaleString("default", {
				month: "long",
				year: "numeric",
			});

			logger.info(
				" MONTHLY BONUS DISTRIBUTION | Starting monthly bonus distribution for all users",
				{
					monthKey,
				},
			);

			// LOCK CHECK: Prevent multiple simultaneous distributions
			const lockKey = `monthly_distribution_lock_${monthKey}`;
			let lockId;
			const existingLock = await BonusDistribution.findOne({
				username: "SYSTEM",
				bonusType: "monthly_system_summary",
				"metadata.monthKey": monthKey,
				status: { $in: ["in_progress", "completed"] },
			}).lean();

			if (existingLock) {
				if (existingLock.status === "in_progress") {
					// Check if lock is stale (older than 2 hours)
					const lockAge =
						Date.now() - new Date(existingLock.loggedAt).getTime();
					if (lockAge < 2 * 60 * 60 * 1000) {
						logger.warn(
							"MONTHLY BONUS DISTRIBUTION | Another distribution is already in progress",
							{ lockAge: Math.round(lockAge / 1000) + "s" },
						);
						return {
							success: false,
							message: "Distribution already in progress",
							alreadyDistributed: true,
						};
					}
					logger.warn(
						"MONTHLY BONUS DISTRIBUTION | Stale lock detected, updating existing lock",
					);
					// Update the existing stale lock to in_progress
					await BonusDistribution.updateOne(
						{ _id: existingLock._id },
						{
							$set: {
								status: "in_progress",
								loggedAt: new Date(),
								metadata: {
									...existingLock.metadata,
									startedAt: new Date(),
									triggerType: force ? "manual" : "automatic",
									traceId: generateTraceId("monthly_system_lock", "SYSTEM"),
								},
							},
						},
					);
					lockId = existingLock._id;
				} else {
					// If status is 'completed', do not return early. Just reuse the lockId and proceed.
					logger.info(
						"MONTHLY BONUS DISTRIBUTION | Previous distribution completed for this month, will scan for new eligible users and update summary.",
					);
					lockId = existingLock._id;
				}
			} else {
				// Create new lock record with in_progress status
				lockId = new mongoose.Types.ObjectId();
				await BonusDistribution.create({
					_id: lockId,
					username: "SYSTEM",
					distributionDate: now.toISOString().split("T")[0],
					bonusType: "monthly_system_summary",
					amountNTE: 0,
					amountUSD: 0,
					status: "in_progress",
					loggedAt: new Date(),
					metadata: {
						lockKey,
						monthKey,
						startedAt: new Date(),
						triggerType: force ? "manual" : "automatic",
						traceId: generateTraceId("monthly_system_lock", "SYSTEM"),
					},
				});
			}

			logger.info(
				"MONTHLY BONUS DISTRIBUTION | Lock acquired, proceeding with distribution",
			);

			// Check cache validity (though monthly might not need it as much, but for consistency)
			if (!(await isUsersWithActivePackagesCacheValid())) {
				logger.info(
					"MONTHLY BONUS DISTRIBUTION | Cache is invalid or expired, updating before processing...",
				);

				try {
					await updateUsersWithActivePackagesCache();
					logger.info(
						"MONTHLY BONUS DISTRIBUTION | Cache updated successfully, proceeding with distribution",
					);
				} catch (cacheUpdateError) {
					logger.error(
						"MONTHLY BONUS DISTRIBUTION | Failed to update cache, proceeding anyway:",
						cacheUpdateError.message,
					);
				}
			} else {
				logger.info(
					"MONTHLY BONUS DISTRIBUTION | Cache is valid, proceeding with distribution",
				);
			}

			// Get all users with active packages that have monthlyBonus enabled (held for at least one month)
			const usersWithMonthlyBonusPackages = await UserPackage.aggregate([
				{
					$lookup: {
						from: "products",
						localField: "packageId",
						foreignField: "_id",
						as: "product",
					},
				},
				{
					$unwind: "$product",
				},
				{
					$match: {
						status: "active",
						"product.hasMonthlyBonus": true,
						purchasedAt: {
							$lte: new Date(Date.now() - 30 * 24 * 60 * 60 * 1000),
						}, // At least 30 days old
					},
				},
				{
					$group: {
						_id: "$username",
						packages: { $push: "$$ROOT" },
					},
				},
			]);

			if (usersWithMonthlyBonusPackages.length === 0) {
				logger.info(
					"MONTHLY BONUS DISTRIBUTION | No users eligible for monthly bonus",
				);
				return {
					success: true,
					totalEligible: 0,
					successful: 0,
					failed: 0,
					alreadyDistributed: 0,
					totalBonusesDistributed: 0,
					message: "No users eligible for monthly bonus",
				};
			}

			logger.info(
				`MONTHLY BONUS DISTRIBUTION | Found ${usersWithMonthlyBonusPackages.length} users eligible for monthly bonus`,
			);

			const results = {
				totalUsers: usersWithMonthlyBonusPackages.length,
				successful: 0,
				failed: 0,
				alreadyDistributed: 0,
				totalBonusesDistributed: 0,
				errors: [],
				distributions: [],
				startTime: new Date(),
				endTime: null,
			};

			logger.info(
				`Processing ${usersWithMonthlyBonusPackages.length} users for monthly bonus`,
			);

			for (const userGroup of usersWithMonthlyBonusPackages) {
				const username = userGroup._id;

				try {
					// Check if already distributed (always check to prevent duplicates)
					if (await this.hasMonthlyBonusBeenDistributed(username)) {
						results.alreadyDistributed++;
						results.distributions.push({
							username,
							status: "already_distributed",
							reason: "Already distributed for current month",
						});
						continue;
					}

					// Get user details
					const user = await User.findOne({ username }).lean();
					if (!user) {
						results.failed++;
						results.errors.push({
							username,
							error: "User not found",
							timestamp: new Date(),
						});
						continue;
					}

					// Use the enhanced monthly bonus distribution function
					const monthlyDistributionResult =
						await this.distributeMonthlyBonusToUser(username, force);

					if (monthlyDistributionResult.success) {
						results.successful++;
						results.totalBonusesDistributed +=
							monthlyDistributionResult.monthlyBonusNTE;
						results.distributions.push({
							username,
							bonusUSD: monthlyDistributionResult.monthlyBonusUSD,
							bonusNTE: monthlyDistributionResult.monthlyBonusNTE,
							packages: monthlyDistributionResult.eligiblePackagesCount,
							status: "success",
							distributionId: monthlyDistributionResult.distributionId,
						});

						logger.info(
							`Distributed monthly bonus ${monthlyDistributionResult.monthlyBonusUSD.toFixed(
								6,
							)} USD to ${username} across ${
								monthlyDistributionResult.eligiblePackagesCount
							} packages`,
						);
					} else {
						results.failed++;
						results.errors.push({
							username,
							error: monthlyDistributionResult.message || "Distribution failed",
							timestamp: new Date(),
						});
						results.distributions.push({
							username,
							status: "error",
							reason: monthlyDistributionResult.message,
						});
					}
				} catch (error) {
					results.failed++;
					results.errors.push({
						username,
						error: error.message,
						timestamp: new Date(),
					});
					results.distributions.push({
						username,
						status: "error",
						reason: error.message,
					});
					logger.error(
						`Error distributing monthly bonus to ${username}:`,
						error.message,
					);
				}
			}

			results.endTime = new Date();
			const duration = (results.endTime - results.startTime) / 1000;
			const tokenValue = await getTokenValue();

			const totalTaxEarned = await getTotalTaxEarned();

			// Update lock record to completed status with final totals
			const traceId = generateTraceId("monthly_system_summary", "SYSTEM");
			await BonusDistribution.updateOne(
				{
					username: "SYSTEM",
					bonusType: "monthly_system_summary",
					"metadata.monthKey": monthKey,
					status: "in_progress",
				},
				{
					$set: {
						status: "completed",
						amountNTE: results.totalBonusesDistributed,
						amountUSD: results.totalBonusesDistributed * tokenValue,
						metadata: {
							...results,
							monthKey,
							totalTaxEarned,
							calculationMethod: "NTE-First",
							tokenValue,
							distributionDuration: duration,
							triggerType: force ? "manual" : "automatic",
							traceId,
							completedAt: new Date(),
						},
					},
				},
			);

			logger.info(
				`MONTHLY BONUS DISTRIBUTION | Job completed: ${results.successful}/${
					results.totalUsers
				} users processed | ${results.totalBonusesDistributed.toFixed(
					4,
				)} NTE distributed | ${results.failed} errors | Month: ${monthKey}`,
			);

			return {
				success: true,
				...results,
				message: `Distributed monthly bonuses to ${results.successful} out of ${results.totalUsers} eligible users`,
			};
		} catch (error) {
			logger.error(
				"MONTHLY BONUS DISTRIBUTION | Error in monthly bonus distribution:",
				error.message,
			);

			// Release lock on error by updating status to failed
			try {
				await BonusDistribution.updateOne(
					{
						username: "SYSTEM",
						bonusType: "monthly_system_summary",
						"metadata.monthKey": monthKey,
						status: "in_progress",
					},
					{
						$set: {
							status: "failed",
							metadata: {
								error: error.message,
								failedAt: new Date(),
								triggerType: force ? "manual" : "automatic",
							},
						},
					},
				);
			} catch (lockError) {
				logger.error(
					"MONTHLY BONUS DISTRIBUTION | Failed to release lock:",
					lockError.message,
				);
			}

			return {
				success: false,
				error: error.message,
				totalEligible: 0,
				successful: 0,
				failed: 0,
			};
		}
	},

	async verifyBonusDistribution(dateString) {
		try {
			logger.info("Running bonus distribution verification", { dateString });

			// Find all active packages
			const allActivePackages = await UserPackage.find({
				status: "active",
			}).lean();

			logger.info(
				`VERIFICATION | Found ${allActivePackages.length} total active packages to verify`,
			);

			const results = {
				totalPackagesChecked: allActivePackages.length,
				missedPackagesProcessed: 0,
				usersAffected: 0,
				additionalBonusesNTE: 0,
				additionalTransactions: 0,
				usersMissed: [],
				distributionSummary: [],
			};

			// Track unique users with missed packages
			const uniqueUsersMissed = new Set();

			// Check each package to see if it has already been processed
			for (const pkg of allActivePackages) {
				try {
					// Skip packages that haven't started yet
					const currentDate = new Date();
					const startDate = new Date(pkg.startDate);
					if (currentDate < startDate) {
						continue;
					}

					// Check if this package already has bonuses distributed today
					const existingDistribution = await BonusDistribution.findOne({
						distributionDate: dateString,
						status: "completed",
						"metadata.userPackageId": pkg._id.toString(),
					}).lean();

					if (!existingDistribution) {
						// This package hasn't been processed, process it now
						logger.info(
							`VERIFICATION | Found missed package for user ${pkg.username} | Package ID: ${pkg._id}`,
						);
						uniqueUsersMissed.add(pkg.username);

						// Process the single package
						const uplineReferrers = await getUserUpline(pkg.username);

						// Get the product details
						const product = await Product.findById(pkg.packageId).lean();
						if (!product) {
							logger.warn(`Product not found for package ${pkg._id}`, {
								packageId: pkg.packageId,
							});
							continue;
						}

						// Create package-specific distribution ID and process it
						const packageDistributionId = new mongoose.Types.ObjectId();

						const packageResult = await this.distributePackageBasedBonuses(
							pkg.username,
							pkg,
							product,
							uplineReferrers,
							packageDistributionId,
						);

						if (packageResult && packageResult.success) {
							results.missedPackagesProcessed++;
							results.additionalBonusesNTE += packageResult.totalBonusNTE || 0;
							results.additionalTransactions +=
								packageResult.transactionsProcessed || 0;
							results.distributionSummary.push(
								...(packageResult.bonusTransactionIds || []),
							); // Collect transaction IDs

							logger.info(
								`VERIFICATION | Recovered package for ${
									pkg.username
								} | Bonus: ${
									formatNTEPrecision(packageResult.totalBonusNTE || 0).display
								} NTE | Transactions: ${
									packageResult.transactionsProcessed || 0
								}`,
							);
						}
					}
				} catch (error) {
					logger.error(
						`VERIFICATION | Error verifying package ${pkg._id}:`,
						error,
					);
					logger.error(`Error verifying package ${pkg._id}:`, {
						message: error?.message,
						stack: error?.stack,
						error,
					});
				}
			}

			results.usersAffected = uniqueUsersMissed.size;
			results.usersMissed = Array.from(uniqueUsersMissed);

			// Log detailed verification results
			if (results.missedPackagesProcessed > 0) {
				logger.info(
					`VERIFICATION SUMMARY | Recovered ${
						results.missedPackagesProcessed
					} packages across ${
						results.usersAffected
					} users | Additional bonuses: ${
						formatNTEPrecision(results.additionalBonusesNTE).display
					} NTE`,
				);

				const getNte = await getTokenValue();
				// Log verification distribution record
				const traceId = generateTraceId("verification_summary", "SYSTEM");
				await this.logBonusDistribution({
					username: "SYSTEM",
					distributionDate: dateString,
					bonusType: "verification_summary",
					amountNTE: results.additionalBonusesNTE,
					amountUSD: results.additionalBonusesNTE * getNte,
					metadata: {
						...results,
						traceId,
						calculationMethod: "NTE-First",
						tokenValue: getNte,
						triggerType: "verification",
						verifiedAt: new Date(),
					},
				});
			}

			return results;
		} catch (error) {
			logger.error("VERIFICATION | Error verifying bonus distribution:", error);
			logger.error("Error in bonus distribution verification:", {
				message: error?.message,
				stack: error?.stack,
				error,
			});
			return {
				totalPackagesChecked: 0,
				missedPackagesProcessed: 0,
				usersAffected: 0,
				additionalBonusesNTE: 0,
				additionalTransactions: 0,
				error: error.message,
			};
		}
	},

	async processEnhancedFinalBonusClaim(packageId, username) {
		try {
			logger.info("Processing final bonus claim", { packageId, username });

			// Get the package
			const userPackage = await UserPackage.findOne({
				_id: new mongoose.Types.ObjectId(packageId),
				username,
				status: "completed",
				finalBonusClaimed: false,
			}).lean();

			if (!userPackage) {
				return {
					success: false,
					message:
						"Package not found, not completed, or final bonus already claimed",
				};
			}

			// Get product details
			const product = await Product.findById(userPackage.packageId).lean();
			if (!product || product.packageType !== "locked") {
				return {
					success: false,
					message: "Product not found or not a locked package",
				};
			}

			// Calculate final bonus based on package type
			let finalBonusNTE = 0;
			let claimType = "";

			if (product.userBonusCalculation?.bonusPercent) {
				// Flexible staking: return bonus + original investment
				const flexibleBonusNTE = calculateFlexibleStakingBonusNTE({
					...product,
					priceNTE: userPackage.priceNTE,
				});
				finalBonusNTE = flexibleBonusNTE + userPackage.priceNTE; // Bonus + Investment
				claimType = "bonus_plus_investment";
			} else if (product.userBonusCalculation?.aprPercent) {
				// APR staking: return only original investment (daily bonuses already distributed)
				finalBonusNTE = userPackage.priceNTE;
				claimType = "investment_return";
			}

			if (finalBonusNTE <= 0) {
				return {
					success: false,
					message: "No final bonus calculated for this package",
				};
			} // Get user details
			const user = await User.findOne({ username }).lean();
			if (!user) {
				return {
					success: false,
					message: "User not found",
				};
			}

			// Mark package as final bonus claimed
			await UserPackage.updateOne(
				{ _id: new mongoose.Types.ObjectId(packageId) },
				{
					$set: {
						finalBonusClaimed: true,
						bonusClaimStatus: "claimed",
						finalBonusClaimedAt: new Date(),
						updatedAt: new Date(),
						finalBonusAmount: finalBonusNTE,
					},
				},
			);

			// Record the transaction with package tracking
			const value = await getTokenValue();
			const traceId = generateTraceId(
				"final_bonus_claim",
				username,
				new Date(),
			);

			const transactionResult = await recordBonusClaim({
				username,
				userId: user._id,
				packageId: userPackage.packageId.toString(),
				packageName: product.name,
				packageType: product.packageType,
				bonusType: "final_bonus_claim",
				claimType: claimType,
				amountNTE: finalBonusNTE,
				amountUSD: finalBonusNTE * value,
				description:
					claimType === "investment_return"
						? `Investment return from ${product.name} package`
						: `Final bonus claim from ${product.name} package`,
				walletBalanceBefore: user.walletBalance || 0,
				originalInvestment: userPackage.priceNTE,
				metadata: {
					traceId,
					userPackageId: userPackage._id.toString(),
					packageId: userPackage.packageId.toString(),
					packageName: product.name,
					packageType: product.packageType,
					originalInvestmentNTE: userPackage.priceNTE,
					originalInvestmentUSD: userPackage.price,
					bonusOrigin: "package_completion",
					claimMethod: "final_bonus_claim",
					claimType: claimType,
					completedAt: new Date(),
					calculationDetails: {
						formula:
							claimType === "investment_return"
								? "APR Package - Return Original Investment"
								: "Flexible Package - Return Bonus + Investment",
						bonusPercent: product.userBonusCalculation?.bonusPercent || null,
						aprPercent: product.userBonusCalculation?.aprPercent || null,
						lockPeriod: product.lockPeriod,
						originalInvestment: userPackage.priceNTE,
						bonusAmount:
							claimType === "bonus_plus_investment"
								? finalBonusNTE - userPackage.priceNTE
								: 0,
						totalClaim: finalBonusNTE,
					},
				},
			});

			if (!transactionResult.success) {
				throw new Error(
					`Failed to record bonus claim transaction: ${transactionResult.error}`,
				);
			}

			const tokenValue = await getTokenValue();

			return {
				success: true,
				data: {
					packageId,
					userPackageId: userPackage._id.toString(),
					username,
					packageName: product.name,
					packageType: product.packageType,
					claimType,
					finalBonusNTE,
					finalBonusUSD: finalBonusNTE * tokenValue,
					claimedAt: new Date(),
					originalInvestmentNTE: userPackage.priceNTE,
					originalInvestmentUSD: userPackage.price,
					bonusAmount:
						claimType === "bonus_plus_investment"
							? finalBonusNTE - userPackage.priceNTE
							: 0,
					bonusAmountUSD:
						claimType === "bonus_plus_investment"
							? (finalBonusNTE - userPackage.priceNTE) * tokenValue
							: 0,
					transactionId: transactionResult.bonusTransactionId,
				},
				message:
					claimType === "investment_return"
						? "Original investment returned successfully"
						: "Bonus and original investment claimed successfully",
			};
		} catch (error) {
			logger.error(
				`Error processing final bonus claim for ${username}:`,
				error,
			);
			return {
				success: false,
				message: "Error processing final bonus claim",
				error: error.message,
			};
		}
	},

	async manualBonusDistribution(username, force = false) {
		try {
			const dateString = getTodayDateString();

			// Check if already distributed today (unless forced)
			if (
				!force &&
				(await this.hasBonusesBeenDistributed(username, dateString))
			) {
				return {
					success: false,
					message:
						"Bonuses already distributed today. Use force parameter to override.",
					alreadyDistributed: true,
				};
			}

			// If forced and already distributed, log the override
			if (force) {
				const traceId = generateTraceId("manual_override", username);

				await this.logBonusDistribution({
					username,
					distributionDate: dateString,
					bonusType: "manual_override",
					amountNTE: 0,
					amountUSD: 0,
					metadata: {
						action: "Manual distribution override",
						reason: "Admin forced distribution",
						calculationMethod: "NTE-First",
						triggerType: "manual",
						traceId: traceId,
					},
				});
			}

			// Execute the distribution
			const result = await this.distributeBonuses(username);

			// Add manual flag to the result and update logging
			if (result.success) {
				result.data.distributionType = "manual";
				result.data.forced = force;

				try {
					// Update the logged distribution summary to mark as manual
					await BonusDistribution.updateOne(
						{
							username,
							distributionDate: dateString,
							bonusType: "daily_summary",
						},
						{
							$set: {
								"metadata.triggerType": "manual",
								"metadata.adminTriggered": true,
								"metadata.forcedDistribution": force,
							},
						},
					);
				} catch (updateError) {
					logger.error("Error updating distribution log:", updateError.message);
				}
			}

			return result;
		} catch (error) {
			logger.error(
				`Error in manual bonus distribution for ${username}:`,
				error,
			);
			return {
				success: false,
				message: "Error in manual bonus distribution",
				error: error.message,
			};
		}
	},

	async distributePackageBasedBonuses(
		username,
		userPackage,
		product,
		uplineReferrers = [],
		packageDistributionId = null,
	) {
		try {
			// Check if users with active packages cache is valid before processing
			if (!(await isUsersWithActivePackagesCacheValid())) {
				logger.warn(
					"CACHE VALIDATION | Users with active packages cache is invalid or expired, updating now...",
				);

				try {
					await updateUsersWithActivePackagesCache();
					logger.info(
						"CACHE VALIDATION | Cache updated successfully, proceeding with bonus distribution",
					);
				} catch (cacheUpdateError) {
					logger.error(
						"CACHE VALIDATION | Failed to update cache, proceeding with existing cache (may have stale data):",
						cacheUpdateError.message,
					);
				}
			} else {
				logger.info(
					"CACHE VALIDATION | Users with active packages cache is valid, proceeding with bonus distribution",
				);
			}

			const tokenValue = await getTokenValue();

			// Get total users who bought this package from cache (sync)
			const totalUsersForPackage = getTotalUsersForPackageSync(
				userPackage.packageId.toString(),
			);

			const minPurchaseUsers = product.minPurchaseUsersForBonus || 0;
			if (totalUsersForPackage < minPurchaseUsers) {
				logger.info(
					`PACKAGE BONUS | SKIPPED: Package ${product.name} for user ${username} | Reason: Insufficient users (${totalUsersForPackage}/${minPurchaseUsers})`,
				);
				return {
					success: true,
					userPackageId: userPackage._id.toString(),
					packageId: userPackage.packageId.toString(),
					packageName: product.name,
					packageType: product.packageType,
					totalBonusNTE: 0,
					totalBonusUSD: 0,
					bonusesCount: 0,
					bonusTypes: [],
					transactionsProcessed: 0,
					distributionId: packageDistributionId?.toString(),
					referralBonusesProcessed: 0,
					skipped: true,
					skipReason: `Minimum user threshold not met (${totalUsersForPackage}/${minPurchaseUsers})`,
				};
			}

			logger.info(
				`PACKAGE BONUS | Processing package ${product.name} for user ${username} | Users: ${totalUsersForPackage}/${minPurchaseUsers} (threshold met) | Type: ${product.packageType}`,
			);

			const productDataForCalculations = {
				...product,
				totalUsers: totalUsersForPackage,
				priceNTE: userPackage.priceNTE,
			};

			// Prepare all bonus payments for this package
			const allBonusPayments = [];
			const bonusTransactionIds = []; // Track all bonus transaction IDs
			let referralBonusesProcessed = 0;
			let referralBonusesTotalNTE = 0;

			logger.info(
				`PACKAGE BONUS | Starting bonus distribution for ${username} | Package: ${product.name} | Type: ${product.packageType}`,
			);

			// 1. USER STAKING/DAILY BONUSES
			if (product.packageType === "unlocked") {
				// Calculate daily bonus for unlocked packages
				if (product.userBonusCalculation?.annualBonus) {
					const dailyBonusNTE = calculateBonusPerUserNTE(
						productDataForCalculations,
					);

					const traceId = generateTraceId("daily_bonus", username, new Date());

					if (dailyBonusNTE > 0) {
						allBonusPayments.push({
							type: "daily_bonus",
							amountNTE: dailyBonusNTE,
							amountUSD: dailyBonusNTE * tokenValue,
							description: `Daily bonus from ${product.name} package`,
							metadata: {
								traceId,
								userPackageId: userPackage._id.toString(),
								packageId: userPackage.packageId.toString(),
								packageName: product.name,
								packageType: product.packageType,
								bonusOrigin: "package_based",
								calculationDetails:
									"Annual Bonus Pool ÷ Lock Period Years ÷ Days in Year ÷ Total Users",
							},
						});
					}
				}
			} else if (product.packageType === "locked") {
				// Calculate staking bonus for locked packages with APR
				if (product.userBonusCalculation?.aprPercent) {
					const stakingBonusNTE = calculateStakingBonusNTE(
						productDataForCalculations,
					);

					const traceId = generateTraceId(
						"staking_bonus",
						username,
						new Date(),
					);

					if (stakingBonusNTE > 0) {
						allBonusPayments.push({
							type: "staking_bonus",
							amountNTE: stakingBonusNTE,
							amountUSD: stakingBonusNTE * tokenValue,
							description: `Daily staking bonus from ${product.name} package (${product.userBonusCalculation.aprPercent}% APR)`,
							metadata: {
								traceId,
								userPackageId: userPackage._id.toString(),
								packageId: userPackage.packageId.toString(),
								packageName: product.name,
								packageType: product.packageType,
								bonusOrigin: "package_based",
								calculationDetails: {
									formula: `Investment × ${product.userBonusCalculation.aprPercent}% APR ÷ 365 days`,
									investment: userPackage.priceNTE,
									percentage: product.userBonusCalculation.aprPercent,
								},
							},
						});
					}
				}

				// Handle flexible staking bonuses (check if lock period has ended)
				if (
					product.userBonusCalculation?.bonusPercent &&
					product.lockPeriod?.value
				) {
					const purchaseDate = new Date(userPackage.purchasedAt);
					const lockPeriodDays = product.lockPeriod.value;
					const lockEndDate = new Date(
						purchaseDate.getTime() + lockPeriodDays * 24 * 60 * 60 * 1000,
					);
					const today = new Date();

					// Check if lock period has ended and bonus hasn't been paid yet
					if (today >= lockEndDate) {
						const traceId = generateTraceId(
							"flexible_staking_bonus",
							username,
							new Date(),
						);

						const existingFlexibleBonus = await BonusDistribution.findOne({
							username,
							"metadata.userPackageId": userPackage._id.toString(),
							bonusType: "flexible_staking_bonus",
							status: "completed",
						}).lean();

						if (!existingFlexibleBonus) {
							const flexibleBonusNTE = calculateFlexibleStakingBonusNTE(
								productDataForCalculations,
							);

							if (flexibleBonusNTE > 0) {
								allBonusPayments.push({
									type: "flexible_staking_bonus",
									amountNTE: flexibleBonusNTE,
									amountUSD: flexibleBonusNTE * tokenValue,
									description: `Flexible staking bonus from ${product.name} package (${product.userBonusCalculation.bonusPercent}% - Lock period ended)`,
									metadata: {
										traceId,
										userPackageId: userPackage._id.toString(),
										packageId: userPackage.packageId.toString(),
										packageName: product.name,
										packageType: product.packageType,
										bonusOrigin: "package_based",
										calculationDetails: {
											formula: `Investment × ${product.userBonusCalculation.bonusPercent}% at term completion`,
											investment: userPackage.priceNTE,
											percentage: product.userBonusCalculation.bonusPercent,
										},
									},
								});
							}
						}
					}
				}
			}

			// 2. DAILY REFERRAL BONUSES FOR UPLINE
			if (
				product.referralBonusCalculation?.perDayPercent &&
				uplineReferrers.length > 0
			) {
				logger.info(
					`DAILY REFERRAL | Processing for ${username}'s package ${product.name} | Referrers: ${uplineReferrers.length}`,
				);
				logger.info(
					`DAILY REFERRAL | Product perDayPercent: first=${
						product.referralBonusCalculation.perDayPercent.firstGen || 0
					}%, second=${
						product.referralBonusCalculation.perDayPercent.secondGen || 0
					}%, others=${
						product.referralBonusCalculation.perDayPercent.othersGen || 0
					}%`,
				);

				// Filter and process eligible referrers only
				const eligibleReferrers = uplineReferrers.filter((ref) => {
					// Check max generation limit
					const maxGen = product?.referralBonusCalculation?.maxGeneration || 3;
					if (ref.level > maxGen) {
						logger.info(
							`${
								ref.level === 1 ? "1st" : ref.level === 2 ? "2nd" : "Others"
							} DAILY REFERRAL | SKIPPED | Gen ${ref.level} | Referrer: ${
								ref.username
							} | Reason: Exceeds max generation limit (${maxGen})`,
						);
						return false;
					}

					// Determine percentage based on level
					const percentage =
						ref.level === 1
							? product.referralBonusCalculation.perDayPercent.firstGen || 0
							: ref.level === 2
							? product.referralBonusCalculation.perDayPercent.secondGen || 0
							: product.referralBonusCalculation.perDayPercent.othersGen || 0;

					// Skip if percentage is 0
					if (percentage === 0) {
						logger.info(
							`${
								ref.level === 1 ? "1st" : ref.level === 2 ? "2nd" : "Others"
							} DAILY REFERRAL | SKIPPED | Gen ${ref.level} | Referrer: ${
								ref.username
							} | Reason: 0% configured for this generation`,
						);
						return false;
					}

					// Check if referrer has active unlocked packages using cache
					if (!hasActiveUnlockedPackageSync(ref.username)) {
						logger.info(
							`${
								ref.level === 1 ? "1st" : ref.level === 2 ? "2nd" : "Others"
							} DAILY REFERRAL | SKIPPED | Gen ${ref.level} | Referrer: ${
								ref.username
							} | Reason: No active unlocked packages`,
						);
						return false;
					}

					return true;
				});

				logger.info(
					`DAILY REFERRAL | Eligible referrers: ${eligibleReferrers.length}/${uplineReferrers.length} (after filtering)`,
				);

				// Process only eligible referrers
				const dailyReferralBonuses = eligibleReferrers.map((ref) => {
					// Determine percentage based on level
					const percentage =
						ref.level === 1
							? product.referralBonusCalculation.perDayPercent.firstGen || 0
							: ref.level === 2
							? product.referralBonusCalculation.perDayPercent.secondGen || 0
							: product.referralBonusCalculation.perDayPercent.othersGen || 0;

					// Calculate bonus
					const dailyBonusNTE = calculateDailyReferralBonusNTE(
						ref.level,
						productDataForCalculations,
					);

					logger.info(
						`${
							ref.level === 1 ? "1st" : ref.level === 2 ? "2nd" : "Others"
						} DAILY REFERRAL | Generation ${ref.level} | Referrer ${
							ref.username
						} | Percentage: ${percentage}% | Calculated bonus: ${dailyBonusNTE.toFixed(
							4,
						)} NTE | Active packages: YES`,
					);

					return {
						referrer: ref,
						level: ref.level,
						type: "daily_referral",
						dailyBonusNTE: dailyBonusNTE,
						percentage: percentage,
					};
				});

				// Generate trace ID for referral bonus distribution
				const traceId = generateTraceId("daily_referral", username);

				// Distribute to each eligible referrer using enhanced referral tracking
				for (const bonus of dailyReferralBonuses) {
					try {
						// Get referrer user details
						const referrerUser = await User.findOne({
							username: bonus.referrer.username,
						}).lean();

						logger.info(
							`${
								bonus.level === 1 ? "1st" : bonus.level === 2 ? "2nd" : "Others"
							} DAILY REFERRAL | PROCESSING Gen ${bonus.level} | Referrer: ${
								bonus.referrer.username
							} | Amount: ${bonus.dailyBonusNTE.toFixed(4)} NTE | Percentage: ${
								bonus.percentage
							}%`,
						);

						if (referrerUser && bonus.dailyBonusNTE > 0) {
							const result = await recordReferralBonus({
								username: bonus.referrer.username,
								userId: referrerUser._id,
								purchaserUsername: username,
								generation: bonus.level,
								percentage: bonus.percentage,
								baseValue: userPackage.priceNTE,
								amountNTE: bonus.dailyBonusNTE,
								amountUSD: bonus.dailyBonusNTE * tokenValue,
								packageDetails: {
									packageId: userPackage.packageId.toString(),
									name: product.name,
									packageType: product.packageType,
									userPackageId: userPackage._id.toString(),
								},
								walletBalanceBefore: referrerUser.walletBalance || 0,
								purchaseDistributionId: packageDistributionId,
								bonusType: "daily_referral",
								traceId,
								session: null,
								metadata: {
									eligibilityCheck: {
										hasActiveUnlockedPackages: true,
										checkedViaCache: true,
									},
									calculationDetails: {
										formula: `Base value (${userPackage.priceNTE} NTE) × ${bonus.percentage}% daily rate`,
										baseValue: userPackage.priceNTE,
										percentage: bonus.percentage,
										generation: bonus.level,
									},
								},
							});

							if (result.success) {
								referralBonusesProcessed++;
								referralBonusesTotalNTE += bonus.dailyBonusNTE;
								bonusTransactionIds.push(result.bonusTransactionId);

								logger.info(
									`${
										bonus.level === 1
											? "1st"
											: bonus.level === 2
											? "2nd"
											: "Others"
									} DAILY REFERRAL | SUCCESS | User ${
										bonus.referrer.username
									} received ${bonus.dailyBonusNTE.toFixed(
										4,
									)} NTE | Generation: ${bonus.level}`,
								);
							} else {
								logger.error(
									`${
										bonus.level === 1
											? "1st"
											: bonus.level === 2
											? "2nd"
											: "Others"
									} DAILY REFERRAL | FAILED | User ${
										bonus.referrer.username
									} | Gen: ${bonus.level} | Error: ${
										result.error || "Unknown error"
									}`,
								);
							}
						} else {
							if (!referrerUser) {
								logger.warn(
									`${
										bonus.level === 1
											? "1st"
											: bonus.level === 2
											? "2nd"
											: "Others"
									} DAILY REFERRAL | SKIPPED | Referrer not found: ${
										bonus.referrer.username
									} | Gen: ${bonus.level}`,
								);
							} else if (bonus.dailyBonusNTE <= 0) {
								logger.warn(
									`${
										bonus.level === 1
											? "1st"
											: bonus.level === 2
											? "2nd"
											: "Others"
									} DAILY REFERRAL | SKIPPED | Zero bonus amount (${
										bonus.dailyBonusNTE
									}) | User: ${bonus.referrer.username} | Gen: ${bonus.level}`,
								);
							}
						}
					} catch (error) {
						logger.error(
							`Error processing daily referral bonus for ${
								bonus.referrer.username
							}: ${error && error.message ? error.message : error}`,
						);
						if (error && error.stack) {
							logger.error(`Stack trace: ${error.stack}`);
						}
					}
				}

				logger.info(
					`DAILY REFERRAL | COMPLETED for ${username}'s package ${product.name} | Eligible referrers processed: ${dailyReferralBonuses.length}/${uplineReferrers.length} | Total bonuses distributed: ${referralBonusesProcessed}`,
				);
			}

			// 3. TARGET FILL BONUSES (For Unlocked Packages)
			if (
				product.packageType === "unlocked" &&
				product.referralBonusTargetFill &&
				product.referralBonusTargetFill.groupSize > 0 &&
				uplineReferrers.length > 0
			) {
				// Calculate potential groups for the user
				const referrerCount = uplineReferrers.length;
				const potentialGroups = Math.floor(
					referrerCount / product.referralBonusTargetFill.groupSize,
				);

				if (potentialGroups > 0) {
					// Check if target fill bonus already paid for current group count
					const existingTargetFill = await BonusDistribution.findOne({
						username,
						"metadata.userPackageId": userPackage._id.toString(),
						bonusType: "target_fill_bonus",
						"metadata.completedGroups": potentialGroups,
						status: "completed",
					}).lean();

					if (!existingTargetFill) {
						const targetFillBonusNTE = await calculateTargetFillBonusNTE(
							productDataForCalculations,
							potentialGroups,
						);

						const traceId = generateTraceId(
							"target_fill_bonus",
							username,
							new Date(),
						);

						if (targetFillBonusNTE > 0) {
							allBonusPayments.push({
								type: "target_fill_bonus",
								amountNTE: targetFillBonusNTE,
								amountUSD: targetFillBonusNTE * tokenValue,
								description: `Target fill bonus from ${product.name} package (${potentialGroups} complete groups of ${product.referralBonusTargetFill.groupSize})`,
								metadata: {
									traceId,
									userPackageId: userPackage._id.toString(),
									packageId: userPackage.packageId.toString(),
									packageName: product.name,
									packageType: product.packageType,
									bonusOrigin: "package_based",
									completedGroups: potentialGroups,
									groupSize: product.referralBonusTargetFill.groupSize,
									referrerCount: referrerCount,
									calculationDetails: {
										formula: `USD ${product.referralBonusTargetFill.bonusPerGroupUSD} per completed group`,
										completedGroups: potentialGroups,
										bonusPerGroup:
											product.referralBonusTargetFill.bonusPerGroupUSD,
									},
								},
							});
						}
					}
				}
			}

			// 4. UPDATE USER'S OWN WALLET
			if (allBonusPayments.length > 0) {
				for (const payment of allBonusPayments) {
					const result = await this.recordBonusTransaction({
						username,
						userId: user._id,
						bonusType: payment.type,
						amountNTE: payment.amountNTE,
						amountUSD: payment.amountUSD,
						description: payment.description,
						metadata: payment.metadata,
						walletBalanceBefore: user.walletBalance || 0,
						distributionId: packageDistributionId,
						userPackageId: userPackage._id,
						packageId: userPackage.packageId,
						packageName: product.name,
						packageType: product.packageType,
					});

					if (result.success && result.bonusTransactionId) {
						bonusTransactionIds.push(result.bonusTransactionId);
					} else {
						if (result.success) {
							logger.info(
								`USER BONUS | DB WRITE: User ${username} received ${payment.amountNTE.toFixed(
									4,
								)} NTE | Type: ${payment.type} | Package: ${product.name}`,
							);
						} else {
							logger.error(
								`USER BONUS | ERROR: Failed to record bonus transaction for ${username} | Type: ${payment.type} | Error: ${result.error}`,
							);
						}
					}
				}
			}

			// Calculate totals
			const totalBonusNTE = allBonusPayments.reduce(
				(sum, payment) => sum + payment.amountNTE,
				0,
			);
			const totalBonusUSD = totalBonusNTE * tokenValue;
			const bonusTypes = allBonusPayments.map((payment) => payment.type);

			logger.info(
				`PACKAGE BONUS | allBonusPayments for ${username} package ${product.name}:`,
				{
					allBonusPayments: allBonusPayments.map((p) => ({
						type: p.type,
						amountNTE: p.amountNTE,
					})),
					totalBonusNTE,
					totalUsersForPackage,
					minPurchaseUsers,
					productType: product.packageType,
					userBonusCalculation: product.userBonusCalculation,
					referralBonusesProcessed,
				},
			);

			return {
				success: true,
				userPackageId: userPackage._id.toString(),
				packageId: userPackage.packageId.toString(),
				packageName: product.name,
				packageType: product.packageType,
				totalBonusNTE,
				totalBonusUSD,
				bonusesCount: allBonusPayments.length,
				bonusTypes,
				transactionsProcessed:
					allBonusPayments.length + referralBonusesProcessed,
				distributionId: packageDistributionId?.toString(),
				referralBonusesProcessed,
				referralBonusesTotalNTE,
				bonusTransactionIds,
			};
		} catch (error) {
			logger.error(
				`Error in distributePackageBasedBonuses for ${username}:`,
				{
					errorMessage: error?.message,
					errorStack: error?.stack,
					username,
					userPackageId: userPackage?._id?.toString(),
					packageId: userPackage?.packageId?.toString(),
					productName: product?.name,
					productType: product?.packageType,
				},
			);
			
			// Log the actual error object for debugging
			logger.error(
				`Full error details for ${username}:`,
				error,
			);
			
			return {
				success: false,
				error: error.message,
				userPackageId: userPackage._id?.toString(),
				packageId: userPackage.packageId?.toString(),
				packageName: product?.name,
				packageType: product?.packageType,
				totalBonusNTE: 0,
				totalBonusUSD: 0,
				bonusesCount: 0,
				bonusTypes: [],
				transactionsProcessed: 0,
				referralBonusesTotalNTE: 0,
			};
		}
	},
};

module.exports = { bonusService, updateAllExpiredPackages };
