const { BonusDistribution } = require("./bonusDistribution.model");
const { getTokenValue } = require("../config/globalConfig.service");

const bonusDistributionService = {
	async getDistributionHistory(queryParams) {
		const {
			page = 1,
			limit = 10,
			sortBy = "loggedAt",
			sortOrder = "desc",
			search,
			username,
			bonusType,
			status,
			startDate,
			endDate,
			timezone = 0,
		} = queryParams;

		const parsedLimit = Math.min(Math.max(Number(limit) || 10, 1), 100);
		const parsedPage = Math.max(1, Number(page) || 1);
		const skip = (parsedPage - 1) * parsedLimit;

		// Build search query
		const searchQuery = {};

		if (username && username.trim()) {
			searchQuery.username = { $regex: username, $options: "i" };
		}

		if (bonusType && bonusType.trim()) {
			searchQuery.bonusType = bonusType;
		}

		if (status && status.trim()) {
			searchQuery.status = status;
		}

		// Date range with timezone handling
		if ((startDate && startDate.trim()) || (endDate && endDate.trim())) {
			searchQuery.loggedAt = {};
			const clientTimezoneOffset = parseInt(timezone) || 0;
			const offsetMs = -clientTimezoneOffset * 60 * 1000;

			const parseLocalDateToUTC = (dateStr, endOfDay = false) => {
				const [year, month, day] = dateStr.split("-").map(Number);
				const utcDate = new Date(
					Date.UTC(
						year,
						month - 1,
						day,
						endOfDay ? 23 : 0,
						endOfDay ? 59 : 0,
						endOfDay ? 59 : 0,
						endOfDay ? 999 : 0,
					),
				);
				return new Date(utcDate.getTime() - offsetMs);
			};

			if (startDate && startDate.trim()) {
				searchQuery.loggedAt.$gte = parseLocalDateToUTC(startDate, false);
			}

			if (endDate && endDate.trim()) {
				searchQuery.loggedAt.$lte = parseLocalDateToUTC(endDate, true);
			}
		}

		// Text search across multiple fields
		if (search && search.trim()) {
			searchQuery.$or = [
				{ username: { $regex: search, $options: "i" } },
				{ referrerUsername: { $regex: search, $options: "i" } },
				{ purchaserUsername: { $regex: search, $options: "i" } },
				{ bonusType: { $regex: search, $options: "i" } },
				{ "metadata.packageName": { $regex: search, $options: "i" } },
				{ transactionHash: { $regex: search, $options: "i" } },
				{ refundTxHash: { $regex: search, $options: "i" } },
				{ "metadata.transactionHash": { $regex: search, $options: "i" } },
				{ "metadata.fromAddress": { $regex: search, $options: "i" } },
				{ "metadata.toAddress": { $regex: search, $options: "i" } },
			];
		}

		const sortObj = {};
		sortObj[sortBy] = sortOrder === "asc" ? 1 : -1;

		const tokenValue = await getTokenValue();

		const [distributions, totalCount, summaryStats, systemSummaryStats] =
			await Promise.all([
				BonusDistribution.find(searchQuery)
					.sort(sortObj)
					.skip(skip)
					.limit(parsedLimit)
					.lean(),
				BonusDistribution.countDocuments(searchQuery),
				this._getFilteredSummary(searchQuery, tokenValue),
				this._getSystemSummary(),
			]);

		const totalPages = Math.ceil(totalCount / parsedLimit);

		// Calculate USD for distributions missing it
		const processedDistributions = distributions.map((dist) => ({
			...dist,
			_id: dist._id.toString(),
			amountNTE: dist.amountNTE || 0,
			amountUSD: this._calculateUSD(dist.amountNTE, dist.amountUSD, tokenValue),
		}));

		const safeStats = {
			totalUsers: summaryStats.totalUsers || 0,
			totalAmountNTE: summaryStats.totalAmountNTE || 0,
			totalAmountUSD: summaryStats.totalAmountUSD || 0,
			averageAmountNTE: summaryStats.averageAmountNTE || 0,
			averageAmountUSD: summaryStats.averageAmountUSD || 0,
			minAmountNTE: summaryStats.minAmountNTE || 0,
			maxAmountNTE: summaryStats.maxAmountNTE || 0,
			uniqueBonusTypes: summaryStats.uniqueBonusTypes || 0,
			uniqueStatuses: summaryStats.uniqueStatuses || 0,
			totalTransactions: summaryStats.totalTransactions || 0,
			transactionsWithOriginalUSD: summaryStats.transactionsWithOriginalUSD || 0,
			transactionsWithCalculatedUSD: summaryStats.transactionsWithCalculatedUSD || 0,
		};

		return {
			success: true,
			distributions: processedDistributions,
			pagination: {
				total: totalCount,
				page: parsedPage,
				limit: parsedLimit,
				totalPages,
				hasNext: parsedPage < totalPages,
				hasPrev: parsedPage > 1,
			},
			summary: {
				filtered: {
					totalDistributions: totalCount,
					totalUsers: safeStats.totalUsers,
					totalAmountNTE: safeStats.totalAmountNTE,
					totalAmountUSD: safeStats.totalAmountUSD,
					averageAmountNTE: safeStats.averageAmountNTE,
					averageAmountUSD: safeStats.averageAmountUSD,
					minAmountNTE: safeStats.minAmountNTE,
					maxAmountNTE: safeStats.maxAmountNTE,
					uniqueBonusTypes: safeStats.uniqueBonusTypes,
					uniqueStatuses: safeStats.uniqueStatuses,
				},
				systemTotals: {
					totalSystemDistributions: systemSummaryStats.totalSystemDistributions || 0,
					totalUsersAffected: systemSummaryStats.totalUsersAffected || 0,
					totalSuccessfulDistributions: systemSummaryStats.totalSuccessfulDistributions || 0,
					totalFailedDistributions: systemSummaryStats.totalFailedDistributions || 0,
					totalBonusesDistributed: systemSummaryStats.totalBonusesDistributed || 0,
					totalTransactions: systemSummaryStats.totalTransactions || 0,
					averageDistributionDuration: systemSummaryStats.averageDistributionDuration || 0,
					totalDistributionDays: systemSummaryStats.totalDistributionDays || 0,
					lastDistributionDate: systemSummaryStats.lastDistributionDate || null,
					firstDistributionDate: systemSummaryStats.firstDistributionDate || null,
					successRate:
						systemSummaryStats.totalUsersAffected > 0
							? ((systemSummaryStats.totalSuccessfulDistributions / systemSummaryStats.totalUsersAffected) * 100).toFixed(2)
							: 0,
				},
				debug: {
					transactionsWithOriginalUSD: safeStats.transactionsWithOriginalUSD,
					transactionsWithCalculatedUSD: safeStats.transactionsWithCalculatedUSD,
					tokenValue,
				},
			},
		};
	},

	_calculateUSD(amountNTE, amountUSD, tokenValue) {
		if (!amountUSD || amountUSD === 0 || isNaN(amountUSD)) {
			return (amountNTE || 0) * tokenValue;
		}
		return amountUSD;
	},

	async _getFilteredSummary(searchQuery, tokenValue) {
		const result = await BonusDistribution.aggregate([
			{ $match: searchQuery },
			{
				$addFields: {
					actualAmountUSD: {
						$cond: {
							if: {
								$or: [
									{ $eq: ["$amountUSD", null] },
									{ $eq: ["$amountUSD", 0] },
									{ $not: ["$amountUSD"] },
									{ $eq: [{ $type: "$amountUSD" }, "missing"] },
								],
							},
							then: { $multiply: [{ $ifNull: ["$amountNTE", 0] }, tokenValue] },
							else: { $ifNull: ["$amountUSD", 0] },
						},
					},
					calculationAmountNTE: {
						$cond: {
							if: { $eq: ["$bonusType", "package_purchase"] },
							then: { $abs: { $ifNull: ["$amountNTE", 0] } },
							else: { $ifNull: ["$amountNTE", 0] },
						},
					},
				},
			},
			{
				$addFields: {
					calculationAmountUSD: {
						$cond: {
							if: { $eq: ["$bonusType", "package_purchase"] },
							then: { $abs: { $ifNull: ["$actualAmountUSD", 0] } },
							else: { $ifNull: ["$actualAmountUSD", 0] },
						},
					},
				},
			},
			{
				$group: {
					_id: null,
					totalUsers: { $addToSet: "$username" },
					totalAmountNTE: { $sum: { $ifNull: ["$calculationAmountNTE", 0] } },
					totalAmountUSD: { $sum: { $ifNull: ["$calculationAmountUSD", 0] } },
					averageAmountNTE: { $avg: { $ifNull: ["$calculationAmountNTE", 0] } },
					averageAmountUSD: { $avg: { $ifNull: ["$calculationAmountUSD", 0] } },
					minAmountNTE: { $min: { $ifNull: ["$calculationAmountNTE", 0] } },
					maxAmountNTE: { $max: { $ifNull: ["$calculationAmountNTE", 0] } },
					bonusTypes: { $addToSet: "$bonusType" },
					statuses: { $addToSet: "$status" },
					totalTransactions: { $sum: 1 },
					transactionsWithOriginalUSD: {
						$sum: {
							$cond: [
								{
									$and: [
										{ $ne: ["$amountUSD", null] },
										{ $ne: ["$amountUSD", 0] },
										{ $ne: [{ $type: "$amountUSD" }, "missing"] },
									],
								},
								1,
								0,
							],
						},
					},
					transactionsWithCalculatedUSD: {
						$sum: {
							$cond: [
								{
									$or: [
										{ $eq: ["$amountUSD", null] },
										{ $eq: ["$amountUSD", 0] },
										{ $not: ["$amountUSD"] },
										{ $eq: [{ $type: "$amountUSD" }, "missing"] },
									],
								},
								1,
								0,
							],
						},
					},
				},
			},
			{
				$project: {
					totalUsers: { $size: "$totalUsers" },
					totalAmountNTE: { $ifNull: ["$totalAmountNTE", 0] },
					totalAmountUSD: { $ifNull: ["$totalAmountUSD", 0] },
					averageAmountNTE: { $ifNull: ["$averageAmountNTE", 0] },
					averageAmountUSD: { $ifNull: ["$averageAmountUSD", 0] },
					minAmountNTE: { $ifNull: ["$minAmountNTE", 0] },
					maxAmountNTE: { $ifNull: ["$maxAmountNTE", 0] },
					uniqueBonusTypes: { $size: "$bonusTypes" },
					uniqueStatuses: { $size: "$statuses" },
					totalTransactions: 1,
					transactionsWithOriginalUSD: { $ifNull: ["$transactionsWithOriginalUSD", 0] },
					transactionsWithCalculatedUSD: { $ifNull: ["$transactionsWithCalculatedUSD", 0] },
				},
			},
		]);

		return result[0] || {};
	},

	async _getSystemSummary() {
		const result = await BonusDistribution.aggregate([
			{
				$match: {
					bonusType: "daily_system_summary",
					username: "SYSTEM",
				},
			},
			{
				$group: {
					_id: null,
					totalSystemDistributions: { $sum: 1 },
					totalUsersAffected: { $sum: "$metadata.totalUsers" },
					totalSuccessfulDistributions: { $sum: "$metadata.successful" },
					totalFailedDistributions: { $sum: "$metadata.failed" },
					totalBonusesDistributed: { $sum: "$metadata.totalBonusesDistributed" },
					totalTransactions: { $sum: "$metadata.totalTransactions" },
					averageDistributionDuration: { $avg: "$metadata.distributionDuration" },
					totalDistributionDays: { $sum: 1 },
					lastDistributionDate: { $max: "$distributionDate" },
					firstDistributionDate: { $min: "$distributionDate" },
				},
			},
		]);

		return result[0] || {};
	},
};

module.exports = { bonusDistributionService };
