const { bonusTransactionService } = require("./bonusTransaction.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");

const bonusTransactionController = {
	// Get all bonus transactions with filters and pagination
	async listBonusTransactions(req, res, next) {
		try {
			const {
				username,
				productId,
				page = 1,
				limit = 10,
				sortBy = "createdAt",
				sortOrder = "desc",
				bonusType,
				status,
				startDate,
				endDate,
				type,
			} = req.query;

			if (!username) {
				throw new ApiError(400, "Username is required");
			}

			const options = {
				username,
				productId,
				page: parseInt(page),
				limit: parseInt(limit),
				sortBy,
				sortOrder,
				bonusType,
				status,
				startDate,
				endDate,
				type,
			};

			const result =
				await bonusTransactionService.listBonusTransactions(options);

			res.json(
				new ApiResponse(
					200,
					result,
					"Bonus transactions retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get a single bonus transaction by ID
	async getBonusTransactionById(req, res, next) {
		try {
			const { id, type } = req.params;

			const transaction = await bonusTransactionService.getBonusTransactionById(
				id,
				type,
			);

			if (!transaction) {
				throw new ApiError(404, "Bonus transaction not found");
			}

			res.json(
				new ApiResponse(
					200,
					{ transaction },
					"Bonus transaction retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Process enhanced final bonus claim
	async processEnhancedFinalBonusClaim(req, res, next) {
		try {
			const { username, packageId } = req.body;
			if (!username || !packageId) {
				throw new ApiError(400, "Username and Package ID are required");
			}
			const result = await bonusTransactionService.claimFinalBonusTransaction(
				packageId,
				username,
			);

			res.json(
				new ApiResponse(200, result, "Final bonus claimed successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get transaction history with filters and stats (admin only)
	async getTransactionHistory(req, res, next) {
		try {
			const result = await bonusTransactionService.getTransactionHistory(
				req.query,
			);
			res.json(
				new ApiResponse(
					200,
					result,
					"Transaction history retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},
};

module.exports = { bonusTransactionController };
