const { BonusTransaction } = require("./bonusTransaction.model");
const { WalletTransaction } = require("../wallets/walletTransaction.model");
const { ApiError } = require("../../utils/ApiError");
const { bonusService } = require("./bonus.service");

const bonusTransactionService = {
	// Get bonus transactions list with filters and pagination
	async listBonusTransactions(options) {
		const {
			username,
			productId,
			page = 1,
			limit = 10,
			sortBy = "createdAt",
			sortOrder = "desc",
			bonusType,
			status,
			startDate,
			endDate,
			type,
		} = options;

		const skip = (page - 1) * limit;
		const filter = {};
		const isWalletType = type === "wallet";
		const TransactionModel = isWalletType
			? WalletTransaction
			: BonusTransaction;

		// Username filter (required)
		if (username) {
			filter.username = username;
		}

		// ProductId filter (optional)
		if (productId) {
			filter["metadata.packageId"] = productId;
		}

		// Bonus type filter (optional)
		if (bonusType) {
			if (isWalletType) {
				// For wallet transactions, map bonusType filter to subType field
				filter.subType = bonusType;
			} else {
				// For bonus transactions, use existing bonusType semantics
				if (bonusType === "referral") {
					filter.bonusType = {
						$in: ["daily_referral", "immediate_referral"],
					};
				} else {
					filter.bonusType = bonusType;
				}
			}
		}

		// Status filter (optional)
		if (status) {
			filter.status = status;
		}

		// Date range filter (optional)
		if (startDate || endDate) {
			filter.createdAt = {};
			if (startDate) {
				filter.createdAt.$gte = new Date(startDate);
			}
			if (endDate) {
				filter.createdAt.$lte = new Date(endDate);
			}
		}

		const sortOptions = {};
		sortOptions[sortBy] = sortOrder === "asc" ? 1 : -1;

		const [transactions, total] = await Promise.all([
			TransactionModel.find(filter)
				.sort(sortOptions)
				.skip(skip)
				.limit(limit)
				.lean(),
			TransactionModel.countDocuments(filter),
		]);

		// Calculate summary statistics
		const summaryPipeline = isWalletType
			? [
					{ $match: filter },
					{
						$group: {
							_id: null,
							totalTransactions: { $sum: 1 },
							totalNTE: { $sum: "$amountNTE" },
							totalUSD: { $sum: "$amountUSD" },
							walletTransactions: { $sum: 1 },
						},
					},
				]
			: [
					{ $match: filter },
					{
						$group: {
							_id: null,
							totalTransactions: { $sum: 1 },
							totalNTE: { $sum: "$amountNTE" },
							totalUSD: { $sum: "$amountUSD" },
							walletTransactions: {
								$sum: {
									$cond: [{ $eq: ["$transactionType", "wallet"] }, 1, 0],
								},
							},
						},
					},
				];

		const summaryResult = await TransactionModel.aggregate(summaryPipeline);
		const summary =
			summaryResult.length > 0
				? {
						totalTransactions: summaryResult[0].totalTransactions,
						walletTransactions: summaryResult[0].walletTransactions,
						totalNTE: summaryResult[0].totalNTE,
						totalUSD: summaryResult[0].totalUSD,
					}
				: {
						totalTransactions: 0,
						walletTransactions: 0,
						totalNTE: 0,
						totalUSD: 0,
					};

		const totalPages = Math.ceil(total / limit);

		return {
			username,
			productId: productId || null,
			transactions,
			summary,
			pagination: {
				total,
				page,
				limit,
				totalPages,
				hasNext: page < totalPages,
				hasPrev: page > 1,
			},
		};
	},

	// Find bonus transaction by ID
	async getBonusTransactionById(id, type) {
		let transaction;
		if (type === "wallet") {
			transaction = await WalletTransaction.findById(id).lean();
		} else {
			transaction = await BonusTransaction.findById(id).lean();
		}
		if (!transaction) {
			throw new ApiError(404, "Bonus transaction not found");
		}

		return transaction;
	},

	// Claim final bonus
	async claimFinalBonusTransaction(packageId, username) {
		if (!packageId || !username) {
			throw new ApiError(
				400,
				"Package ID and username are required to claim final bonus",
			);
		}

		const result = await bonusService.processEnhancedFinalBonusClaim(
			packageId,
			username,
		);
		return result;
	},

	// Get transaction history with filters, pagination, and statistics (admin only)
	async getTransactionHistory(queryParams) {
		const {
			page = 1,
			limit = 10,
			sortBy = "createdAt",
			sortOrder = "desc",
			search,
			username,
			bonusType,
			status,
			productId,
			startDate,
			endDate,
			timezone = 0,
		} = queryParams;

		const parsedLimit = Math.min(Math.max(Number(limit) || 10, 1), 100);
		const parsedPage = Math.max(1, Number(page) || 1);
		const skip = (parsedPage - 1) * parsedLimit;

		// Build search query
		const searchQuery = {};

		if (username && username.trim()) {
			searchQuery.username = { $regex: username, $options: "i" };
		}

		if (bonusType && bonusType.trim()) {
			searchQuery.bonusType = bonusType;
		}

		if (status && status.trim()) {
			searchQuery.status = status;
		}

		if (productId && productId.trim()) {
			searchQuery["metadata.packageId"] = productId;
		}

		// Date range with timezone handling
		if ((startDate && startDate.trim()) || (endDate && endDate.trim())) {
			searchQuery.createdAt = {};
			const clientTimezoneOffset = parseInt(timezone) || 0;
			const offsetMs = -clientTimezoneOffset * 60 * 1000;

			const parseLocalDateToUTC = (dateStr, endOfDay = false) => {
				const [year, month, day] = dateStr.split("-").map(Number);
				const utcDate = new Date(
					Date.UTC(
						year,
						month - 1,
						day,
						endOfDay ? 23 : 0,
						endOfDay ? 59 : 0,
						endOfDay ? 59 : 0,
						endOfDay ? 999 : 0,
					),
				);
				return new Date(utcDate.getTime() - offsetMs);
			};

			if (startDate && startDate.trim()) {
				searchQuery.createdAt.$gte = parseLocalDateToUTC(startDate, false);
			}

			if (endDate && endDate.trim()) {
				searchQuery.createdAt.$lte = parseLocalDateToUTC(endDate, true);
			}
		}

		// Text search across multiple fields
		if (search && search.trim()) {
			searchQuery.$or = [
				{ username: { $regex: search, $options: "i" } },
				{ bonusType: { $regex: search, $options: "i" } },
				{ "metadata.packageName": { $regex: search, $options: "i" } },
				{ "metadata.packageId": { $regex: search, $options: "i" } },
				{ transactionType: { $regex: search, $options: "i" } },
			];
		}

		const sortObj = {};
		sortObj[sortBy] = sortOrder === "asc" ? 1 : -1;

		const [transactions, totalCount, summaryStats] = await Promise.all([
			BonusTransaction.find(searchQuery)
				.sort(sortObj)
				.skip(skip)
				.limit(parsedLimit)
				.lean(),
			BonusTransaction.countDocuments(searchQuery),
			BonusTransaction.aggregate([
				{ $match: searchQuery },
				{
					$group: {
						_id: null,
						totalTransactions: { $sum: 1 },
						totalNTE: { $sum: "$amountNTE" },
						totalUSD: { $sum: "$amountUSD" },
						walletTransactions: {
							$sum: { $cond: [{ $eq: ["$transactionType", "wallet"] }, 1, 0] },
						},
						pendingCount: {
							$sum: { $cond: [{ $eq: ["$status", "pending"] }, 1, 0] },
						},
						completedCount: {
							$sum: { $cond: [{ $eq: ["$status", "completed"] }, 1, 0] },
						},
						failedCount: {
							$sum: { $cond: [{ $eq: ["$status", "failed"] }, 1, 0] },
						},
					},
				},
			]),
		]);

		const summary =
			summaryStats.length > 0
				? {
						totalTransactions: summaryStats[0].totalTransactions || 0,
						walletTransactions: summaryStats[0].walletTransactions || 0,
						totalNTE: summaryStats[0].totalNTE || 0,
						totalUSD: summaryStats[0].totalUSD || 0,
						pendingCount: summaryStats[0].pendingCount || 0,
						completedCount: summaryStats[0].completedCount || 0,
						failedCount: summaryStats[0].failedCount || 0,
					}
				: {
						totalTransactions: 0,
						walletTransactions: 0,
						totalNTE: 0,
						totalUSD: 0,
						pendingCount: 0,
						completedCount: 0,
						failedCount: 0,
					};

		const totalPages = Math.ceil(totalCount / parsedLimit);

		return {
			transactions,
			summary,
			pagination: {
				total: totalCount,
				page: parsedPage,
				limit: parsedLimit,
				totalPages,
				hasNext: parsedPage < totalPages,
				hasPrev: parsedPage > 1,
			},
			filters: {
				username: username || null,
				bonusType: bonusType || null,
				status: status || null,
				productId: productId || null,
				startDate: startDate || null,
				endDate: endDate || null,
				search: search || null,
			},
		};
	},
};

module.exports = { bonusTransactionService };
