const mongoose = require("mongoose");

const adminPriorityChangeSchema = new mongoose.Schema(
	{
		adminUsername: {
			type: String,
			required: true,
			trim: true,
			index: true,
		},
		adminUserId: {
			type: String,
			required: true,
			trim: true,
			index: true,
		},
		claimRequestId: {
			type: mongoose.Schema.Types.ObjectId,
			required: true,
			ref: "ClaimRequest",
			index: true,
		},
		priorityBefore: {
			type: Number,
			required: true,
		},
		priorityAfter: {
			type: Number,
			required: true,
		},
		priorityChange: {
			type: Number,
			required: true,
		},
		reason: {
			type: String,
			trim: true,
			required: true,
		},
		ipAddress: {
			type: String,
			trim: true,
		},
		userAgent: {
			type: String,
			trim: true,
		},
	},
	{
		timestamps: true,
		collection: "admin_priority_changes",
	},
);

adminPriorityChangeSchema.index({ adminUsername: 1, createdAt: -1 });
adminPriorityChangeSchema.index({ createdAt: -1 });

adminPriorityChangeSchema.statics.getDailyChangeCount = async function (
	adminUsername,
) {
	const last24Hours = new Date(Date.now() - 24 * 60 * 60 * 1000);

	return await this.countDocuments({
		adminUsername,
		createdAt: { $gte: last24Hours },
	});
};

adminPriorityChangeSchema.statics.getDailyPriorityBoost = async function (
	adminUsername,
) {
	const last24Hours = new Date(Date.now() - 24 * 60 * 60 * 1000);

	const result = await this.aggregate([
		{
			$match: {
				adminUsername,
				createdAt: { $gte: last24Hours },
			},
		},
		{
			$group: {
				_id: null,
				totalBoost: { $sum: { $max: [0, "$priorityChange"] } },
				changeCount: { $sum: 1 },
			},
		},
	]);

	return result[0] || { totalBoost: 0, changeCount: 0 };
};

const AdminPriorityChange = mongoose.model(
	"AdminPriorityChange",
	adminPriorityChangeSchema,
);

module.exports = { AdminPriorityChange };
