const mongoose = require("mongoose");

const auditLogSchema = new mongoose.Schema(
	{
		timestamp: {
			type: Date,
			required: true,
			index: true,
			default: Date.now,
		},
		action: {
			type: String,
			required: true,
			index: true,
			enum: [
				"CLAIM_APPROVAL",
				"CLAIM_REJECTION",
				"MASTER_PASSWORD_VERIFICATION",
				"WITHDRAWAL_PROCESSING",
				"SECURITY_VIOLATION",
				"BALANCE_ADJUSTMENT",
				"ADMIN_ACTION",
				"WITHDRAWAL_ROLLBACK",
				"SYSTEM_EVENT",
			],
		},
		severity: {
			type: String,
			required: true,
			enum: ["LOW", "MEDIUM", "HIGH", "CRITICAL"],
			index: true,
			default: "MEDIUM",
		},
		userId: {
			type: String,
			index: true,
		},
		username: {
			type: String,
			index: true,
		},
		adminUserId: {
			type: String,
			index: true,
		},
		adminUsername: {
			type: String,
			index: true,
		},
		ipAddress: {
			type: String,
			index: true,
		},
		userAgent: {
			type: String,
		},
		details: {
			type: mongoose.Schema.Types.Mixed,
		},
		metadata: {
			type: mongoose.Schema.Types.Mixed,
		},
		result: {
			success: Boolean,
			error: String,
		},
		previousHash: {
			type: String,
			default: null,
		},
		recordHash: {
			type: String,
			default: function () {
				const crypto = require("crypto");
				return crypto.randomBytes(16).toString("hex");
			},
		},
		claimRequestId: {
			type: String,
			index: true,
		},
		withdrawalRequestId: {
			type: String,
			index: true,
		},
		transactionHash: {
			type: String,
			index: true,
		},
		amountNTE: {
			type: Number,
		},
		amountUSD: {
			type: Number,
		},
	},
	{
		timestamps: true,
		collection: "auditLogs",
		capped: { size: 100000000, max: 1000000 },
	},
);

auditLogSchema.index({ timestamp: -1 });
auditLogSchema.index({ username: 1, timestamp: -1 });
auditLogSchema.index({ adminUsername: 1, timestamp: -1 });
auditLogSchema.index({ action: 1, severity: 1, timestamp: -1 });
auditLogSchema.index({ ipAddress: 1, timestamp: -1 });
auditLogSchema.index({ claimRequestId: 1 });
auditLogSchema.index({ withdrawalRequestId: 1 });
auditLogSchema.index({ "result.success": 1, timestamp: -1 });

auditLogSchema.statics.generateHash = function (logData) {
	const crypto = require("crypto");
	const dataToHash = JSON.stringify({
		timestamp: logData.timestamp,
		action: logData.action,
		username: logData.username,
		adminUsername: logData.adminUsername,
		details: logData.details,
	});
	return crypto.createHash("sha256").update(dataToHash).digest("hex");
};

// Pre-save hook to create hash chain
auditLogSchema.pre("save", async function (next) {
	try {
		if (this.isNew) {
			// Ensure timestamp is set
			if (!this.timestamp) {
				this.timestamp = new Date();
			}

			// Get previous log entry
			const previousLog = await this.constructor
				.findOne({})
				.sort({ timestamp: -1 })
				.select("recordHash");

			this.previousHash = previousLog ? previousLog.recordHash : null;

			// Generate hash for this record
			const dataToHash = {
				timestamp: this.timestamp,
				action: this.action,
				username: this.username,
				adminUsername: this.adminUsername,
				details: this.details,
			};
			const crypto = require("crypto");
			this.recordHash = crypto
				.createHash("sha256")
				.update(JSON.stringify(dataToHash))
				.digest("hex");
		}
		next();
	} catch (error) {
		console.error("Pre-save hook error:", error);
		next(error);
	}
});

const AuditLog = mongoose.model("AuditLog", auditLogSchema);

module.exports = { AuditLog };
