const { logger } = require("../../core/logger/logger");
const { AuditLog } = require("./auditLog.model");

const auditLogService = {
	/**
	 * Log claim approval attempt
	 */
	async logClaimApproval(data) {
		const {
			claimRequestId,
			adminUsername,
			adminUserId,
			username,
			amountNTE,
			usdValue,
			success,
			error,
			transactionHash,
			clientIP,
			userAgent,
		} = data;

		const logEntry = {
			timestamp: new Date(),
			action: "CLAIM_APPROVAL",
			severity: success ? "MEDIUM" : "HIGH",
			userId: data.userId || null,
			username,
			adminUserId,
			adminUsername,
			ipAddress: clientIP,
			userAgent,
			claimRequestId,
			transactionHash: transactionHash || null,
			amountNTE,
			amountUSD: usdValue,
			details: {
				amountNTE,
				usdValue,
			},
			result: {
				success,
				error: error || null,
			},
		};

		// Persist to database
		try {
			await AuditLog.create(logEntry);
		} catch (dbError) {
			logger.error("Failed to persist audit log to database:", dbError);
		}

		// Also log to console for real-time monitoring
		logger.info("AUDIT | Claim Approval", logEntry);

		return logEntry;
	},

	/**
	 * Log claim rejection attempt
	 */
	async logClaimRejection(data) {
		const {
			claimRequestId,
			adminUsername,
			adminUserId,
			username,
			amountNTE,
			rejectionReason,
			success,
			error,
			clientIP,
			userAgent,
		} = data;

		const logEntry = {
			timestamp: new Date(),
			action: "CLAIM_REJECTION",
			severity: "MEDIUM",
			userId: data.userId || null,
			username,
			adminUserId,
			adminUsername,
			ipAddress: clientIP,
			userAgent,
			claimRequestId,
			amountNTE,
			details: {
				amountNTE,
				rejectionReason,
			},
			result: {
				success,
				error: error || null,
			},
		};

		// Persist to database
		try {
			await AuditLog.create(logEntry);
		} catch (dbError) {
			logger.error("Failed to persist audit log to database:", dbError);
		}

		logger.info("AUDIT | Claim Rejection", logEntry);

		return logEntry;
	},

	/**
	 * Log master password verification attempt
	 */
	async logMasterPasswordAttempt(data) {
		const { adminUsername, adminUserId, action, success, clientIP, userAgent } =
			data;

		const logEntry = {
			timestamp: new Date(),
			action: "MASTER_PASSWORD_VERIFICATION",
			severity: success ? "LOW" : "HIGH",
			adminUserId,
			adminUsername,
			ipAddress: clientIP,
			userAgent,
			details: {
				action,
			},
			result: {
				success,
			},
		};

		// Persist to database
		try {
			await AuditLog.create(logEntry);
		} catch (dbError) {
			logger.error("Failed to persist audit log to database:", dbError);
		}

		if (!success) {
			logger.warn("SECURITY | Failed master password attempt", logEntry);
		} else {
			logger.info("AUDIT | Master password verified", logEntry);
		}

		return logEntry;
	},

	/**
	 * Log withdrawal processing
	 */
	async logWithdrawalProcessing(data) {
		const {
			withdrawalRequestId,
			claimRequestId,
			username,
			amountNTE,
			toAddress,
			status,
			transactionHash,
			blockNumber,
			error,
		} = data;

		const logEntry = {
			timestamp: new Date(),
			action: "WITHDRAWAL_PROCESSING",
			severity: error ? "HIGH" : "MEDIUM",
			username,
			withdrawalRequestId,
			claimRequestId: claimRequestId || null,
			transactionHash: transactionHash || null,
			amountNTE,
			details: {
				amountNTE,
				toAddress,
				status,
				blockNumber: blockNumber || null,
			},
			result: {
				success: !error,
				error: error || null,
			},
		};

		// Persist to database
		try {
			await AuditLog.create(logEntry);
		} catch (dbError) {
			logger.error("Failed to persist audit log to database:", dbError);
		}

		logger.info("AUDIT | Withdrawal Processing", logEntry);

		return logEntry;
	},

	/**
	 * Log security violation
	 */
	async logSecurityViolation(data) {
		const {
			type,
			username,
			adminUsername,
			description,
			severity,
			clientIP,
			userAgent,
			metadata,
		} = data;

		const logEntry = {
			timestamp: new Date(),
			action: "SECURITY_VIOLATION",
			severity: severity || "HIGH",
			username: username || null,
			adminUsername: adminUsername || null,
			ipAddress: clientIP,
			userAgent,
			details: {
				type,
				description,
			},
			metadata: metadata || {},
		};

		// Persist to database
		try {
			await AuditLog.create(logEntry);
		} catch (dbError) {
			logger.error("Failed to persist audit log to database:", dbError);
		}

		logger.error("SECURITY VIOLATION", logEntry);

		return logEntry;
	},

	/**
	 * Log admin action
	 */
	async logAdminAction(data) {
		const {
			adminUsername,
			adminUserId,
			action,
			description,
			targetId,
			severity,
			clientIP,
			userAgent,
		} = data;

		const logEntry = {
			timestamp: new Date(),
			action: "ADMIN_ACTION",
			severity: severity || "MEDIUM",
			adminUsername,
			adminUserId: adminUserId || null,
			ipAddress: clientIP || null,
			userAgent: userAgent || null,
			details: {
				actionType: action,
				description,
				targetId: targetId || null,
			},
		};

		// Persist to database
		try {
			await AuditLog.create(logEntry);
		} catch (dbError) {
			logger.error("Failed to persist admin action log to database:", dbError);
		}

		logger.info("AUDIT | Admin Action", logEntry);

		return logEntry;
	},
};

module.exports = { auditLogService };
