const { claimRequestService } = require("./claimRequest.service");
const { ClaimRequest } = require("./claimRequest.model");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");
const { User } = require("../users/user.model");
const { logger } = require("../../core/logger/logger");
const { auditLogService } = require("./auditLog.service");

let queueControl = null;
try {
	queueControl = require("../../workers/job.worker");
} catch (err) {
	logger.warn("Queue control not available - worker may not be loaded");
}

const claimRequestController = {
	async validateClaimRequest(req, res, next) {
		try {
			const { username, walletAddress, amountNTE, withdrawalType, password } =
				req.body;

			// Basic validation
			if (!username || !walletAddress || !amountNTE) {
				throw new ApiError(400, "Missing required fields");
			}

			// Ensure user is validating their own request
			if (req.user.username !== username) {
				logger.warn(
					`Unauthorized claim validation attempt by ${req.user.username} for ${username}`,
				);
				await User.updateOne(
					{ username: req.user.username },
					{
						$set: {
							isActive: false,
							isBlocked: true,
							blockedAt: new Date(),
							blockedBy: "system",
							blockReason: "Attempted unauthorized claim request.",
						},
					},
				);
				throw new ApiError(
					403,
					"You are not authorized to validate claim requests for other users.",
				);
			}

			const result = await claimRequestService.validateClaimRequest({
				username,
				walletAddress,
				amountNTE,
				withdrawalType,
				password,
				clientIP: req.clientIp || req.ip,
				userAgent: req.headers["user-agent"],
				authUser: req.user,
			});

			// Simplify response for users
			const userResponse = {
				checks: result.checks,
				canProceed: result.canProceed,
				message: result.canProceed
					? "All validation checks passed - you can proceed with claim"
					: result.blockingIssues.join("; "),
				warnings: result.warnings,
			};

			res.json(new ApiResponse(200, userResponse, "Validation complete"));
		} catch (err) {
			next(err);
		}
	},

	async getQueueStatus(req, res, next) {
		try {
			const { claimRequestId } = req.params;

			if (!claimRequestId) {
				throw new ApiError(400, "Claim request ID is required");
			}

			const claimRequest = await claimRequestService.getClaimRequestById(
				claimRequestId,
			);

			if (!claimRequest) {
				throw new ApiError(404, "Claim request not found");
			}

			// Ensure user can only check their own claim status
			if (
				req.user.username !== claimRequest.username &&
				req.user.role !== "admin"
			) {
				throw new ApiError(
					403,
					"You are not authorized to view this claim request",
				);
			}

			// Calculate current queue position if still pending
			let currentQueuePosition = claimRequest.queuePosition;
			let updatedEstimatedTime = claimRequest.estimatedApprovalTime;

			if (claimRequest.status === "pending" && claimRequest.queuedForApproval) {
				const claimsAhead = await claimRequestService.getClaimsAheadInQueue(
					claimRequest.queuedAt,
				);
				currentQueuePosition = claimsAhead + 1;

				const CLAIM_MIN_WAIT_TIME_MS = 60 * 1000; // 1 minute initial wait
				const CLAIM_APPROVAL_INTERVAL_MS = 30 * 1000; // 30 seconds between approvals
				updatedEstimatedTime = new Date(
					Date.now() +
						CLAIM_MIN_WAIT_TIME_MS +
						(currentQueuePosition - 1) * CLAIM_APPROVAL_INTERVAL_MS,
				);
			}

			// Return simplified response for regular users, detailed for admins
			const responseData =
				req.user.role === "admin"
					? {
							claimRequestId: claimRequest._id,
							status: claimRequest.status,
							queuedForApproval: claimRequest.queuedForApproval,
							queuePosition: currentQueuePosition,
							estimatedApprovalTime: updatedEstimatedTime,
							estimatedWaitMinutes: currentQueuePosition
								? Math.ceil(1 + (currentQueuePosition - 1) * 0.5)
								: 0,
							requestedAt: claimRequest.requestedAt,
							processedAt: claimRequest.processedAt,
							amountNTE: claimRequest.amountNTE,
					  }
					: {
							status: claimRequest.status,
							amountNTE: claimRequest.amountNTE,
							queuePosition: currentQueuePosition,
							estimatedWaitMinutes: currentQueuePosition
								? Math.ceil(1 + (currentQueuePosition - 1) * 0.5)
								: 0,
							requestedAt: claimRequest.requestedAt,
					  };

			res.json(
				new ApiResponse(
					200,
					responseData,
					"Queue status retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	async createClaimRequest(req, res, next) {
		try {
			const { username, walletAddress, amountNTE, withdrawalType, password } =
				req.body;

			// Validation
			if (!username || typeof username !== "string" || username.trim() === "") {
				throw new ApiError(400, "Valid username is required");
			}

			if (
				!walletAddress ||
				typeof walletAddress !== "string" ||
				walletAddress.trim() === ""
			) {
				throw new ApiError(400, "Valid wallet address is required");
			}

			if (!amountNTE || typeof amountNTE !== "number" || amountNTE <= 0) {
				throw new ApiError(400, "Valid amount is required");
			}

			if (!password || typeof password !== "string" || password.trim() === "") {
				throw new ApiError(400, "Password is required");
			}

			// Validate minimum claim amount
			if (amountNTE < 1) {
				throw new ApiError(400, "Minimum claim amount is 1 NTE");
			}

			const result = await claimRequestService.createClaimRequest({
				username,
				walletAddress,
				amountNTE,
				withdrawalType,
				password,
				clientIP: req.clientIp || req.ip,
				userAgent: req.headers["user-agent"],
				authUser: req.user,
			});

			// Simplify response for users
			const userResponse = {
				claimRequestId: result.claimRequestId,
				status: result.status,
				amountNTE: result.amountNTE,
				queuePosition: result.queuePosition,
				estimatedWaitMinutes: result.estimatedWaitMinutes,
				message: result.message,
			};

			res
				.status(201)
				.json(
					new ApiResponse(
						201,
						userResponse,
						"Claim request submitted successfully",
					),
				);
		} catch (err) {
			next(err);
		}
	},

	async verifyClaimRequest(req, res, next) {
		try {
			const { claimRequestId } = req.query;

			if (!claimRequestId) {
				throw new ApiError(400, "Claim request ID is required");
			}

			const result = await claimRequestService.verifyClaimRequest({
				claimRequestId,
				authUser: req.user,
			});

			res.json(
				new ApiResponse(
					200,
					result,
					result.isLegit
						? "Claim request verification passed"
						: "Claim request verification failed",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	async getClaimRequests(req, res, next) {
		try {
			const { page, limit, status, username, startDate, endDate } = req.query;

			const result = await claimRequestService.getClaimRequests({
				page: parseInt(page) || 1,
				limit: parseInt(limit) || 50,
				status,
				username,
				startDate,
				endDate,
			});

			res.json(
				new ApiResponse(200, result, "Claim requests retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	async approveClaimRequest(req, res, next) {
		try {
			const { claimRequestId, masterPassword, notes } = req.body;

			// Validation
			if (!claimRequestId) {
				throw new ApiError(400, "Claim request ID is required");
			}

			if (
				!masterPassword ||
				typeof masterPassword !== "string" ||
				masterPassword.trim().length < 8
			) {
				// Log failed password attempt
				await auditLogService.logMasterPasswordAttempt({
					adminUsername: req.user.username,
					adminUserId: req.user.id,
					action: "claim_approval",
					success: false,
					clientIP: req.clientIp || req.ip,
					userAgent: req.headers["user-agent"],
				});

				throw new ApiError(400, "Valid master password is required");
			}

			const result = await claimRequestService.approveClaimRequest({
				claimRequestId,
				masterPassword,
				notes,
				adminUsername: req.user.username,
				adminUserId: req.user.id,
				clientIP: req.clientIp || req.ip,
				userAgent: req.headers["user-agent"],
			});

			res.json(
				new ApiResponse(
					200,
					result,
					result.success
						? "Claim request approved and withdrawal processed successfully"
						: "Claim approved but withdrawal failed",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	async rejectClaimRequest(req, res, next) {
		try {
			const { claimRequestId, masterPassword, rejectionReason, notes } =
				req.body;

			// Validation
			if (!claimRequestId) {
				throw new ApiError(400, "Claim request ID is required");
			}

			if (
				!masterPassword ||
				typeof masterPassword !== "string" ||
				masterPassword.trim().length < 8
			) {
				// Log failed password attempt
				await auditLogService.logMasterPasswordAttempt({
					adminUsername: req.user.username,
					adminUserId: req.user.id,
					action: "claim_rejection",
					success: false,
					clientIP: req.clientIp || req.ip,
					userAgent: req.headers["user-agent"],
				});

				throw new ApiError(400, "Valid master password is required");
			}

			if (
				!rejectionReason ||
				typeof rejectionReason !== "string" ||
				rejectionReason.trim().length < 10
			) {
				throw new ApiError(
					400,
					"Rejection reason must be at least 10 characters",
				);
			}

			const result = await claimRequestService.rejectClaimRequest({
				claimRequestId,
				masterPassword,
				rejectionReason,
				notes,
				adminUsername: req.user.username,
				adminUserId: req.user.id,
				clientIP: req.clientIp || req.ip,
				userAgent: req.headers["user-agent"],
			});

			res.json(
				new ApiResponse(200, result, "Claim request rejected successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// ==================== USER ENDPOINTS ====================

	async getUserClaims(req, res, next) {
		try {
			const { page, limit, status } = req.query;

			const result = await claimRequestService.getUserClaims(
				req.user.username,
				{
					page: parseInt(page) || 1,
					limit: parseInt(limit) || 20,
					status,
				},
			);

			const simplifiedClaims = result.claims.map((claim) => ({
				_id: claim._id,
				amountNTE: claim.amountNTE,
				usdValue: claim.usdValue,
				status: claim.status,
				requestedAt: claim.requestedAt,
				processedAt: claim.processedAt,
				queuePosition: claim.currentQueuePosition || claim.queuePosition,
				estimatedWaitMinutes: claim.estimatedWaitMinutes,
				queueStatus: claim.queueStatus,
				// Rejection details
				rejectionReason: claim.rejectionReason,
				rejectedAt: claim.rejectedAt,
				// Approval details
				approvedAt: claim.approvedAt,
				// Auto-approval tracking
				autoApprovalError: claim.autoApprovalError,
				lastAutoApprovalAttempt: claim.lastAutoApprovalAttempt,
				// Transaction details
				transactionHash: claim.transactionHash,
				blockNumber: claim.blockNumber,
				withdrawalError: claim.withdrawalError,
				// Additional info
				notes: claim.notes,
				withdrawalType: claim.withdrawalType,
			}));

			res.json(
				new ApiResponse(
					200,
					{
						claims: simplifiedClaims,
						pagination: result.pagination,
					},
					"Your claims retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	async getQueueStats(req, res, next) {
		try {
			// Get stats from service
			const dbStats = await claimRequestService.getQueueStats();

			// Get runtime queue status from worker
			let workerStatus = null;
			if (queueControl && queueControl.getQueueStatus) {
				workerStatus = await queueControl.getQueueStatus();
			}

			res.json(
				new ApiResponse(
					200,
					{
						database: dbStats,
						worker: workerStatus,
						timestamp: new Date(),
					},
					"Queue statistics retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	async getPendingQueue(req, res, next) {
		try {
			const { page, limit, sortBy, sortOrder } = req.query;

			const result = await claimRequestService.getPendingQueue({
				page: parseInt(page) || 1,
				limit: parseInt(limit) || 50,
				sortBy: sortBy || "queuedAt",
				sortOrder: sortOrder === "desc" ? -1 : 1,
			});

			res.json(
				new ApiResponse(200, result, "Pending queue retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	async getQueueHistory(req, res, next) {
		try {
			const { page, limit, startDate, endDate, status } = req.query;

			const result = await claimRequestService.getQueueHistory({
				page: parseInt(page) || 1,
				limit: parseInt(limit) || 50,
				startDate,
				endDate,
				status,
			});

			res.json(
				new ApiResponse(200, result, "Queue history retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	async getFailedClaims(req, res, next) {
		try {
			const { page, limit } = req.query;

			const result = await claimRequestService.getFailedClaims({
				page: parseInt(page) || 1,
				limit: parseInt(limit) || 50,
			});

			res.json(
				new ApiResponse(200, result, "Failed claims retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	async getManualReviewClaims(req, res, next) {
		try {
			const { page, limit } = req.query;

			const result = await claimRequestService.getManualReviewClaims({
				page: parseInt(page) || 1,
				limit: parseInt(limit) || 50,
			});

			res.json(
				new ApiResponse(
					200,
					result,
					"Manual review claims retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	async retryFailedClaim(req, res, next) {
		try {
			const { claimRequestId } = req.params;

			if (!claimRequestId) {
				throw new ApiError(400, "Claim request ID is required");
			}

			const result = await claimRequestService.retryFailedClaim(
				claimRequestId,
				req.user.username,
			);

			res.json(
				new ApiResponse(200, result, "Claim queued for retry successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	async skipClaim(req, res, next) {
		try {
			const { claimRequestId } = req.params;
			const { reason } = req.body;

			if (!claimRequestId) {
				throw new ApiError(400, "Claim request ID is required");
			}

			if (!reason || reason.trim().length < 5) {
				throw new ApiError(400, "Reason is required (min 5 characters)");
			}

			const result = await claimRequestService.skipClaim(
				claimRequestId,
				req.user.username,
				reason,
			);

			res.json(
				new ApiResponse(200, result, "Claim skipped from auto-approval"),
			);
		} catch (err) {
			next(err);
		}
	},

	async prioritizeClaim(req, res, next) {
		try {
			const { claimRequestId } = req.params;
			const { priority } = req.body;

			if (!claimRequestId) {
				throw new ApiError(400, "Claim request ID is required");
			}

			const priorityValue = parseInt(priority) || 10;
			if (priorityValue < 0 || priorityValue > 100) {
				throw new ApiError(400, "Priority must be between 0 and 100");
			}

			const result = await claimRequestService.prioritizeClaim(
				claimRequestId,
				req.user.username,
				priorityValue,
			);

			res.json(
				new ApiResponse(200, result, "Claim priority updated successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	async markForManualReview(req, res, next) {
		try {
			const { claimRequestId } = req.params;
			const { reason } = req.body;

			if (!claimRequestId) {
				throw new ApiError(400, "Claim request ID is required");
			}

			if (!reason || reason.trim().length < 5) {
				throw new ApiError(400, "Reason is required (min 5 characters)");
			}

			const result = await claimRequestService.markForManualReview(
				claimRequestId,
				req.user.username,
				reason,
			);

			res.json(new ApiResponse(200, result, "Claim marked for manual review"));
		} catch (err) {
			next(err);
		}
	},

	async clearManualReview(req, res, next) {
		try {
			const { claimRequestId } = req.params;

			if (!claimRequestId) {
				throw new ApiError(400, "Claim request ID is required");
			}

			const result = await claimRequestService.clearManualReview(
				claimRequestId,
				req.user.username,
			);

			res.json(
				new ApiResponse(200, result, "Manual review cleared successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	async pauseQueue(req, res, next) {
		try {
			const { reason } = req.body;

			if (!reason || reason.trim().length < 5) {
				throw new ApiError(400, "Reason is required (min 5 characters)");
			}

			if (!queueControl || !queueControl.pauseClaimQueue) {
				throw new ApiError(500, "Queue control not available");
			}

			const result = queueControl.pauseClaimQueue(reason, req.user.username);

			// Log audit
			await auditLogService.logAdminAction({
				adminUsername: req.user.username,
				action: "PAUSE_CLAIM_QUEUE",
				description: reason,
			});

			res.json(new ApiResponse(200, result, "Claim queue paused successfully"));
		} catch (err) {
			next(err);
		}
	},

	async resumeQueue(req, res, next) {
		try {
			if (!queueControl || !queueControl.resumeClaimQueue) {
				throw new ApiError(500, "Queue control not available");
			}

			const result = queueControl.resumeClaimQueue(req.user.username);

			// Log audit
			await auditLogService.logAdminAction({
				adminUsername: req.user.username,
				action: "RESUME_CLAIM_QUEUE",
				description: `Queue resumed. Previous reason: ${result.previousReason}`,
			});

			res.json(
				new ApiResponse(200, result, "Claim queue resumed successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	async refreshQueuePositions(req, res, next) {
		try {
			const result = await claimRequestService.refreshQueuePositions();

			res.json(
				new ApiResponse(200, result, "Queue positions refreshed successfully"),
			);
		} catch (err) {
			next(err);
		}
	},
};

module.exports = claimRequestController;
