const mongoose = require("mongoose");

const claimRequestSchema = new mongoose.Schema(
	{
		userId: {
			type: String,
			required: true,
			trim: true,
			index: true,
		},
		username: {
			type: String,
			required: true,
			trim: true,
			index: true,
		},
		email: {
			type: String,
			trim: true,
			default: null,
		},
		walletAddress: {
			type: String,
			required: true,
			trim: true,
		},
		amountNTE: {
			type: Number,
			required: true,
		},
		usdValue: {
			type: Number,
			required: true,
		},
		ntePrice: {
			type: Number,
			required: true,
		},
		withdrawalType: {
			type: String,
			required: true,
			trim: true,
			default: "auto",
		},
		status: {
			type: String,
			required: true,
			trim: true,
			index: true,
			default: "pending",
		},
		requestedAt: {
			type: Date,
			required: true,
			index: true,
			default: Date.now,
		},
		processedAt: {
			type: Date,
			default: null,
		},
		transactionHash: {
			type: mongoose.Schema.Types.Mixed,
			default: null,
		},
		walletBalanceBefore: {
			type: Number,
			required: true,
		},
		balanceBreakdown: {
			core: { type: Number, default: 0 },
			elite: { type: Number, default: 0 },
			meta_pulse: { type: Number, default: 0 },
			others: { type: Number, default: 0 },
			stake: { type: Number, default: 0 },
			total: { type: Number, default: 0 },
		},
		availableBalanceBreakdown: {
			core: { type: Number, default: 0 },
			elite: { type: Number, default: 0 },
			meta_pulse: { type: Number, default: 0 },
			others: { type: Number, default: 0 },
			stake: { type: Number, default: 0 },
			total: { type: Number, default: 0 },
		},
		flaggedTransactions: {
			count: { type: Number, default: 0 },
			total: { type: Number, default: 0 },
			byCategory: {
				core: { type: Number, default: 0 },
				elite: { type: Number, default: 0 },
				meta_pulse: { type: Number, default: 0 },
				others: { type: Number, default: 0 },
				stake: { type: Number, default: 0 },
				total: { type: Number, default: 0 },
			},
		},
		withdrawalRequestId: {
			type: mongoose.Schema.Types.Mixed,
			default: null,
		},
		approvedAt: {
			type: Date,
			default: null,
		},
		approvedBy: {
			type: String,
			trim: true,
			default: null,
		},
		rejectedAt: {
			type: Date,
			default: null,
		},
		rejectedBy: {
			type: String,
			trim: true,
			default: null,
		},
		rejectionReason: {
			type: String,
			trim: true,
			default: null,
		},
		withdrawalCreated: {
			type: Boolean,
			default: false,
		},
		withdrawalCreatedAt: {
			type: Date,
			default: null,
		},
		withdrawalCreatedBy: {
			type: String,
			trim: true,
			default: null,
		},
		withdrawalError: {
			type: String,
			trim: true,
			default: null,
		},
		idempotencyKey: {
			type: String,
			trim: true,
			unique: true,
			sparse: true,
			index: true,
		},
		lastManualEmailAt: {
			type: Date,
			default: null,
		},
		lastManualEmailBy: {
			type: String,
			trim: true,
			default: null,
		},
		ipAddress: {
			type: String,
			trim: true,
			default: null,
		},
		userAgent: {
			type: String,
			trim: true,
			default: null,
		},
		notes: {
			type: String,
			trim: true,
			default: null,
		},
		processingLock: {
			type: Boolean,
			default: false,
		},
		processingLockedAt: {
			type: Date,
			default: null,
		},
		processingLockedBy: {
			type: String,
			trim: true,
			default: null,
		},
		distributedLock: {
			type: String,
			trim: true,
			default: null,
			index: true,
		},
		distributedLockAcquiredAt: {
			type: Date,
			default: null,
		},
		distributedLockExpiry: {
			type: Date,
			default: null,
			index: true,
		},
		distributedLockWorker: {
			type: String,
			trim: true,
			default: null,
		},
		distributedLockHeartbeat: {
			type: Date,
			default: null,
			index: true,
		},
		blockchainVerificationPassed: {
			type: Boolean,
			default: false,
		},
		blockNumber: {
			type: Number,
			default: null,
		},
		queuedForApproval: {
			type: Boolean,
			default: false,
			index: true,
		},
		queuedAt: {
			type: Date,
			default: null,
			index: true,
		},
		queuePosition: {
			type: Number,
			default: null,
		},
		estimatedApprovalTime: {
			type: Date,
			default: null,
		},
		// Queue processing status
		queueStatus: {
			type: String,
			enum: [
				"waiting",
				"ready",
				"processing",
				"completed",
				"failed",
				"skipped",
				"manual_required",
			],
			default: "waiting",
			index: true,
		},
		// Auto-approval attempt tracking
		autoApprovalAttempts: {
			type: Number,
			default: 0,
		},
		maxAutoApprovalAttempts: {
			type: Number,
			default: 3,
		},
		lastAutoApprovalAttempt: {
			type: Date,
			default: null,
		},
		autoApprovalError: {
			type: String,
			trim: true,
			default: null,
		},
		autoApprovalErrorHistory: [
			{
				attempt: Number,
				error: String,
				timestamp: Date,
			},
		],
		// Pre-approval verification snapshot
		preApprovalVerification: {
			verified: { type: Boolean, default: false },
			verifiedAt: { type: Date, default: null },
			balanceAtVerification: { type: Number, default: null },
			riskLevel: { type: String, default: null },
			checksPassedCount: { type: Number, default: 0 },
			checksTotalCount: { type: Number, default: 0 },
			warnings: [{ type: String }],
			errors: [{ type: String }],
		},
		// Priority and scheduling
		priority: {
			type: Number,
			default: 0, // Higher = higher priority (0-100 scale)
			index: true,
			min: 0,
			max: 100,
		},
		priorityUpdatedAt: {
			type: Date,
			default: null,
		},
		priorityUpdatedBy: {
			type: String,
			default: null,
		},
		priorityHistory: [
			{
				previousPriority: Number,
				newPriority: Number,
				changedBy: String,
				changedAt: Date,
				reason: String,
			},
		],
		scheduledProcessingTime: {
			type: Date,
			default: null,
		},
		// Admin override flags
		skipAutoApproval: {
			type: Boolean,
			default: false,
		},
		requiresManualReview: {
			type: Boolean,
			default: false,
		},
		manualReviewReason: {
			type: String,
			trim: true,
			default: null,
		},
		manualReviewRequestedBy: {
			type: String,
			trim: true,
			default: null,
		},
		manualReviewRequestedAt: {
			type: Date,
			default: null,
		},
		// Queue position updates tracking
		queuePositionHistory: [
			{
				position: Number,
				estimatedTime: Date,
				updatedAt: { type: Date, default: Date.now },
			},
		],
	},
	{
		timestamps: true,
		collection: "claimRequests",
	},
);

// Compound indexes for queue operations
claimRequestSchema.index({ userId: 1, status: 1 });
claimRequestSchema.index({ username: 1, requestedAt: -1 });
claimRequestSchema.index({ status: 1, requestedAt: -1 });
claimRequestSchema.index({ queueStatus: 1, queuedAt: 1 });
claimRequestSchema.index({ queueStatus: 1, priority: -1, queuedAt: 1 });
claimRequestSchema.index({
	status: 1,
	queuedForApproval: 1,
	scheduledProcessingTime: 1,
});
claimRequestSchema.index({ idempotencyKey: 1 }, { unique: true, sparse: true });
claimRequestSchema.index({ withdrawalCreated: 1, withdrawalCreatedAt: 1 });

// Static methods for queue operations
claimRequestSchema.statics.getQueueStats = async function () {
	const stats = await this.aggregate([
		{
			$match: {
				queuedForApproval: true,
				status: "pending",
			},
		},
		{
			$group: {
				_id: "$queueStatus",
				count: { $sum: 1 },
				totalAmount: { $sum: "$amountNTE" },
				totalUSD: { $sum: "$usdValue" },
				avgWaitTime: { $avg: { $subtract: [new Date(), "$queuedAt"] } },
			},
		},
	]);

	const totalPending = await this.countDocuments({
		status: "pending",
		queuedForApproval: true,
	});

	const requiresManualCount = await this.countDocuments({
		status: "pending",
		requiresManualReview: true,
	});

	const failedAutoApproval = await this.countDocuments({
		status: "pending",
		queueStatus: "failed",
	});

	return {
		byStatus: stats.reduce((acc, s) => {
			acc[s._id] = {
				count: s.count,
				totalAmount: s.totalAmount,
				totalUSD: s.totalUSD,
				avgWaitMinutes: Math.round(s.avgWaitTime / 60000),
			};
			return acc;
		}, {}),
		totalPending,
		requiresManualCount,
		failedAutoApproval,
		lastUpdated: new Date(),
	};
};

claimRequestSchema.statics.getNextInQueue = async function () {
	const now = new Date();

	return await this.findOne({
		status: "pending",
		queuedForApproval: true,
		queueStatus: { $in: ["waiting", "ready"] },
		skipAutoApproval: { $ne: true },
		requiresManualReview: { $ne: true },
		processingLock: { $ne: true },
		$or: [
			{ scheduledProcessingTime: { $lte: now } },
			{ scheduledProcessingTime: null },
		],
		$expr: { $lt: ["$autoApprovalAttempts", "$maxAutoApprovalAttempts"] },
	})
		.sort({ priority: -1, queuedAt: 1 })
		.lean();
};

claimRequestSchema.statics.acquireProcessingLock = async function (
	claimId,
	lockedBy,
) {
	const { LOCK_TIMEOUTS } = require("../../utils/constants");
	const lockTimeout = LOCK_TIMEOUTS.CLAIM_PROCESSING_LOCK_MS;
	const staleTime = new Date(Date.now() - lockTimeout);

	return await this.findOneAndUpdate(
		{
			_id: claimId,
			status: "pending",
			$or: [
				{ processingLock: { $ne: true } },
				{ processingLockedAt: { $lt: staleTime } },
			],
		},
		{
			$set: {
				processingLock: true,
				processingLockedAt: new Date(),
				processingLockedBy: lockedBy,
				queueStatus: "processing",
			},
		},
		{ new: true },
	);
};

claimRequestSchema.statics.releaseProcessingLock = async function (claimId) {
	return await this.findByIdAndUpdate(claimId, {
		$set: {
			processingLock: false,
			processingLockedAt: null,
			processingLockedBy: null,
		},
	});
};

claimRequestSchema.statics.markAutoApprovalFailed = async function (
	claimId,
	error,
) {
	const claim = await this.findById(claimId);
	if (!claim) return null;

	const newAttemptCount = (claim.autoApprovalAttempts || 0) + 1;
	const maxAttempts = claim.maxAutoApprovalAttempts || 3;

	const updateData = {
		autoApprovalAttempts: newAttemptCount,
		lastAutoApprovalAttempt: new Date(),
		autoApprovalError: error,
		processingLock: false,
		processingLockedAt: null,
		processingLockedBy: null,
		$push: {
			autoApprovalErrorHistory: {
				attempt: newAttemptCount,
				error: error,
				timestamp: new Date(),
			},
		},
	};

	// If max attempts reached, mark for manual review
	if (newAttemptCount >= maxAttempts) {
		updateData.queueStatus = "manual_required";
		updateData.requiresManualReview = true;
		updateData.manualReviewReason = `Auto-approval failed ${newAttemptCount} times. Last error: ${error}`;
	} else {
		updateData.queueStatus = "failed";
		// Schedule retry after 5 minutes
		updateData.scheduledProcessingTime = new Date(Date.now() + 5 * 60 * 1000);
	}

	return await this.findByIdAndUpdate(claimId, updateData, { new: true });
};

claimRequestSchema.statics.updateQueuePositions = async function () {
	const pendingClaims = await this.find({
		status: "pending",
		queuedForApproval: true,
		queueStatus: { $in: ["waiting", "ready", "failed"] },
	})
		.sort({ priority: -1, queuedAt: 1 })
		.select("_id queuePosition");

	const CLAIM_MIN_WAIT_TIME_MS = 60 * 1000; // 1 minute initial wait
	const CLAIM_APPROVAL_INTERVAL_MS = 30 * 1000; // 30 seconds between approvals
	const bulkOps = pendingClaims.map((claim, index) => {
		const newPosition = index + 1;
		const estimatedTime = new Date(
			Date.now() +
				CLAIM_MIN_WAIT_TIME_MS +
				(newPosition - 1) * CLAIM_APPROVAL_INTERVAL_MS,
		);

		return {
			updateOne: {
				filter: { _id: claim._id },
				update: {
					$set: {
						queuePosition: newPosition,
						estimatedApprovalTime: estimatedTime,
					},
					$push: {
						queuePositionHistory: {
							$each: [
								{ position: newPosition, estimatedTime, updatedAt: new Date() },
							],
							$slice: -10, // Keep last 10 position updates
						},
					},
				},
			},
		};
	});

	if (bulkOps.length > 0) {
		await this.bulkWrite(bulkOps);
	}

	return pendingClaims.length;
};

const ClaimRequest = mongoose.model("ClaimRequest", claimRequestSchema);

module.exports = { ClaimRequest };
