const { ClaimRequest } = require("./claimRequest.model");
const { User } = require("../users/user.model");
const { WalletTransaction } = require("../wallets/walletTransaction.model");
const { WithdrawalRequest } = require("../withdrawals/withdrawalRequest.model");
const { AdminSystemSettings } = require("../admin/adminSystemSettings.model");
const { AdminConfig } = require("../admin/adminConfig.model");
const { walletService } = require("../wallets/wallet.service");
const { companyWalletService } = require("../wallets/companyWallet.service");
const { auditLogService } = require("./auditLog.service");
const { taskService } = require("../tasks/task.service");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const bcrypt = require("bcryptjs");
const mongoose = require("mongoose");
const { createWalletHash } = require("../users/user.service");
const { isValidWalletAddress } = require("../../utils/contractUtils");
const { getTokenValue } = require("../config/globalConfig.service");
const {
	createSessionIfSupported,
	withSession,
} = require("../../utils/transaction.utils");
const {
	CLAIM_LIMITS,
	TIME_CONSTANTS,
	CLAIM_QUEUE_TIMING,
} = require("../../utils/constants");
const crypto = require("crypto");
const systemNotificationService = require("../../utils/system-notification.util");
const generateEmailFromTemplate = require("../../templates/generateEmailFromTemplate");

// Check if queue is paused and enforce for manual approvals
const checkQueuePauseStatus = async (
	autoApproval,
	claimRequestId,
	adminUsername,
	clientIP,
	userAgent,
) => {
	if (autoApproval) return;

	const systemConfig = await AdminSystemSettings.findOne({
		type: "system_config",
	});

	if (systemConfig?.queuePaused === true) {
		if (systemConfig.allowManualOverrideDuringPause === false) {
			logger.warn(
				`Manual approval blocked - queue is paused: ${systemConfig.queuePausedReason}`,
			);

			await auditLogService.logSecurityViolation({
				type: "MANUAL_APPROVAL_DURING_PAUSE_BLOCKED",
				username: adminUsername,
				description: `Manual approval blocked due to queue pause: ${systemConfig.queuePausedReason}`,
				severity: "HIGH",
				clientIP,
				userAgent,
				metadata: { claimRequestId, queuePausedBy: systemConfig.queuePausedBy },
			});

			throw new ApiError(
				403,
				`Claim approvals are currently paused: ${
					systemConfig.queuePausedReason
				}. Paused by ${
					systemConfig.queuePausedBy
				} at ${systemConfig.queuePausedAt?.toISOString()}. Contact senior admin for emergency override.`,
			);
		} else {
			logger.warn(
				`Manual approval during pause - override allowed for ${adminUsername}`,
			);

			await auditLogService.logSecurityViolation({
				type: "MANUAL_APPROVAL_DURING_PAUSE_OVERRIDE",
				username: adminUsername,
				description: `Manual approval override during queue pause for claim ${claimRequestId}`,
				severity: "CRITICAL",
				clientIP,
				userAgent,
				metadata: {
					claimRequestId,
					queuePausedReason: systemConfig.queuePausedReason,
				},
			});
		}
	}
};

const claimRequestService = {
	async validateClaimRequest(data) {
		const {
			username,
			walletAddress,
			amountNTE,
			withdrawalType,
			clientIP,
			userAgent,
			authUser,
		} = data;

		const validation = {
			canProceed: true,
			checks: [],
			warnings: [],
			errors: [],
			blockingIssues: [],
			details: {},
			riskLevel: "LOW",
		};

		// 0. Check global withdrawal settings
		const systemConfig = await AdminSystemSettings.findOne({
			type: "system_config",
		});

		if (systemConfig && systemConfig.withdrawalsEnabled === false) {
			validation.canProceed = false;
			validation.riskLevel = "CRITICAL";
			validation.blockingIssues.push(
				"Withdrawals are globally disabled by system administrator",
			);
			validation.checks.push({
				name: "Global Withdrawal Status",
				status: "failed",
				message:
					systemConfig.withdrawalsMessage ||
					"Withdrawals are currently disabled system-wide",
			});
			return validation;
		}

		validation.checks.push({
			name: "Global Withdrawal Status",
			status: "passed",
			message: "Withdrawals are enabled globally",
		});

		// 1. Get user and check account status
		const user = await User.findOne({ username }).select("+password");

		if (!user) {
			validation.canProceed = false;
			validation.blockingIssues.push("User not found");
			validation.checks.push({
				name: "Account Status",
				status: "failed",
				message: "User account does not exist",
			});
			return validation;
		}

		// 2. Check if user is blocked
		if (user.isBlocked === true) {
			validation.canProceed = false;
			validation.riskLevel = "HIGH";
			validation.blockingIssues.push(
				user.blockReason || "Your account is blocked. Please contact support.",
			);
			validation.checks.push({
				name: "Account Status",
				status: "failed",
				message: "Account is blocked",
			});
			return validation;
		}

		validation.checks.push({
			name: "Account Status",
			status: "passed",
			message: "Account is active and not blocked",
		});

		// 3. Check if withdrawals are disabled for user
		if (user.withdrawalsDisabled === true) {
			validation.canProceed = false;
			validation.riskLevel = "HIGH";
			validation.blockingIssues.push(
				"Withdrawals are disabled for your account. Please contact support.",
			);
			validation.checks.push({
				name: "Withdrawal Permissions",
				status: "failed",
				message: "Withdrawals are disabled for this account",
			});
			return validation;
		}

		validation.checks.push({
			name: "Withdrawal Permissions",
			status: "passed",
			message: "Withdrawals are enabled for your account",
		});

		// 4. Verify wallet address matches
		const requestedWalletHash = createWalletHash(walletAddress);

		if (user.walletHash && user.walletHash !== requestedWalletHash) {
			validation.canProceed = false;
			validation.riskLevel = "CRITICAL";
			validation.blockingIssues.push(
				"Wallet address does not match your registered wallet",
			);
			validation.checks.push({
				name: "Wallet Verification",
				status: "failed",
				message: "Wallet address mismatch",
			});
			return validation;
		}

		validation.checks.push({
			name: "Wallet Verification",
			status: "passed",
			message: "Wallet address matches registered wallet",
		});

		// 5. Account age check
		const userAge = user.createdAt
			? Math.floor(
					(Date.now() - new Date(user.createdAt).getTime()) /
						TIME_CONSTANTS.ONE_DAY_MS,
				)
			: 0;

		if (userAge < CLAIM_LIMITS.MIN_ACCOUNT_AGE_DAYS) {
			validation.canProceed = false;
			validation.riskLevel = "HIGH";
			validation.blockingIssues.push(
				"Account must be at least 24 hours old to request withdrawal. Please try again tomorrow.",
			);
			validation.checks.push({
				name: "Account Age",
				status: "failed",
				message: `Account is ${userAge} day(s) old - minimum ${CLAIM_LIMITS.MIN_ACCOUNT_AGE_DAYS} day required`,
			});
			return validation;
		}

		if (userAge < CLAIM_LIMITS.NEW_ACCOUNT_WARNING_DAYS) {
			validation.warnings.push(
				`Account is ${userAge} days old - relatively new account`,
			);
			validation.checks.push({
				name: "Account Age",
				status: "warning",
				message: `Account is ${userAge} day(s) old - new account`,
			});
		} else {
			validation.checks.push({
				name: "Account Age",
				status: "passed",
				message: `Account is ${userAge} day(s) old`,
			});
		}

		// 6. Get balance breakdown with flagged transactions
		const balanceResult = await walletService.getBalanceBreakdown(
			username,
			true,
		);

		if (!balanceResult) {
			validation.canProceed = false;
			validation.blockingIssues.push("Unable to retrieve balance information");
			validation.checks.push({
				name: "Balance Verification",
				status: "failed",
				message: "Failed to get balance breakdown",
			});
			return validation;
		}

		const originalBreakdown = balanceResult.breakdown;
		const flaggedByCategory = balanceResult.flaggedInfo.flaggedByCategory;
		const availableBreakdown = balanceResult.availableBreakdown;

		validation.details.balance = {
			original: originalBreakdown,
			flagged: flaggedByCategory,
			available: availableBreakdown,
			flaggedCount: balanceResult.flaggedInfo.totalCount,
		};

		// Check balance by category
		const requestedWithdrawalType = withdrawalType || "total";
		let availableBalance = 0;
		let balanceCategory = "";

		switch (requestedWithdrawalType) {
			case "core":
				availableBalance = availableBreakdown.core;
				balanceCategory = "Core Balance";
				break;
			case "elite":
				availableBalance = availableBreakdown.elite;
				balanceCategory = "Elite Balance";
				break;
			case "stake":
				availableBalance = availableBreakdown.stake;
				balanceCategory = "Stake Balance";
				break;
			case "meta_pulse":
				availableBalance = availableBreakdown.meta_pulse;
				balanceCategory = "Meta Pulse Balance";
				break;
			case "others":
				availableBalance = availableBreakdown.others;
				balanceCategory = "Others Balance";
				break;
			default:
				availableBalance = availableBreakdown.total;
				balanceCategory = "Total Available Balance";
		}

		if (flaggedByCategory.total > 0) {
			validation.warnings.push(
				`You have ${flaggedByCategory.total.toFixed(
					6,
				)} NTE in flagged transactions that are pending refund. These amounts have been deducted from your available balance.`,
			);
			validation.checks.push({
				name: "Flagged Transactions",
				status: "warning",
				message: `${flaggedByCategory.total.toFixed(6)} NTE flagged for refund`,
			});
		}

		if (availableBalance < amountNTE || user.walletBalance < amountNTE) {
			validation.canProceed = false;
			validation.blockingIssues.push(
				`Insufficient ${balanceCategory.toLowerCase()}`,
			);
			validation.checks.push({
				name: "Balance Verification",
				status: "failed",
				message: `Requested: ${amountNTE.toFixed(
					6,
				)} NTE | Available: ${availableBalance.toFixed(6)} NTE`,
			});
			return validation;
		}

		validation.checks.push({
			name: "Balance Verification",
			status: "passed",
			message: `Sufficient ${balanceCategory.toLowerCase()}: ${availableBalance.toFixed(
				6,
			)} NTE`,
		});

		// 7. Check for duplicate pending claims
		const pendingClaims = await ClaimRequest.countDocuments({
			username,
			status: "pending",
		});

		if (pendingClaims > 0) {
			validation.canProceed = false;
			validation.blockingIssues.push(
				"You already have a pending claim request. Please wait for it to be processed.",
			);
			validation.checks.push({
				name: "Duplicate Claims",
				status: "failed",
				message: `${pendingClaims} pending claim(s) found`,
			});
			return validation;
		}

		validation.checks.push({
			name: "Duplicate Claims",
			status: "passed",
			message: "No pending claims found",
		});

		// 8. Check amount limits
		const minAmount = CLAIM_LIMITS.MIN_AMOUNT_NTE;

		if (amountNTE < minAmount) {
			validation.canProceed = false;
			validation.blockingIssues.push(
				`Minimum withdrawal amount is ${minAmount} NTE`,
			);
			validation.checks.push({
				name: "Amount Validation",
				status: "failed",
				message: `Amount ${amountNTE} is below minimum ${minAmount} NTE`,
			});
			return validation;
		}

		validation.checks.push({
			name: "Amount Validation",
			status: "passed",
			message: "Amount meets minimum requirements",
		});

		// 9. Withdrawal cooldown check (15 min for users, 5 min for admin)
		const isAdminUser = user.role === "admin";
		const cooldownMinutes = isAdminUser
			? CLAIM_LIMITS.COOLDOWN_MINUTES_ADMIN
			: CLAIM_LIMITS.COOLDOWN_MINUTES_USER;
		const cooldownTime = new Date(Date.now() - cooldownMinutes * 60 * 1000);

		if (user.lastWithdrawalAt && user.lastWithdrawalAt > cooldownTime) {
			const remainingTime = Math.ceil(
				(user.lastWithdrawalAt.getTime() - cooldownTime.getTime()) /
					(1000 * 60),
			);
			validation.canProceed = false;
			validation.blockingIssues.push(
				`Please wait ${remainingTime} minute(s) before requesting another withdrawal. Last withdrawal was at ${user.lastWithdrawalAt.toLocaleString()}.`,
			);
			validation.checks.push({
				name: "Withdrawal Cooldown",
				status: "failed",
				message: `${remainingTime} minute(s) remaining (${cooldownMinutes} min required)`,
			});
			return validation;
		}

		validation.checks.push({
			name: "Withdrawal Cooldown",
			status: "passed",
			message: `No active cooldown (${cooldownMinutes} min required)`,
		});

		// 10. Check for pending withdrawals
		const pendingWithdrawal = await WithdrawalRequest.findOne({
			username,
			status: { $in: ["pending", "processing", "validating", "transferring"] },
		});

		if (pendingWithdrawal) {
			validation.canProceed = false;
			validation.blockingIssues.push(
				"You have an active withdrawal in progress. Please wait for it to complete.",
			);
			validation.checks.push({
				name: "Concurrent Withdrawals",
				status: "failed",
				message: "Active withdrawal in progress",
			});
			return validation;
		}

		validation.checks.push({
			name: "Concurrent Withdrawals",
			status: "passed",
			message: "No active withdrawal in progress",
		});

		// 11. Check withdrawal frequency (24h limit)
		const last24Hours = new Date(Date.now() - TIME_CONSTANTS.ONE_DAY_MS);
		const recentWithdrawalsCount = await WithdrawalRequest.countDocuments({
			username,
			createdAt: { $gte: last24Hours },
			status: { $in: ["completed", "processing", "pending"] },
		});

		if (
			recentWithdrawalsCount >= CLAIM_LIMITS.MAX_WITHDRAWALS_PER_DAY &&
			!isAdminUser
		) {
			validation.canProceed = false;
			validation.blockingIssues.push(
				`Maximum ${CLAIM_LIMITS.MAX_WITHDRAWALS_PER_DAY} withdrawals per 24 hours. Please try again tomorrow.`,
			);
			validation.checks.push({
				name: "Withdrawal Frequency",
				status: "failed",
				message: `${recentWithdrawalsCount}/${CLAIM_LIMITS.MAX_WITHDRAWALS_PER_DAY} withdrawals in last 24h`,
			});
			return validation;
		}

		if (recentWithdrawalsCount >= CLAIM_LIMITS.MAX_CLAIMS_PER_DAY) {
			validation.warnings.push(
				`You have ${recentWithdrawalsCount} withdrawals in the last 24 hours`,
			);
			validation.checks.push({
				name: "Withdrawal Frequency",
				status: "warning",
				message: `${recentWithdrawalsCount}/5 withdrawals in last 24h`,
			});
		} else {
			validation.checks.push({
				name: "Withdrawal Frequency",
				status: "passed",
				message: `${recentWithdrawalsCount}/5 withdrawals in last 24h`,
			});
		}

		// 12. Check withdrawal history pattern (last 5 withdrawals)
		const last5Withdrawals = await WithdrawalRequest.find({
			username,
			status: { $in: ["completed", "failed", "rejected"] },
		})
			.sort({ createdAt: -1 })
			.limit(5)
			.lean();

		const last3Withdrawals = last5Withdrawals.slice(0, 3);
		const last3AllFailed =
			last3Withdrawals.length === 3 &&
			last3Withdrawals.every((w) => {
				const errorMsg = w.error?.toLowerCase() || "";
				return (
					(w.status === "failed" || w.status === "rejected") &&
					!errorMsg.includes("insufficient company balance") &&
					!errorMsg.includes("gas fee") &&
					!errorMsg.includes("network")
				);
			});

		if (last3AllFailed) {
			validation.canProceed = false;
			validation.riskLevel = "HIGH";
			validation.blockingIssues.push(
				"Your last 3 withdrawal attempts failed. Please contact support to resolve the issue.",
			);
			validation.checks.push({
				name: "Withdrawal History Pattern",
				status: "failed",
				message: "Last 3 consecutive attempts failed",
			});
			return validation;
		}

		const isBlockchainFailure = (withdrawal) => {
			const errorMsg = withdrawal.error?.toLowerCase() || "";
			return (
				errorMsg.includes("insufficient company balance") ||
				errorMsg.includes("gas fee") ||
				errorMsg.includes("network")
			);
		};

		const failedCount = last5Withdrawals.filter(
			(w) =>
				(w.status === "failed" || w.status === "rejected") &&
				!isBlockchainFailure(w),
		).length;

		if (last5Withdrawals.length === 5 && failedCount >= 4) {
			validation.canProceed = false;
			validation.riskLevel = "HIGH";
			validation.blockingIssues.push(
				"Too many failed withdrawal attempts. Please contact support.",
			);
			validation.checks.push({
				name: "Withdrawal History Pattern",
				status: "failed",
				message: `${failedCount}/5 recent failures`,
			});
			return validation;
		}

		if (failedCount > 0) {
			validation.warnings.push(
				`${failedCount} failed withdrawal(s) in recent history`,
			);
		}

		validation.checks.push({
			name: "Withdrawal History Pattern",
			status: failedCount > 2 ? "warning" : "passed",
			message:
				failedCount > 0
					? `${failedCount}/${last5Withdrawals.length} recent failures`
					: "No recent failures",
		});

		// 13. Large withdrawal warning
		const withdrawalPercentage =
			availableBreakdown.total > 0
				? (amountNTE / availableBreakdown.total) * 100
				: 0;

		if (withdrawalPercentage > 90) {
			validation.warnings.push(
				`You are withdrawing ${withdrawalPercentage.toFixed(
					1,
				)}% of your total available balance. Make sure you want to proceed.`,
			);
			validation.checks.push({
				name: "Withdrawal Size",
				status: "warning",
				message: `${withdrawalPercentage.toFixed(1)}% of total balance`,
			});
		} else {
			validation.checks.push({
				name: "Withdrawal Size",
				status: "passed",
				message: `${withdrawalPercentage.toFixed(1)}% of total balance`,
			});
		}

		// 14. Validate wallet address format
		if (!isValidWalletAddress(walletAddress)) {
			validation.canProceed = false;
			validation.blockingIssues.push("Invalid wallet address format");
			validation.checks.push({
				name: "Address Format",
				status: "failed",
				message: "Wallet address format is invalid",
			});
			return validation;
		}

		validation.checks.push({
			name: "Address Format",
			status: "passed",
			message: "Wallet address format is valid",
		});

		return validation;
	},

	async createClaimRequest(data) {
		const {
			username,
			walletAddress,
			amountNTE,
			withdrawalType,
			password,
			clientIP,
			userAgent,
			authUser,
		} = data;

		// Validate address format
		if (!isValidWalletAddress(walletAddress.trim())) {
			throw new ApiError(
				400,
				"Invalid wallet address format. Must be a valid Ethereum/BSC address.",
			);
		}

		// Check global withdrawal status
		const systemConfig = await AdminSystemSettings.findOne({
			type: "system_config",
		});

		if (systemConfig && systemConfig.withdrawalsEnabled === false) {
			throw new ApiError(
				403,
				systemConfig.withdrawalsMessage ||
					"Withdrawals are currently disabled system-wide",
			);
		}

		// Get user data
		const user = await User.findOne({ username: username.trim() }).select(
			"+password",
		);

		if (!user) {
			throw new ApiError(404, "User not found");
		}

		// Verify password
		if (!user.password) {
			throw new ApiError(
				400,
				"Account password not set. Please contact support.",
			);
		}

		const isPasswordValid = await bcrypt.compare(
			password.trim(),
			user.password,
		);

		if (!isPasswordValid) {
			logger.warn("Failed claim request - invalid password", {
				username: username.trim(),
				ip: clientIP,
			});

			// Audit log security violation
			await auditLogService.logSecurityViolation({
				type: "INVALID_PASSWORD_CLAIM_REQUEST",
				username: username.trim(),
				description:
					"User attempted to create claim request with invalid password",
				severity: "MEDIUM",
				clientIP,
				userAgent,
			});

			throw new ApiError(401, "Invalid password");
		}

		// Block claims for blocked users
		if (user.isBlocked === true) {
			throw new ApiError(
				403,
				user.blockReason || "Your account is blocked. Please contact support.",
			);
		}

		// Check if withdrawals are disabled for this user
		if (user.withdrawalsDisabled === true) {
			throw new ApiError(
				403,
				"Withdrawals are disabled for your account. Please contact support.",
			);
		}

		// VERIFICATION 1: Check account age
		const userAge = user.createdAt
			? Math.floor(
					(Date.now() - new Date(user.createdAt).getTime()) /
						TIME_CONSTANTS.ONE_DAY_MS,
				)
			: 0;

		if (userAge < CLAIM_LIMITS.MIN_ACCOUNT_AGE_DAYS) {
			throw new ApiError(
				403,
				"Account must be at least 24 hours old to request withdrawal. Please try again tomorrow.",
			);
		} else if (userAge < CLAIM_LIMITS.NEW_ACCOUNT_WARNING_DAYS) {
			logger.info("New account claim request", {
				username: username.trim(),
				accountAge: userAge,
			});
		}

		// VERIFICATION 2: Verify wallet hash matches registered wallet
		const requestedWalletHash = createWalletHash(walletAddress.trim());

		if (user.walletHash && user.walletHash !== requestedWalletHash) {
			logger.warn("Wallet mismatch in claim request", {
				username: username.trim(),
				ip: clientIP,
			});

			// Audit log security violation
			await auditLogService.logSecurityViolation({
				type: "WALLET_MISMATCH_CLAIM_REQUEST",
				username: username.trim(),
				description:
					"User attempted to create claim request with mismatched wallet address",
				severity: "CRITICAL",
				clientIP,
				userAgent,
				metadata: {
					requestedAddress: walletAddress.trim(),
					requestedHash: requestedWalletHash,
					storedHash: user.walletHash,
				},
			});

			throw new ApiError(
				403,
				"Wallet address does not match your registered wallet. Please contact support if you believe this is an error.",
			);
		}

		// VERIFICATION 3: Get balance breakdown with flagged transactions
		const balanceResult = await walletService.getBalanceBreakdown(
			username.trim(),
			true,
		);

		if (!balanceResult) {
			throw new ApiError(
				500,
				"Unable to retrieve balance information. Please try again later.",
			);
		}

		const originalBreakdown = balanceResult.breakdown;
		const flaggedByCategory = balanceResult.flaggedInfo.flaggedByCategory;
		const availableBreakdown = balanceResult.availableBreakdown;

		// Verify balance by category
		const requestedWithdrawalType = withdrawalType || "total";
		let availableBalance = 0;
		let balanceCategory = "";

		switch (requestedWithdrawalType) {
			case "core":
				availableBalance = availableBreakdown.core;
				balanceCategory = "Core Balance";
				break;
			case "elite":
				availableBalance = availableBreakdown.elite;
				balanceCategory = "Elite Balance";
				break;
			case "stake":
				availableBalance = availableBreakdown.stake;
				balanceCategory = "Stake Balance";
				break;
			case "meta_pulse":
				availableBalance = availableBreakdown.meta_pulse;
				balanceCategory = "Meta Pulse Balance";
				break;
			case "others":
				availableBalance = availableBreakdown.others;
				balanceCategory = "Others Balance";
				break;
			default:
				availableBalance = availableBreakdown.total;
				balanceCategory = "Total Available Balance";
		}

		// max claim per request check
		if (amountNTE > CLAIM_LIMITS.MAX_CLAIM_PER_TRANSACTION_USD) {
			throw new ApiError(
				400,
				`Maximum claim amount per request is ${CLAIM_LIMITS.MAX_CLAIM_PER_TRANSACTION_USD} NTE`,
			);
		}

		// Check if user has sufficient available balance (after flagged deductions)
		if (availableBalance < amountNTE || user.walletBalance < amountNTE) {
			logger.warn("Insufficient balance in claim request", {
				username: username.trim(),
				requested: amountNTE,
				available: availableBalance,
			});
			throw new ApiError(
				400,
				`Insufficient ${balanceCategory.toLowerCase()}. Available: ${availableBalance.toFixed(
					6,
				)} NTE`,
			);
		}

		// Check for pending claim requests
		const pendingClaim = await ClaimRequest.findOne({
			username: username.trim(),
			status: "pending",
		});

		if (pendingClaim) {
			throw new ApiError(
				400,
				"You already have a pending claim request. Please wait for it to be processed.",
			);
		}

		// VERIFICATION 4: Withdrawal cooldown check
		const isAdminUser = user.role === "admin";
		const cooldownMinutes = isAdminUser
			? CLAIM_LIMITS.COOLDOWN_MINUTES_ADMIN
			: CLAIM_LIMITS.COOLDOWN_MINUTES_USER;
		const cooldownTime = new Date(Date.now() - cooldownMinutes * 60 * 1000);

		if (user.lastWithdrawalAt && user.lastWithdrawalAt > cooldownTime) {
			const remainingTime = Math.ceil(
				(user.lastWithdrawalAt.getTime() - cooldownTime.getTime()) /
					(1000 * 60),
			);
			throw new ApiError(
				429,
				`Please wait ${remainingTime} minute(s) before requesting another withdrawal. Last withdrawal was at ${user.lastWithdrawalAt.toLocaleString()}.`,
			);
		}

		// VERIFICATION 5: Check for pending/processing withdrawals
		const pendingWithdrawal = await WithdrawalRequest.findOne({
			username: username.trim(),
			status: { $in: ["pending", "processing", "validating", "transferring"] },
		});

		if (pendingWithdrawal) {
			throw new ApiError(
				400,
				"You have an active withdrawal in progress. Please wait for it to complete before requesting another.",
			);
		}

		// VERIFICATION 6: Withdrawal frequency check (24h)
		const last24Hours = new Date(Date.now() - TIME_CONSTANTS.ONE_DAY_MS);
		const recentWithdrawalsCount = await WithdrawalRequest.countDocuments({
			username: username.trim(),
			createdAt: { $gte: last24Hours },
			status: { $in: ["completed", "processing", "pending"] },
		});

		if (
			recentWithdrawalsCount >= CLAIM_LIMITS.MAX_WITHDRAWALS_PER_DAY &&
			!isAdminUser
		) {
			throw new ApiError(
				429,
				`Maximum ${CLAIM_LIMITS.MAX_WITHDRAWALS_PER_DAY} withdrawals per 24 hours. Please try again tomorrow.`,
			);
		}

		// VERIFICATION 7: Withdrawal history pattern (last 5 withdrawals)
		const last5Withdrawals = await WithdrawalRequest.find({
			username: username.trim(),
			status: { $in: ["completed", "failed", "rejected"] },
		})
			.sort({ createdAt: -1 })
			.limit(5)
			.lean();

		// Check if last 3 consecutive attempts failed (excluding blockchain-related failures)
		const last3Withdrawals = last5Withdrawals.slice(0, 3);
		const last3AllFailed =
			last3Withdrawals.length === 3 &&
			last3Withdrawals.every((w) => {
				const errorMsg = w.error?.toLowerCase() || "";
				return (
					(w.status === "failed" || w.status === "rejected") &&
					!errorMsg.includes("insufficient company balance") &&
					!errorMsg.includes("gas fee") &&
					!errorMsg.includes("network")
				);
			});

		if (last3AllFailed) {
			throw new ApiError(
				403,
				"Your last 3 withdrawal attempts failed. Please contact support to resolve the issue before trying again.",
			);
		}

		// Check if 4 out of last 5 failed (excluding blockchain-related failures)
		const isBlockchainFailure = (withdrawal) => {
			const errorMsg = withdrawal.error?.toLowerCase() || "";
			return (
				errorMsg.includes("insufficient company balance") ||
				errorMsg.includes("gas fee") ||
				errorMsg.includes("network")
			);
		};

		const failedCount = last5Withdrawals.filter(
			(w) =>
				(w.status === "failed" || w.status === "rejected") &&
				!isBlockchainFailure(w),
		).length;

		if (last5Withdrawals.length === 5 && failedCount >= 4) {
			throw new ApiError(
				403,
				"Too many failed withdrawal attempts. Please contact support to resolve any issues.",
			);
		}

		// VERIFICATION 8: Check claim history and patterns
		const userClaimHistory = await ClaimRequest.find({
			username: username.trim(),
			status: { $in: ["approved", "rejected", "completed", "failed"] },
		})
			.sort({ createdAt: -1 })
			.limit(10)
			.lean();

		const approvedClaims = userClaimHistory.filter(
			(c) => c.status === "approved" || c.status === "completed",
		).length;
		const rejectedClaims = userClaimHistory.filter(
			(c) => c.status === "rejected",
		).length;
		const failedClaims = userClaimHistory.filter(
			(c) => c.status === "failed",
		).length;
		const totalClaims = userClaimHistory.length;

		// Check for suspicious patterns
		if (rejectedClaims + failedClaims > 5) {
			logger.warn("High failure rate in claim history", {
				username: username.trim(),
				rejectedClaims,
				failedClaims,
				totalClaims,
			});
		}

		// Log claim history
		logger.info("Claim history", {
			username: username.trim(),
			approved: approvedClaims,
			rejected: rejectedClaims,
			failed: failedClaims,
			total: totalClaims,
		});

		// VERIFICATION 9: Check claim frequency (24-hour window)
		const recentClaims = await ClaimRequest.countDocuments({
			username: username.trim(),
			createdAt: { $gte: last24Hours },
		});

		if (recentClaims >= CLAIM_LIMITS.MAX_CLAIMS_PER_DAY) {
			throw new ApiError(
				429,
				`You have already submitted ${recentClaims} claim requests today. Maximum ${CLAIM_LIMITS.MAX_CLAIMS_PER_DAY} per day. Please try again tomorrow.`,
			);
		}

		// Get current NTE price
		const currentNTEPrice = await getTokenValue();
		const usdValue = amountNTE * (currentNTEPrice || 0.001);

		// Check if amount exceeds auto-approval limit (requires manual review)
		const requiresManualReview = usdValue > CLAIM_LIMITS.MAX_AUTO_APPROVE_NTE;

		// Calculate queue position atomically using aggregation
		// Get current maximum queue position to avoid race conditions
		const maxPositionResult = await ClaimRequest.findOne({
			status: "pending",
			queuedForApproval: true,
		})
			.sort({ queuePosition: -1 })
			.select("queuePosition")
			.lean();

		const queuePosition = (maxPositionResult?.queuePosition || 0) + 1;

		const CLAIM_MIN_WAIT_TIME_MS = 60 * 1000; // 1 minute minimum from request
		const CLAIM_APPROVAL_INTERVAL_MS = CLAIM_QUEUE_TIMING.APPROVAL_INTERVAL_MS; // 30 seconds between approvals

		// First claim waits 1 minute, subsequent claims add 30s each
		const estimatedWaitTime =
			queuePosition === 1
				? CLAIM_MIN_WAIT_TIME_MS
				: CLAIM_MIN_WAIT_TIME_MS +
					(queuePosition - 1) * CLAIM_APPROVAL_INTERVAL_MS;

		const estimatedApprovalTime = new Date(Date.now() + estimatedWaitTime);

		logger.info(
			`QUEUE POSITION | Assigned position ${queuePosition} for claim`,
			{
				username: username.trim(),
				amountNTE,
				queuePosition,
				estimatedApprovalTime,
			},
		);

		// Create claim request with full queue management fields
		const claimRequest = new ClaimRequest({
			username: username.trim(),
			userId: user._id.toString(),
			email: user.email || null,
			walletAddress: walletAddress.trim(),
			amountNTE,
			usdValue,
			ntePrice: currentNTEPrice,
			withdrawalType: withdrawalType || "auto",
			status: "pending",
			walletBalanceBefore: user.walletBalance,
			balanceBreakdown: originalBreakdown,
			availableBalanceBreakdown: availableBreakdown,
			flaggedTransactions: {
				count: balanceResult.flaggedInfo.totalCount,
				total: flaggedByCategory.total,
				byCategory: flaggedByCategory,
			},
			requestedAt: new Date(),
			ipAddress: clientIP,
			userAgent: userAgent,
			approvedBy: null,
			approvedAt: null,
			rejectedBy: null,
			rejectedAt: null,
			rejectionReason: null,
			processedAt: null,
			transactionHash: null,
			notes: null,
			// Queue management fields
			queuedForApproval: true,
			queuedAt: new Date(),
			queuePosition,
			estimatedApprovalTime,
			queueStatus: "waiting",
			autoApprovalAttempts: 0,
			maxAutoApprovalAttempts: 3,
			requiresManualReview,
			manualReviewReason: requiresManualReview
				? `Amount ${amountNTE} NTE exceeds auto-approval limit of ${CLAIM_LIMITS.MAX_AUTO_APPROVE_NTE} NTE`
				: null,
			priority: 0,
			scheduledProcessingTime: estimatedApprovalTime,
			queuePositionHistory: [
				{
					position: queuePosition,
					estimatedTime: estimatedApprovalTime,
					updatedAt: new Date(),
				},
			],
		});

		await claimRequest.save();

		logger.info("Claim request created and queued", {
			claimRequestId: claimRequest._id.toString(),
			username: username.trim(),
			amountNTE,
			usdValue,
			queuePosition,
			estimatedApprovalTime,
			requiresManualReview,
		});

		// Audit log claim request creation
		await auditLogService.logWithdrawalProcessing({
			withdrawalRequestId: null,
			claimRequestId: claimRequest._id.toString(),
			username: username.trim(),
			amountNTE,
			toAddress: walletAddress.trim(),
			status: requiresManualReview
				? "claim_created_manual_review"
				: "claim_created",
			transactionHash: null,
			blockNumber: null,
			error: null,
		});

		// Build response message
		let message = `Your claim request has been queued. Position: ${queuePosition}.`;
		if (requiresManualReview) {
			message = `Your claim request has been submitted for manual review due to the amount exceeding ${CLAIM_LIMITS.MAX_AUTO_APPROVE_NTE} NTE. An administrator will review your request shortly.`;
		} else {
			message += `Estimated approval time: ${estimatedApprovalTime.toLocaleString()}`;
		}

		// Send notification email to user
		try {
			const userEmailBody = `
				<div style="color: #e1e4eb; line-height: 1.8;">
					<h2 style="color: #00e5ff; margin-bottom: 20px; font-size: 24px;">Claim Request Submitted</h2>
					
					<p style="margin-bottom: 15px;">Hello <strong>${username.trim()}</strong>,</p>
					
					<p style="margin-bottom: 20px;">Your claim request has been successfully submitted and queued for processing.</p>
					
					<div style="background: linear-gradient(135deg, rgba(0, 217, 255, 0.15), rgba(0, 217, 255, 0.05)); border-left: 4px solid #00e5ff; padding: 20px; margin: 25px 0; border-radius: 8px;">
						<h3 style="color: #00e5ff; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Request Details</h3>
						<table style="width: 100%; border-collapse: collapse;">
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Request ID:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace;">${claimRequest._id.toString()}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Amount:</td>
								<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-weight: 600; font-size: 18px;">${amountNTE.toFixed(
									6,
								)} NTE</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">USD Value:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">$${usdValue.toFixed(
									2,
								)}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Queue Position:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">#${queuePosition}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
								<td style="padding: 8px 0; color: #ffa500; text-align: right; font-weight: 600;">${
									requiresManualReview ? "Manual Review" : "Queued"
								}</td>
							</tr>
							${
								!requiresManualReview
									? `
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Estimated Approval:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">${estimatedApprovalTime.toLocaleString()}</td>
							</tr>
							`
									: ""
							}
						</table>
					</div>
					
					${
						requiresManualReview
							? `
						<div style="background: rgba(255, 165, 0, 0.1); border-left: 4px solid #ffa500; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #ffa500;"><strong>⚠️ Manual Review Required</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">Your withdrawal amount exceeds ${CLAIM_LIMITS.MAX_AUTO_APPROVE_NTE} NTE. An administrator will review and process your request shortly.</p>
						</div>
					`
							: `
						<div style="background: rgba(16, 185, 129, 0.1); border-left: 4px solid #10b981; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #10b981;"><strong>✓ Auto-Approval Queued</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">Your request will be automatically processed in approximately ${Math.ceil(
								1 + (queuePosition - 1) * 0.5,
							)} minute(s).</p>
						</div>
					`
					}
					
					<p style="margin-top: 25px; color: #9aa3b2; font-size: 14px;">You can track your claim status in your dashboard. You will receive another notification once your claim is processed.</p>
					
					<div style="text-align: center; margin-top: 30px;">
						<a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View Dashboard</a>
					</div>
				</div>
			`;

			const userEmailHtml = generateEmailFromTemplate(userEmailBody);

			systemNotificationService.sendToUser({
				username: username.trim(),
				email: user.email,
				subject: "Claim Request Submitted - NodeMeta",
				title: "Claim Request Submitted",
				body: `Your claim request for ${amountNTE.toFixed(
					6,
				)} NTE has been submitted. Queue position: ${queuePosition}`,
				html: userEmailHtml,
				pushPayload: {
					icon: "/icons/icon-192x192.png",
					badge: "/icons/badge-72x72.png",
					data: {
						url: "/dashboard/claims",
						claimRequestId: claimRequest._id.toString(),
					},
				},
				sendEmail: true,
				sendPush: true,
			});
		} catch (error) {
			logger.error("Failed to send user notification for claim request:", {
				errorMessage: error.message,
				errorStack: error.stack,
				errorName: error.name,
				username: username.trim(),
				email: user.email,
				claimRequestId: claimRequest._id.toString(),
				amountNTE,
			});
		}

		// Send notification to admins
		try {
			const adminEmailBody = `
				<div style="color: #e1e4eb; line-height: 1.8;">
					<h2 style="color: #00e5ff; margin-bottom: 20px; font-size: 24px;">New Claim Request ${
						requiresManualReview ? "- Manual Review Required" : ""
					}</h2>
					
					<p style="margin-bottom: 20px;">A new claim request has been submitted${
						requiresManualReview
							? " and requires manual review"
							: " and is queued for auto-approval"
					}.</p>
					
					<div style="background: linear-gradient(135deg, rgba(0, 217, 255, 0.15), rgba(0, 217, 255, 0.05)); border-left: 4px solid #00e5ff; padding: 20px; margin: 25px 0; border-radius: 8px;">
						<h3 style="color: #00e5ff; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Request Details</h3>
						<table style="width: 100%; border-collapse: collapse;">
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Request ID:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 12px;">${claimRequest._id.toString()}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Username:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${username.trim()}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Amount:</td>
								<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-weight: 600; font-size: 18px;">${amountNTE.toFixed(
									6,
								)} NTE</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">USD Value:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">$${usdValue.toFixed(
									2,
								)}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Wallet Address:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 11px;">${walletAddress
									.trim()
									.substring(0, 10)}...${walletAddress
									.trim()
									.substring(walletAddress.trim().length - 8)}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Queue Position:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">#${queuePosition}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Claim Type:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; text-transform: capitalize;">${
									withdrawalType || "auto"
								}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Requested At:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">${claimRequest.requestedAt.toLocaleString()}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">IP Address:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 12px;">${clientIP}</td>
							</tr>
						</table>
					</div>
					
					${
						requiresManualReview
							? `
						<div style="background: rgba(255, 165, 0, 0.15); border-left: 4px solid #ffa500; padding: 20px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #ffa500; font-size: 16px; font-weight: 600;">⚠️ Manual Review Required</p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">This claim exceeds the auto-approval limit of ${CLAIM_LIMITS.MAX_AUTO_APPROVE_NTE} NTE and requires administrator review.</p>
						</div>
					`
							: `
						<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #3b82f6;"><strong>ℹ️ Auto-Approval Queued</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">This claim will be automatically processed. Estimated approval: ${estimatedApprovalTime.toLocaleString()}</p>
						</div>
					`
					}
					
					<div style="background: linear-gradient(135deg, rgba(59, 130, 246, 0.1), rgba(59, 130, 246, 0.05)); border-left: 4px solid #3b82f6; padding: 20px; margin: 25px 0; border-radius: 8px;">
						<h3 style="color: #3b82f6; margin-top: 0; margin-bottom: 15px; font-size: 18px;">User Balance Info</h3>
						<table style="width: 100%; border-collapse: collapse;">
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Balance Before:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">${user.walletBalance.toFixed(
									6,
								)} NTE</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Available Balance:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">${availableBreakdown.total.toFixed(
									6,
								)} NTE</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Account Age:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">${userAge} days</td>
							</tr>
						</table>
					</div>
					
					<div style="text-align: center; margin-top: 30px;">
						<a href="https://node-meta.com/dashboard/admin/claim-request" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">View in Admin Panel</a>
						${
							requiresManualReview
								? `
							<a href="https://node-meta.com/dashboard/admin/claim-request" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #ffa500 0%, #cc8400 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">Review Now</a>
						`
								: ""
						}
					</div>
				</div>
			`;

			const adminEmailHtml = generateEmailFromTemplate(adminEmailBody);

			systemNotificationService.sendToAdmin({
				subject: `New Claim Request: ${amountNTE.toFixed(
					2,
				)} NTE - ${username.trim()}${
					requiresManualReview ? " [MANUAL REVIEW]" : ""
				}`,
				title: "New Claim Request",
				body: `${username.trim()} requested ${amountNTE.toFixed(
					6,
				)} NTE Claim. Queue position: ${queuePosition}${
					requiresManualReview ? " - Manual review required" : ""
				}`,
				html: adminEmailHtml,
				pushPayload: {
					icon: "/icons/icon-192x192.png",
					badge: "/icons/badge-72x72.png",
					data: {
						url: "/admin/claims",
						claimRequestId: claimRequest._id.toString(),
						requiresManualReview,
					},
					// Use urgent tag for manual review
					...(requiresManualReview && { tag: "urgent-claim" }),
				},
				sendEmail: true,
				sendPush: true,
			});
		} catch (error) {
			logger.error("Failed to send admin notification for claim request:", {
				errorMessage: error.message,
				errorStack: error.stack,
				errorName: error.name,
				username: username.trim(),
				claimRequestId: claimRequest._id.toString(),
				amountNTE,
				requiresManualReview,
			});
		}

		return {
			claimRequestId: claimRequest._id.toString(),
			status: "pending",
			amountNTE,
			usdValue,
			requestedAt: claimRequest.requestedAt,
			queuedForApproval: true,
			queuePosition,
			estimatedApprovalTime,
			estimatedWaitMinutes: Math.ceil(1 + (queuePosition - 1) * 0.5),
			requiresManualReview,
			message,
		};
	},

	async verifyClaimRequest(data) {
		const { claimRequestId, authUser } = data;

		// Validation
		if (!mongoose.Types.ObjectId.isValid(claimRequestId)) {
			throw new ApiError(400, "Valid claim request ID is required");
		}

		// Get claim request
		const claimRequest = await ClaimRequest.findById(claimRequestId);

		if (!claimRequest) {
			throw new ApiError(404, "Claim request not found");
		}

		// Check if claim request is still pending
		if (claimRequest.status !== "pending") {
			throw new ApiError(
				400,
				`Claim request status is '${claimRequest.status}' - can only verify pending claims`,
			);
		}

		// Verification results object
		const verification = {
			isLegit: true,
			checks: [],
			warnings: [],
			errors: [],
			details: {},
			riskLevel: "LOW",
			recommendations: [],
			canApprove: true,
		};

		// SECURITY CHECK: Verify claim is in pending status
		if (claimRequest.status !== "pending") {
			verification.isLegit = false;
			verification.riskLevel = "CRITICAL";
			verification.canApprove = false;
			verification.errors.push(
				`Claim is not in pending status (current: ${claimRequest.status})`,
			);
			verification.checks.push({
				name: "Claim Status Validation",
				status: "failed",
				message: `Claim status is '${claimRequest.status}' - only pending claims can be verified`,
			});
			verification.recommendations.push(
				"Claim is not pending - Do not approve",
			);
		}

		// 0. Check global withdrawal status (PRIORITY CHECK)
		const systemConfig = await AdminSystemSettings.findOne({
			type: "system_config",
		});

		if (systemConfig && systemConfig.withdrawalsEnabled === false) {
			verification.isLegit = false;
			verification.riskLevel = "CRITICAL";
			verification.errors.push(
				"Claims are globally disabled by system administrator",
			);
			verification.checks.push({
				name: "Global Claim Status",
				status: "failed",
				message:
					systemConfig.withdrawalsMessage ||
					"Claims are currently disabled system-wide",
			});
			verification.recommendations.push(
				"Claims are globally disabled - Do not approve",
			);
		} else {
			verification.checks.push({
				name: "Global Claim Status",
				status: "passed",
				message: "Claims are enabled globally",
			});
		}

		// Check claim request age (reject stale claims > 24 hours)
		const claimAge = claimRequest.createdAt
			? Math.floor(
					(Date.now() - new Date(claimRequest.createdAt).getTime()) /
						TIME_CONSTANTS.ONE_HOUR_MS,
				)
			: 0;

		if (claimAge > CLAIM_LIMITS.MAX_CLAIM_AGE_HOURS) {
			verification.warnings.push(
				`Claim request is ${claimAge} hours old - balance data may be stale`,
			);
			verification.checks.push({
				name: "Claim Request Age",
				status: "warning",
				message: `Claim is ${claimAge} hour(s) old - verify current balance`,
				details: { ageInHours: claimAge },
			});
			verification.recommendations.push(
				"Stale claim - verify user still has sufficient balance",
			);
			if (claimAge > 48) {
				verification.riskLevel = "MEDIUM";
			}
		} else {
			verification.checks.push({
				name: "Claim Request Age",
				status: "passed",
				message: `Claim is ${claimAge} hour(s) old - fresh request`,
				details: { ageInHours: claimAge },
			});
		}

		// 1. Verify user exists and get wallet hash
		const user = await User.findOne({ username: claimRequest.username });

		if (!user) {
			verification.isLegit = false;
			verification.errors.push("User not found in database");
			verification.checks.push({
				name: "User Verification",
				status: "failed",
				message: "User account does not exist",
			});
		} else {
			// Create hash from requested wallet address for comparison
			const requestedWalletHash = createWalletHash(claimRequest.walletAddress);

			verification.details.user = {
				username: user.username,
				email: user.email,
				role: user.role,
				isBlocked: user.isBlocked || false,
				withdrawalsDisabled: user.withdrawalsDisabled || false,
				createdAt: user.createdAt,
				walletHash: user.walletHash || null,
			};

			// Check if user is blocked
			if (user.isBlocked) {
				verification.isLegit = false;
				verification.riskLevel = "CRITICAL";
				verification.errors.push("User account is blocked");
				verification.checks.push({
					name: "Account Status",
					status: "failed",
					message: user.blockReason || "Account is blocked",
				});
				verification.recommendations.push("User is blocked - Do not approve");
			} else {
				verification.checks.push({
					name: "Account Status",
					status: "passed",
					message: "Account is active and not blocked",
				});
			}

			// Check if withdrawals are disabled for user
			if (user.withdrawalsDisabled) {
				verification.isLegit = false;
				verification.riskLevel = "CRITICAL";
				verification.errors.push("Claims are disabled for this user");
				verification.checks.push({
					name: "Claim Permissions",
					status: "failed",
					message: "Claims are disabled for this account",
				});
				verification.recommendations.push(
					"Claims disabled for user - Do not approve",
				);
			} else {
				verification.checks.push({
					name: "Claim Permissions",
					status: "passed",
					message: "Claims are enabled for this account",
				});
			}

			// 2. Verify wallet address matches user's wallet hash
			const walletMatches = user.walletHash === requestedWalletHash;

			if (!walletMatches) {
				verification.isLegit = false;
				verification.riskLevel = "CRITICAL";
				verification.errors.push(
					"Wallet address does not match user's registered wallet",
				);
				verification.checks.push({
					name: "Wallet Verification",
					status: "failed",
					message: "Wallet address mismatch",
					details: {
						requestedAddress: claimRequest.walletAddress,
						requestedHash: requestedWalletHash,
						storedHash: user.walletHash,
					},
				});
				verification.recommendations.push(
					"Wallet mismatch - Possible fraud attempt - Do not approve",
				);
			} else {
				verification.checks.push({
					name: "Wallet Verification",
					status: "passed",
					message: "Wallet address matches registered wallet",
					details: {
						walletAddress: claimRequest.walletAddress,
					},
				});
			}

			// 3. Get live balance breakdown with flagged transactions
			logger.info("Getting live balance breakdown", {
				username: claimRequest.username,
			});

			const balanceResult = await walletService.getBalanceBreakdown(
				claimRequest.username,
				true,
			);

			if (!balanceResult) {
				verification.isLegit = false;
				verification.errors.push("Unable to retrieve balance information");
				verification.checks.push({
					name: "Balance Retrieval",
					status: "failed",
					message: "Failed to get balance breakdown",
				});
			} else {
				const originalBreakdown = balanceResult.breakdown;
				const flaggedByCategory = balanceResult.flaggedInfo.flaggedByCategory;
				const availableBreakdown = balanceResult.availableBreakdown;

				verification.details.balance = {
					original: originalBreakdown,
					flagged: flaggedByCategory,
					available: availableBreakdown,
					flaggedCount: balanceResult.flaggedInfo.totalCount,
				};

				const negativeBalances = [];
				const balanceCategories = [
					"core",
					"elite",
					"stake",
					"meta_pulse",
					"others",
					"total",
				];

				for (const category of balanceCategories) {
					if (originalBreakdown[category] < 0) {
						negativeBalances.push({
							category: category,
							amount: originalBreakdown[category],
						});
					}
				}

				if (negativeBalances.length > 0) {
					verification.isLegit = false;
					verification.riskLevel = "CRITICAL";

					for (const negBalance of negativeBalances) {
						const categoryName = negBalance.category
							.replace("_", " ")
							.replace(/\b\w/g, (l) => l.toUpperCase());
						verification.errors.push(
							`${categoryName} has negative balance: ${negBalance.amount.toFixed(
								6,
							)} NTE - DATA INTEGRITY ISSUE`,
						);
					}

					verification.checks.push({
						name: "Negative Balance Check",
						status: "failed",
						message: `${negativeBalances.length} balance categor${
							negativeBalances.length > 1 ? "ies have" : "y has"
						} negative value(s) - CRITICAL DATA INTEGRITY ISSUE`,
						details: {
							negativeBalances: negativeBalances.map((nb) => ({
								category: nb.category,
								amount: nb.amount,
								displayName: nb.category
									.replace("_", " ")
									.replace(/\b\w/g, (l) => l.toUpperCase()),
							})),
							message:
								"User account has corrupted balance data. Investigation required before any withdrawal.",
						},
					});

					verification.recommendations.push(
						"Negative balance detected - DO NOT APPROVE. Investigate transaction history immediately.",
					);
				} else {
					verification.checks.push({
						name: "Negative Balance Check",
						status: "passed",
						message: "All balance categories are non-negative",
					});
				}

				// Verify balance by category
				const requestedWithdrawalType = claimRequest.withdrawalType || "total";
				let availableBalance = 0;
				let balanceCategory = "";

				switch (requestedWithdrawalType) {
					case "core":
						availableBalance = availableBreakdown.core;
						balanceCategory = "Core Balance";
						break;
					case "elite":
						availableBalance = availableBreakdown.elite;
						balanceCategory = "Elite Balance";
						break;
					case "stake":
						availableBalance = availableBreakdown.stake;
						balanceCategory = "Stake Balance";
						break;
					case "meta_pulse":
						availableBalance = availableBreakdown.meta_pulse;
						balanceCategory = "Meta Pulse Balance";
						break;
					case "others":
						availableBalance = availableBreakdown.others;
						balanceCategory = "Others Balance";
						break;
					default:
						availableBalance = availableBreakdown.total;
						balanceCategory = "Total Available Balance";
				}

				// Check if sufficient balance
				if (availableBalance < claimRequest.amountNTE) {
					verification.isLegit = false;
					verification.riskLevel = "HIGH";
					verification.errors.push(
						`Insufficient ${balanceCategory.toLowerCase()}`,
					);
					verification.checks.push({
						name: "Balance Verification",
						status: "failed",
						message: `Insufficient balance | Requested: ${claimRequest.amountNTE.toFixed(
							6,
						)} | Available: ${availableBalance.toFixed(6)}`,
					});
					verification.recommendations.push(
						`Insufficient balance in ${balanceCategory} - Do not approve`,
					);
				} else {
					verification.checks.push({
						name: "Balance Verification",
						status: "passed",
						message: `Sufficient ${balanceCategory.toLowerCase()} | Requested: ${claimRequest.amountNTE.toFixed(
							6,
						)} | Available: ${availableBalance.toFixed(6)}`,
					});
				}

				// Check wallet balance consistency
				if (user.walletBalance !== undefined) {
					const difference = Math.abs(
						user.walletBalance - originalBreakdown.total,
					);

					if (difference > 0.000001) {
						verification.warnings.push(
							`Wallet balance (${user.walletBalance.toFixed(
								6,
							)}) differs from breakdown total (${originalBreakdown.total.toFixed(
								6,
							)}) by ${difference.toFixed(6)}`,
						);
						verification.checks.push({
							name: "Balance Consistency",
							status: "warning",
							message: `Mismatch: ${difference.toFixed(6)} NTE`,
						});
						verification.recommendations.push(
							"Balance mismatch - consider running balance fix before approving",
						);
					} else {
						verification.checks.push({
							name: "Balance Consistency",
							status: "passed",
							message: "Wallet balance matches breakdown",
						});
					}
				}
			}

			// 4. Check for duplicate pending claims
			const pendingClaims = await ClaimRequest.countDocuments({
				username: claimRequest.username,
				status: "pending",
				_id: { $ne: claimRequest._id },
			});

			if (pendingClaims > 0) {
				verification.warnings.push(
					`User has ${pendingClaims} other pending claim(s)`,
				);
				verification.checks.push({
					name: "Duplicate Claims",
					status: "warning",
					message: `${pendingClaims} other pending claim(s)`,
				});
				verification.recommendations.push(
					"Multiple pending claims - verify intentional",
				);
			} else {
				verification.checks.push({
					name: "Duplicate Claims",
					status: "passed",
					message: "No duplicate pending claims",
				});
			}

			// 5. Check wallet address format
			if (!isValidWalletAddress(claimRequest.walletAddress)) {
				verification.isLegit = false;
				verification.riskLevel = "HIGH";
				verification.errors.push("Invalid wallet address format");
				verification.checks.push({
					name: "Address Format",
					status: "failed",
					message: "Wallet address format is invalid",
				});
				verification.recommendations.push(
					"Invalid address format - Do not approve",
				);
			} else {
				verification.checks.push({
					name: "Address Format",
					status: "passed",
					message: "Wallet address format is valid",
				});
			}

			// 6. Check user registration age (anti-fraud)
			const userAge = user.createdAt
				? Math.floor(
						(Date.now() - new Date(user.createdAt).getTime()) /
							TIME_CONSTANTS.ONE_DAY_MS,
					)
				: 0;

			if (userAge < CLAIM_LIMITS.MIN_ACCOUNT_AGE_DAYS) {
				verification.isLegit = false;
				verification.riskLevel = "CRITICAL";
				verification.errors.push("Account is less than 24 hours old");
				verification.checks.push({
					name: "Account Age",
					status: "failed",
					message: `Account is ${userAge} day(s) old - minimum ${CLAIM_LIMITS.MIN_ACCOUNT_AGE_DAYS} day required`,
				});
				verification.recommendations.push(
					"Account too new (< 24h) - Do not approve",
				);
			} else if (userAge < CLAIM_LIMITS.NEW_ACCOUNT_WARNING_DAYS) {
				// For accounts less than 7 days old, check if they have previous successful claims
				const successfulClaimsCount = await ClaimRequest.countDocuments({
					username: claimRequest.username,
					_id: { $ne: claimRequest._id },
					status: { $in: ["approved", "completed"] },
				});
				const hasSuccessfulClaim = successfulClaimsCount > 0;

				if (hasSuccessfulClaim) {
					verification.checks.push({
						name: "Account Age",
						status: "passed",
						message: `Account created ${userAge} day(s) ago with previous successful claim(s)`,
						details: { ageInDays: userAge, hasSuccessfulClaim: true },
					});
				} else {
					verification.warnings.push(
						`Account is ${userAge} days old - new account (< ${CLAIM_LIMITS.NEW_ACCOUNT_WARNING_DAYS} days) with no successful claims`,
					);
					verification.checks.push({
						name: "Account Age",
						status: "warning",
						message: `Account is ${userAge} day(s) old - no previous successful claims`,
						details: { ageInDays: userAge, hasSuccessfulClaim: false },
					});
					verification.recommendations.push(
						"New account - verify activity is legitimate",
					);
				}
			} else {
				verification.checks.push({
					name: "Account Age",
					status: "passed",
					message: `Account is ${userAge} day(s) old`,
					details: { ageInDays: userAge },
				});
			}

			// 4b. Check for active withdrawal requests for this claim
			const activeWithdrawal = await WithdrawalRequest.findOne({
				claimRequestId: claimRequest._id.toString(),
				status: {
					$in: ["pending", "processing", "validating", "transferring"],
				},
			});

			if (activeWithdrawal) {
				verification.isLegit = false;
				verification.errors.push(
					"This claim is already being processed as a withdrawal",
				);
				verification.checks.push({
					name: "Active Withdrawal Check",
					status: "failed",
					message: `Withdrawal request already exists (status: ${activeWithdrawal.status})`,
					details: {
						withdrawalId: activeWithdrawal._id.toString(),
						status: activeWithdrawal.status,
						createdAt: activeWithdrawal.createdAt,
					},
				});
				verification.recommendations.push(
					"Claim is already being processed - Do not approve again",
				);
			} else {
				verification.checks.push({
					name: "Active Withdrawal Check",
					status: "passed",
					message: "No active withdrawal found for this claim",
				});
			}

			// 7. Amount Validation
			const minAmount = CLAIM_LIMITS.MIN_AMOUNT_NTE; // Minimum 1 NTE
			const maxAmount = verification.details.balance?.available?.total || 0;

			if (claimRequest.amountNTE < minAmount) {
				verification.isLegit = false;
				verification.errors.push(`Amount below minimum (${minAmount} NTE)`);
				verification.checks.push({
					name: "Amount Validation",
					status: "failed",
					message: `Claim amount is below minimum of ${minAmount} NTE`,
				});
			} else if (claimRequest.amountNTE > maxAmount && maxAmount > 0) {
				verification.warnings.push(
					"Amount exceeds user's total available balance",
				);
				verification.checks.push({
					name: "Amount Validation",
					status: "warning",
					message: "Claim amount exceeds user's total available balance",
					details: {
						requested: claimRequest.amountNTE,
						maxAvailable: maxAmount,
					},
				});
			} else {
				verification.checks.push({
					name: "Amount Validation",
					status: "passed",
					message: "Claim amount is within valid range",
				});
			}

			// 8. Claim History and Patterns
			const userClaimHistory = await ClaimRequest.find({
				username: claimRequest.username,
				_id: { $ne: claimRequest._id },
			})
				.sort({ createdAt: -1 })
				.limit(10)
				.lean();

			const completedClaims = userClaimHistory.filter(
				(c) => c.status === "completed",
			).length;
			const rejectedClaims = userClaimHistory.filter(
				(c) => c.status === "rejected",
			).length;
			const failedClaims = userClaimHistory.filter(
				(c) => c.status === "failed",
			).length;
			const approvedClaims = userClaimHistory.filter(
				(c) => c.status === "approved",
			).length;
			const otherPendingClaims = userClaimHistory.filter(
				(c) => c.status === "pending",
			).length;
			const totalOtherClaims = userClaimHistory.length;

			if (rejectedClaims > 2) {
				verification.warnings.push(
					`User has ${rejectedClaims} previously rejected claims`,
				);
				verification.checks.push({
					name: "Claim History",
					status: "warning",
					message: `${rejectedClaims} rejected claim(s) in history`,
					details: {
						completed: completedClaims,
						rejected: rejectedClaims,
						failed: failedClaims,
						approved: approvedClaims,
						pending: otherPendingClaims,
						total: totalOtherClaims,
					},
				});
				verification.riskLevel =
					verification.riskLevel === "CRITICAL" ? "CRITICAL" : "HIGH";
			} else if (totalOtherClaims === 0) {
				verification.recommendations.push(
					"First-time claim - verify user identity carefully",
				);
				verification.checks.push({
					name: "Claim History",
					status: "warning",
					message: "This is user's first claim request",
					details: { isFirstClaim: true },
				});
			} else {
				const statusParts = [];
				if (completedClaims > 0)
					statusParts.push(`${completedClaims} completed`);
				if (approvedClaims > 0) statusParts.push(`${approvedClaims} approved`);
				if (rejectedClaims > 0) statusParts.push(`${rejectedClaims} rejected`);
				if (failedClaims > 0) statusParts.push(`${failedClaims} failed`);
				if (otherPendingClaims > 0)
					statusParts.push(`${otherPendingClaims} pending`);

				verification.checks.push({
					name: "Claim History",
					status: "passed",
					message: `User has ${totalOtherClaims} other claim${
						totalOtherClaims === 1 ? "" : "s"
					} (${statusParts.join(", ")})`,
					details: {
						completed: completedClaims,
						rejected: rejectedClaims,
						failed: failedClaims,
						approved: approvedClaims,
						pending: otherPendingClaims,
						total: totalOtherClaims,
					},
				});
			}

			// 9. Claim Frequency (24h)
			const recentClaims = await ClaimRequest.find({
				username: claimRequest.username,
				_id: { $ne: claimRequest._id },
				createdAt: { $gte: new Date(Date.now() - TIME_CONSTANTS.ONE_DAY_MS) },
			}).lean();

			if (recentClaims.length > CLAIM_LIMITS.MAX_CLAIMS_PER_DAY) {
				verification.warnings.push(
					`${recentClaims.length} claims submitted in last 24 hours`,
				);
				verification.checks.push({
					name: "Claim Frequency",
					status: "warning",
					message: `${recentClaims.length} claims in last 24 hours - unusual activity`,
					details: { claimsIn24h: recentClaims.length },
				});
				verification.riskLevel = "HIGH";
			} else {
				verification.checks.push({
					name: "Claim Frequency",
					status: "passed",
					message: `Normal claim frequency (${recentClaims.length} in 24h)`,
					details: { claimsIn24h: recentClaims.length },
				});
			}

			// 10. Large Withdrawal Percentage Warning
			const totalAvailableBalance =
				verification.details.balance?.available?.total || 0;
			const claimPercentage =
				totalAvailableBalance > 0
					? (claimRequest.amountNTE / totalAvailableBalance) * 100
					: 0;

			if (claimPercentage > 90) {
				verification.warnings.push(
					"Claim amount is more than 90% of total available balance",
				);
				verification.checks.push({
					name: "Withdrawal Percentage",
					status: "warning",
					message: `Withdrawing ${claimPercentage.toFixed(
						1,
					)}% of total available balance`,
					details: {
						percentage: claimPercentage.toFixed(2),
						available: totalAvailableBalance,
						requested: claimRequest.amountNTE,
					},
				});
				verification.recommendations.push("Large Claim - confirm user intent");
			} else if (claimPercentage > 50) {
				verification.warnings.push(
					"Claim amount is more than 50% of total available balance",
				);
				verification.checks.push({
					name: "Claim Percentage",
					status: "warning",
					message: `Claiming ${claimPercentage.toFixed(
						1,
					)}% of total available balance`,
					details: {
						percentage: claimPercentage.toFixed(2),
						available: totalAvailableBalance,
						requested: claimRequest.amountNTE,
					},
				});
			} else {
				verification.checks.push({
					name: "Claim Percentage",
					status: "passed",
					message: `Claiming ${claimPercentage.toFixed(
						1,
					)}% of total available balance`,
					details: {
						percentage: claimPercentage.toFixed(2),
						available: totalAvailableBalance,
						requested: claimRequest.amountNTE,
					},
				});
			}

			// 11. Transaction History Integrity Check
			const last4Transactions = await WalletTransaction.find({
				username: claimRequest.username,
			})
				.sort({ createdAt: -1 })
				.limit(4)
				.lean();

			if (last4Transactions.length > 0) {
				let balanceIntegrityPassed = true;
				const balanceDiscrepancies = [];
				const transactionChecks = [];

				// Check each transaction's internal calculation
				for (let i = 0; i < last4Transactions.length; i++) {
					const tx = last4Transactions[i];
					const isPackagePurchase = tx.type === "package_purchase";

					if (
						tx.balanceBefore !== undefined &&
						tx.balanceAfter !== undefined &&
						tx.amountNTE !== undefined
					) {
						const expectedBalanceAfter = isPackagePurchase
							? tx.balanceBefore
							: tx.balanceBefore + tx.amountNTE;
						const actualBalanceAfter = tx.balanceAfter;
						const difference = Math.abs(
							expectedBalanceAfter - actualBalanceAfter,
						);
						const isValid = difference <= 0.000001;

						transactionChecks.push({
							txId: tx._id.toString(),
							txType: tx.type,
							txDate: tx.createdAt,
							balanceBefore: tx.balanceBefore,
							amountNTE: tx.amountNTE,
							expectedBalanceAfter,
							actualBalanceAfter,
							difference,
							isValid,
							isPackagePurchase,
						});

						if (!isValid) {
							balanceIntegrityPassed = false;
						}
					}
				}

				// Check balance continuity between consecutive transactions
				for (let i = 0; i < last4Transactions.length - 1; i++) {
					const newerTx = last4Transactions[i];
					const olderTx = last4Transactions[i + 1];
					const newerPosition = i + 1;
					const olderPosition = i + 2;

					if (
						olderTx.balanceAfter !== undefined &&
						newerTx.balanceBefore !== undefined
					) {
						const difference = Math.abs(
							olderTx.balanceAfter - newerTx.balanceBefore,
						);

						if (difference > 0.000001) {
							const isOlderPackagePurchase =
								olderTx.type === "package_purchase";
							const packagePurchaseInSafePosition =
								isOlderPackagePurchase &&
								(olderPosition === 3 || olderPosition === 4);

							if (!packagePurchaseInSafePosition) {
								balanceIntegrityPassed = false;
							}

							balanceDiscrepancies.push({
								olderTransaction: {
									id: olderTx._id.toString(),
									type: olderTx.type,
									balanceAfter: olderTx.balanceAfter,
								},
								newerTransaction: {
									id: newerTx._id.toString(),
									type: newerTx.type,
									balanceBefore: newerTx.balanceBefore,
								},
								discrepancy: difference,
								isHarmful: !packagePurchaseInSafePosition,
							});
						}
					}
				}

				const harmfulDiscrepancies = balanceDiscrepancies.filter(
					(d) => d.isHarmful,
				);

				if (!balanceIntegrityPassed) {
					verification.warnings.push(
						"Transaction balance integrity issues detected",
					);
					verification.checks.push({
						name: "Transaction History Integrity",
						status: "warning",
						message:
							"Balance calculation or continuity errors found in recent transactions",
						details: {
							totalTransactionsChecked: last4Transactions.length,
							transactionCalculations: transactionChecks,
							continuityErrors: balanceDiscrepancies.length,
							harmfulErrors: harmfulDiscrepancies.length,
							continuityDiscrepancies: balanceDiscrepancies,
						},
					});
					verification.riskLevel =
						verification.riskLevel === "CRITICAL" ? "CRITICAL" : "HIGH";
				} else {
					verification.checks.push({
						name: "Transaction History Integrity",
						status: "passed",
						message: `All ${last4Transactions.length} transaction(s) have valid balance calculations and continuity`,
						details: {
							totalTransactionsChecked: last4Transactions.length,
							transactionCalculations: transactionChecks,
						},
					});
				}

				verification.details.recentTransactions = last4Transactions.map(
					(tx) => ({
						id: tx._id.toString(),
						type: tx.type,
						subType: tx.subType,
						amountNTE: tx.amountNTE,
						balanceBefore: tx.balanceBefore,
						balanceAfter: tx.balanceAfter,
						status: tx.status,
						createdAt: tx.createdAt,
					}),
				);
			} else {
				verification.warnings.push("No transaction history found for user");
				verification.checks.push({
					name: "Transaction History Integrity",
					status: "warning",
					message: "No transaction history found",
				});
			}

			// 12. Last 2 Withdrawals Verification
			const last2Withdrawals = await WalletTransaction.find({
				username: claimRequest.username,
				type: "withdrawal",
				status: "completed",
			})
				.sort({ createdAt: -1 })
				.limit(2)
				.lean();

			if (last2Withdrawals.length > 0) {
				const lastWithdrawal = last2Withdrawals[0];
				const timeSinceLastWithdrawal = lastWithdrawal.createdAt
					? Math.floor(
							(Date.now() - new Date(lastWithdrawal.createdAt).getTime()) /
								(1000 * 60),
						)
					: null;

				const withdrawalDetails = last2Withdrawals.map((w, idx) => ({
					withdrawalNumber: idx + 1,
					id: w._id.toString(),
					date: w.createdAt,
					amount: w.amountNTE,
					hash: w.transactionHash,
					minutesSince: w.createdAt
						? Math.floor(
								(Date.now() - new Date(w.createdAt).getTime()) / (1000 * 60),
							)
						: null,
					balanceBefore: w.balanceBefore,
					balanceAfter: w.balanceAfter,
				}));

				if (timeSinceLastWithdrawal !== null && timeSinceLastWithdrawal < 20) {
					verification.warnings.push(
						`Recent withdrawal detected ${timeSinceLastWithdrawal} minute(s) ago`,
					);
					verification.checks.push({
						name: "Recent Withdrawal Check",
						status: "warning",
						message: `Last withdrawal was ${timeSinceLastWithdrawal} minute(s) ago`,
						details: {
							withdrawalsChecked: last2Withdrawals.length,
							withdrawals: withdrawalDetails,
						},
					});
					verification.riskLevel =
						verification.riskLevel === "CRITICAL"
							? "CRITICAL"
							: verification.riskLevel === "HIGH"
								? "HIGH"
								: "MEDIUM";
				} else {
					verification.checks.push({
						name: "Recent Withdrawal Check",
						status: "passed",
						message: timeSinceLastWithdrawal
							? `Last withdrawal was ${timeSinceLastWithdrawal} minute(s) ago`
							: "Last withdrawal date available",
						details: {
							withdrawalsChecked: last2Withdrawals.length,
							withdrawals: withdrawalDetails,
						},
					});
				}

				// 13. Verify balance integrity in last withdrawal
				if (
					lastWithdrawal.balanceBefore !== undefined &&
					lastWithdrawal.balanceAfter !== undefined &&
					lastWithdrawal.amountNTE !== undefined
				) {
					const expectedBalanceAfter =
						lastWithdrawal.balanceBefore + lastWithdrawal.amountNTE;
					const actualBalanceAfter = lastWithdrawal.balanceAfter;
					const difference = Math.abs(
						expectedBalanceAfter - actualBalanceAfter,
					);
					const wasDeducted = lastWithdrawal.amountNTE < 0;

					if (difference > 0.000001) {
						verification.warnings.push(
							`Last withdrawal has balance calculation mismatch of ${difference.toFixed(
								6,
							)} NTE`,
						);
						verification.checks.push({
							name: "Withdrawal Balance Deduction",
							status: "warning",
							message: "Balance was NOT properly deducted during withdrawal",
							details: {
								withdrawalId: lastWithdrawal._id.toString(),
								balanceBefore: lastWithdrawal.balanceBefore,
								withdrawalAmount: lastWithdrawal.amountNTE,
								expectedBalanceAfter,
								actualBalanceAfter,
								difference,
								wasAmountNegative: wasDeducted,
							},
						});
						verification.riskLevel = "HIGH";
					} else if (!wasDeducted) {
						verification.warnings.push(
							`Last withdrawal amount is not negative (${lastWithdrawal.amountNTE})`,
						);
						verification.checks.push({
							name: "Withdrawal Balance Deduction",
							status: "warning",
							message: "Withdrawal amount should be negative but is positive",
							details: {
								withdrawalId: lastWithdrawal._id.toString(),
								withdrawalAmount: lastWithdrawal.amountNTE,
								wasAmountNegative: wasDeducted,
							},
						});
					} else {
						verification.checks.push({
							name: "Withdrawal Balance Deduction",
							status: "passed",
							message: "Balance was correctly deducted during last withdrawal",
							details: {
								withdrawalId: lastWithdrawal._id.toString(),
								balanceBefore: lastWithdrawal.balanceBefore,
								withdrawalAmount: lastWithdrawal.amountNTE,
								balanceAfter: actualBalanceAfter,
							},
						});
					}
				}
			} else {
				verification.checks.push({
					name: "Recent Withdrawal Check",
					status: "passed",
					message: "No previous withdrawal found",
				});
				verification.checks.push({
					name: "Withdrawal Balance Deduction",
					status: "passed",
					message: "No previous withdrawal to verify",
				});
			}
		}

		// Add claim request details
		verification.details.claim = {
			id: claimRequest._id,
			username: claimRequest.username,
			amount: claimRequest.amountNTE,
			usdValue: claimRequest.usdValue,
			walletAddress: claimRequest.walletAddress,
			withdrawalType: claimRequest.withdrawalType || "total",
			status: claimRequest.status,
			createdAt: claimRequest.createdAt,
		};

		// Determine overall risk level
		const criticalErrors = verification.errors.length;
		const warnings = verification.warnings.length;

		if (criticalErrors > 0) {
			verification.riskLevel = "CRITICAL";
		} else if (warnings > 3) {
			verification.riskLevel = "HIGH";
		} else if (warnings > 1) {
			verification.riskLevel = "MEDIUM";
		} else {
			verification.riskLevel = "LOW";
		}

		if (verification.isLegit === false) {
			verification.canApprove = false;
		} else if (verification.riskLevel === "CRITICAL") {
			verification.canApprove = false;
		}

		// Add automated recommendations
		if (verification.isLegit) {
			if (verification.riskLevel === "LOW") {
				verification.recommendations.push(
					"All checks passed - Safe to approve",
				);
			} else if (verification.riskLevel === "MEDIUM") {
				verification.recommendations.push(
					"Minor warnings detected - Review before approval",
				);
			} else if (verification.riskLevel === "HIGH") {
				verification.recommendations.push(
					"Multiple warnings - Careful review recommended",
				);
			}
		} else {
			verification.recommendations.push(
				"Critical issues found - Do not approve without investigation",
			);
		}

		return verification;
	},

	/**
	 * Approve claim request and process withdrawal
	 */
	async approveClaimRequest(data) {
		const {
			claimRequestId,
			masterPassword,
			notes,
			adminUsername,
			adminUserId,
			autoApproval,
		} = data;

		const session = await createSessionIfSupported();
		let withdrawalRequest = null;
		let claimRequest = null;
		let user;

		try {
			if (session) {
				await session.startTransaction();
			}

			await checkQueuePauseStatus(
				autoApproval,
				claimRequestId,
				adminUsername,
				data.clientIP,
				data.userAgent,
			);

			// Verify master password (skip for auto-approval)
			if (!autoApproval) {
				const adminConfig = await AdminConfig.findOne({
					type: "master_password",
				});
				if (!adminConfig || !adminConfig.passwordHash) {
					throw new ApiError(
						500,
						"Master password not configured. Please contact system administrator.",
					);
				}

				const isMasterPasswordValid = await bcrypt.compare(
					masterPassword,
					adminConfig.passwordHash,
				);

				// Log master password attempt
				await auditLogService.logMasterPasswordAttempt({
					adminUsername,
					adminUserId,
					action: "claim_approval",
					success: isMasterPasswordValid,
					clientIP: data.clientIP,
					userAgent: data.userAgent,
				});

				if (!isMasterPasswordValid) {
					throw new ApiError(401, "Invalid master password");
				}
			}

			// STEP 1: Atomically acquire processing lock to prevent duplicate approvals
			// Skip lock acquisition for auto-approval (worker already has the lock)
			if (autoApproval) {
				// For auto-approval, just get the claim without trying to lock
				claimRequest = await withSession(
					ClaimRequest.findOne({
						_id: claimRequestId,
						status: "pending",
					}),
					session,
				);

				if (!claimRequest) {
					const existingClaim = await withSession(
						ClaimRequest.findById(claimRequestId),
						session,
					);

					if (!existingClaim) {
						throw new ApiError(404, "Claim request not found");
					}

					throw new ApiError(
						400,
						`Claim request already ${existingClaim.status}`,
					);
				}

				logger.info(
					`CLAIM LOCK | Auto-approval using existing lock for claim ${claimRequestId}`,
				);
			} else {
				// For manual approval, acquire the lock
				claimRequest = await withSession(
					ClaimRequest.findOneAndUpdate(
						{
							_id: claimRequestId,
							status: "pending",
							$or: [
								{ processingLock: { $ne: true } },
								{ processingLock: { $exists: false } },
							],
						},
						{
							processingLock: true,
							processingLockedAt: new Date(),
							processingLockedBy: adminUsername,
						},
						{ new: true },
					),
					session,
				);

				if (!claimRequest) {
					// Check why lock failed
					const existingClaim = await withSession(
						ClaimRequest.findById(claimRequestId),
						session,
					);

					if (!existingClaim) {
						throw new ApiError(404, "Claim request not found");
					}

					if (existingClaim.status !== "pending") {
						throw new ApiError(
							400,
							`Claim request already ${existingClaim.status}`,
						);
					}

					if (existingClaim.processingLock === true) {
						// Check if it's being processed by queue worker
						const lockedBy =
							existingClaim.processingLockedBy || "another admin";
						const lockedAt = existingClaim.processingLockedAt;
						const lockAge = lockedAt
							? Math.floor((Date.now() - new Date(lockedAt).getTime()) / 1000)
							: 0;

						throw new ApiError(
							409,
							`This claim is currently being processed by ${lockedBy}${
								lockAge > 0 ? ` (${lockAge}s ago)` : ""
							}. Please wait or use queue management to skip it.`,
						);
					}

					throw new ApiError(500, "Failed to acquire claim processing lock");
				}

				logger.info(
					`CLAIM LOCK | Acquired processing lock for claim ${claimRequestId} by ${adminUsername}`,
				);

				// Check if claim is in auto-approval queue and log warning
				if (claimRequest.queuedForApproval && claimRequest.queuePosition) {
					logger.warn(
						`MANUAL OVERRIDE | Admin ${adminUsername} manually approving claim ${claimRequestId} that was queued for auto-approval (position ${claimRequest.queuePosition})`,
					);
				}
			}

			// STEP 2: Check global withdrawal settings
			const systemConfig = await AdminSystemSettings.findOne({
				type: "system_config",
			});

			if (systemConfig && systemConfig.withdrawalsEnabled === false) {
				throw new ApiError(
					403,
					systemConfig.withdrawalsMessage ||
						"Withdrawals are currently disabled system-wide",
				);
			}

			// STEP 3: Check if withdrawal already exists for this claim
			const existingWithdrawal = await withSession(
				WithdrawalRequest.findOne({
					claimRequestId: claimRequestId.toString(),
					status: {
						$in: [
							"pending",
							"processing",
							"validating",
							"transferring",
							"logging",
							"completed",
						],
					},
				}),
				session,
			);

			if (existingWithdrawal) {
				throw new ApiError(
					409,
					`A withdrawal request (${existingWithdrawal._id}) already exists for this claim with status: ${existingWithdrawal.status}`,
				);
			}

			logger.info(
				`CLAIM VERIFICATION | No existing withdrawal found for claim ${claimRequestId}`,
			);

			// 2. Get user
			user = await withSession(User.findById(claimRequest.userId), session);

			if (!user) {
				throw new ApiError(404, "User not found");
			}

			// Verify wallet address matches user's registered wallet
			const requestedWalletHash = createWalletHash(claimRequest.walletAddress);
			if (user.walletHash && user.walletHash !== requestedWalletHash) {
				logger.error(
					`SECURITY | Wallet address mismatch during approval for claim ${claimRequestId}`,
					{
						username: user.username,
						claimWalletAddress: claimRequest.walletAddress,
						claimWalletHash: requestedWalletHash,
						userWalletHash: user.walletHash,
					},
				);

				// Audit log critical security violation
				await auditLogService.logSecurityViolation({
					type: "WALLET_MISMATCH_DURING_APPROVAL",
					username: user.username,
					description: `Wallet address mismatch detected during claim approval for claim ${claimRequestId}`,
					severity: "CRITICAL",
					clientIP: data.clientIP,
					userAgent: data.userAgent,
					metadata: {
						claimRequestId,
						claimWalletAddress: claimRequest.walletAddress,
						requestedWalletHash,
						userWalletHash: user.walletHash,
						approvedBy: adminUsername,
					},
				});

				throw new ApiError(
					403,
					"Wallet address does not match user's registered wallet. Claim cannot be approved.",
				);
			}

			// 3. Check if user has sufficient balance
			if (user.walletBalance < claimRequest.amountNTE) {
				throw new ApiError(400, "User has insufficient balance for this claim");
			}

			// Account age verification
			const userAge = user.createdAt
				? Math.floor(
						(Date.now() - new Date(user.createdAt).getTime()) /
							TIME_CONSTANTS.ONE_DAY_MS,
					)
				: 0;

			if (userAge < CLAIM_LIMITS.MIN_ACCOUNT_AGE_DAYS) {
				logger.warn(
					`User ${user.username} account age is less than ${CLAIM_LIMITS.MIN_ACCOUNT_AGE_DAYS} day (${userAge} days)`,
				);
				throw new ApiError(
					400,
					"Account must be at least 1 day old to withdraw. Please contact support.",
				);
			}

			// Check withdrawal frequency (last 24 hours)
			const last24Hours = new Date(Date.now() - TIME_CONSTANTS.ONE_DAY_MS);
			const recentWithdrawalsCount = await withSession(
				WithdrawalRequest.countDocuments({
					username: user.username,
					createdAt: { $gte: last24Hours },
					status: { $in: ["completed", "processing", "pending"] },
				}),
				session,
			);

			const isAdminUser = user.role === "admin";
			if (
				recentWithdrawalsCount >= CLAIM_LIMITS.MAX_WITHDRAWALS_PER_DAY &&
				!isAdminUser
			) {
				throw new ApiError(
					429,
					`Too many withdrawal attempts. Limit: ${CLAIM_LIMITS.MAX_WITHDRAWALS_PER_DAY} per 24 hours. You have ${recentWithdrawalsCount} withdrawals in the last 24 hours.`,
				);
			}

			// Check if user is blocked
			if (user.isBlocked === true) {
				throw new ApiError(
					403,
					user.blockReason ||
						"Your account is blocked. Please contact support.",
				);
			}

			// Check if withdrawals disabled for user
			if (user.withdrawalsDisabled === true) {
				throw new ApiError(
					403,
					"Withdrawals are disabled for your account. Please contact support.",
				);
			}

			// Check for concurrent withdrawals
			const pendingWithdrawals = await withSession(
				WithdrawalRequest.countDocuments({
					username: user.username,
					status: {
						$in: ["pending", "processing", "validating", "transferring"],
					},
				}),
				session,
			);

			if (pendingWithdrawals > 0) {
				throw new ApiError(
					409,
					"You have a withdrawal in progress. Please wait for it to complete before requesting another.",
				);
			}

			logger.info(`All withdrawal checks passed for ${user.username}`);

			// 4. Get current token value
			const tokenValue = await getTokenValue();

			// 6. Determine source balance type
			const sourceBalanceInfo =
				await companyWalletService.determineSourceBalanceType(
					claimRequest.username,
					claimRequest.amountNTE,
					claimRequest.withdrawalType,
				);

			// 7. Atomically mark withdrawal as created with idempotency protection
			// Generate cryptographically secure idempotency key to prevent duplicate withdrawals
			const idempotencySalt = crypto.randomBytes(16).toString("hex");
			const idempotencyKey = crypto
				.createHash("sha256")
				.update(
					`${claimRequestId}:${user._id}:${
						claimRequest.amountNTE
					}:${Date.now()}:${idempotencySalt}`,
				)
				.digest("hex");

			// Check for existing withdrawal with this idempotency key (across ALL statuses)
			const existingWithdrawalByKey = await withSession(
				WithdrawalRequest.findOne({ idempotencyKey }),
				session,
			);

			if (existingWithdrawalByKey) {
				logger.warn(
					`IDEMPOTENCY | Duplicate withdrawal prevented for claim ${claimRequestId}`,
					{
						existingWithdrawalId: existingWithdrawalByKey._id,
						existingStatus: existingWithdrawalByKey.status,
						idempotencyKey,
					},
				);

				// Audit log duplicate prevention
				await auditLogService.logSecurityViolation({
					type: "DUPLICATE_WITHDRAWAL_PREVENTED",
					username: user.username,
					description: `Duplicate withdrawal prevented via idempotency key for claim ${claimRequestId}`,
					severity: "HIGH",
					clientIP: data.clientIP,
					userAgent: data.userAgent,
					metadata: {
						idempotencyKey,
						claimRequestId,
						existingWithdrawalId: existingWithdrawalByKey._id,
					},
				});

				throw new ApiError(
					409,
					`Withdrawal already exists for this claim (ID: ${existingWithdrawalByKey._id}, Status: ${existingWithdrawalByKey.status}). Duplicate prevented.`,
				);
			}

			// Get current claim version for optimistic locking
			const currentClaim = await withSession(
				ClaimRequest.findById(claimRequestId).select(
					"withdrawalCreated withdrawalCreatedAt withdrawalCreatedBy __v",
				),
				session,
			);

			if (currentClaim.withdrawalCreated === true) {
				throw new ApiError(
					409,
					`Withdrawal already created at ${currentClaim.withdrawalCreatedAt?.toISOString()}`,
				);
			}

			// Atomic withdrawal creation lock with version check
			const withdrawalCreationLock = await withSession(
				ClaimRequest.findOneAndUpdate(
					{
						_id: claimRequestId,
						status: "pending",
						processingLock: true,
						withdrawalCreated: { $ne: true },
						__v: currentClaim.__v, // Version check to detect concurrent modifications
					},
					{
						$set: {
							withdrawalCreated: true,
							withdrawalCreatedAt: new Date(),
							withdrawalCreatedBy: adminUsername,
							idempotencyKey,
						},
						$inc: { __v: 1 },
					},
					{ new: true },
				),
				session,
			);

			if (!withdrawalCreationLock) {
				const currentClaim = await withSession(
					ClaimRequest.findById(claimRequestId),
					session,
				);

				if (currentClaim.withdrawalCreated === true) {
					logger.error(
						`RACE CONDITION | Withdrawal already created for claim ${claimRequestId}`,
						{
							createdAt: currentClaim.withdrawalCreatedAt,
							createdBy: currentClaim.withdrawalCreatedBy,
							attemptedBy: adminUsername,
						},
					);

					// Audit log race condition detection
					await auditLogService.logSecurityViolation({
						type: "RACE_CONDITION_DETECTED",
						username: user.username,
						description: `Race condition detected in withdrawal creation for claim ${claimRequestId}`,
						severity: "CRITICAL",
						clientIP: data.clientIP,
						userAgent: data.userAgent,
						metadata: {
							claimRequestId,
							createdBy: currentClaim.withdrawalCreatedBy,
							attemptedBy: adminUsername,
							timeDiff:
								Date.now() - currentClaim.withdrawalCreatedAt?.getTime(),
						},
					});

					throw new ApiError(
						409,
						`Withdrawal already created for this claim at ${currentClaim.withdrawalCreatedAt?.toISOString()} by ${
							currentClaim.withdrawalCreatedBy || "another admin"
						}. Race condition prevented.`,
					);
				}

				throw new ApiError(
					500,
					"Failed to acquire withdrawal creation lock - concurrent modification detected",
				);
			}

			logger.info(
				` WITHDRAWAL LOCK | Acquired by ${adminUsername} for claim ${claimRequestId}`,
				{ idempotencyKey },
			);

			// 8. Lock user balance and validate at execution time with version control
			logger.info(
				`BALANCE LOCK | Acquiring balance lock for user ${user.username}...`,
			);

			// Get current user version for optimistic locking
			const currentUser = await withSession(
				User.findById(user._id).select(
					"walletBalance balanceLocked balanceLockedAt __v",
				),
				session,
			);

			if (!currentUser) {
				throw new ApiError(404, "User not found during balance lock");
			}

			// Enhanced atomic lock with balance snapshot and checksum

			const balanceSnapshot = currentUser.walletBalance;
			const lockNonce = crypto.randomBytes(16).toString("hex");
			const balanceChecksum = crypto
				.createHash("sha256")
				.update(`${user._id}:${balanceSnapshot}:${Date.now()}:${lockNonce}`)
				.digest("hex");

			// Atomically lock user's balance with version check to prevent race conditions
			const balanceLockResult = await withSession(
				User.findOneAndUpdate(
					{
						_id: user._id,
						walletBalance: { $gte: claimRequest.amountNTE }, // Re-verify balance
						balanceLocked: { $ne: true }, // Must not be locked
						__v: currentUser.__v, // Version check prevents concurrent modifications
					},
					{
						$set: {
							balanceLocked: true,
							balanceLockedAt: new Date(),
							balanceLockedBy: `claim_${claimRequestId}_${lockNonce}`,
							balanceLockedAmount: claimRequest.amountNTE,
							balanceSnapshot: balanceSnapshot, // NEW: Store balance at lock time
							balanceChecksum: balanceChecksum, // NEW: Cryptographic verification
							balanceLockNonce: lockNonce, // NEW: Unique lock identifier
						},
						$inc: { __v: 1 }, // Increment version
					},
					{ new: true },
				),
				session,
			);

			if (!balanceLockResult) {
				const currentUser = await withSession(
					User.findById(user._id).select(
						"walletBalance balanceLocked balanceLockedBy balanceLockedAt",
					),
					session,
				);

				logger.error(`BALANCE LOCK FAILED | User: ${user.username}`, {
					requiredAmount: claimRequest.amountNTE,
					currentBalance: currentUser?.walletBalance,
					alreadyLocked: currentUser?.balanceLocked,
					lockedBy: currentUser?.balanceLockedBy,
					lockedAt: currentUser?.balanceLockedAt,
				});

				// Audit log balance lock failure
				await auditLogService.logSecurityViolation({
					type: "BALANCE_LOCK_FAILED",
					username: user.username,
					description: `Failed to lock balance for claim ${claimRequestId}. Balance: ${currentUser?.walletBalance}, Required: ${claimRequest.amountNTE}`,
					severity: "HIGH",
					clientIP: data.clientIP,
					userAgent: data.userAgent,
					metadata: {
						claimRequestId,
						currentBalance: currentUser?.walletBalance,
						requiredAmount: claimRequest.amountNTE,
						alreadyLocked: currentUser?.balanceLocked,
						lockedBy: currentUser?.balanceLockedBy,
					},
				});

				if (currentUser?.walletBalance < claimRequest.amountNTE) {
					throw new ApiError(
						400,
						`Insufficient balance at execution time. Required: ${claimRequest.amountNTE} NTE, Available: ${currentUser.walletBalance} NTE`,
					);
				}

				if (currentUser?.balanceLocked) {
					throw new ApiError(
						409,
						`Balance is locked by another operation: ${currentUser.balanceLockedBy}. Please try again later.`,
					);
				}

				throw new ApiError(500, "Failed to acquire balance lock");
			}

			logger.info(`Balance lock acquired for user ${user.username}`, {
				lockedAmount: claimRequest.amountNTE,
				currentBalance: balanceLockResult.walletBalance,
			});

			// 9. Create withdrawal request with idempotency key
			withdrawalRequest = await companyWalletService.createWithdrawalRequest({
				userId: user._id,
				username: user.username,
				toAddress: claimRequest.walletAddress,
				amountNTE: claimRequest.amountNTE,
				amountUSD: claimRequest.usdValue,
				selectedWithdrawalType: claimRequest.withdrawalType,
				metadata: {
					clientIP: claimRequest.ipAddress,
					tokenValue,
					userAgent: claimRequest.userAgent,
					claimRequestId: claimRequest._id.toString(),
					approvedBy: adminUsername,
					idempotencyKey,
				},
				sourceBalanceInfo,
				claimRequestId: claimRequest._id.toString(),
				walletBalanceBefore: balanceLockResult.walletBalance,
				idempotencyKey,
			});

			// 10. Update claim request to approved
			const updateOptions = session ? { session } : {};
			await ClaimRequest.findByIdAndUpdate(
				claimRequestId,
				{
					status: "approved",
					approvedAt: new Date(),
					approvedBy: adminUsername,
					withdrawalRequestId: withdrawalRequest._id.toString(),
					notes: notes || null,
				},
				updateOptions,
			);

			logger.info(
				`Claim request ${claimRequestId} approved by ${adminUsername}`,
			);

			if (session) {
				await session.commitTransaction();
			}

			// Audit log successful lock acquisition and approval
			await auditLogService.logClaimApproval({
				claimRequestId: claimRequestId.toString(),
				adminUsername,
				adminUserId,
				username: claimRequest.username,
				amountNTE: claimRequest.amountNTE,
				usdValue: claimRequest.usdValue,
				success: true,
				phase: "approved",
				idempotencyKey,
				balanceLocked: true,
				clientIP: data.clientIP,
				userAgent: data.userAgent,
			});

			// 11. TWO-PHASE COMMIT: Process withdrawal with rollback safety
			let withdrawalResult = null;

			try {
				logger.info(
					`WITHDRAWAL PROCESSING | Starting TWO-PHASE COMMIT for claim ${claimRequestId}...`,
				);

				// PHASE 1: Balance is already reserved (locked) but not deducted yet
				logger.info(
					`PHASE 1 | Balance reserved: ${claimRequest.amountNTE} NTE for user ${user.username}`,
				);

				// PHASE 2: Execute blockchain transfer
				logger.info(`PHASE 2 | Executing blockchain transfer...`);
				withdrawalResult =
					await companyWalletService.processWithdrawal(withdrawalRequest);

				// Verify blockchain transfer succeeded
				if (!withdrawalResult || !withdrawalResult.success) {
					throw new Error(
						`Blockchain transfer failed: ${
							withdrawalResult?.error || "Unknown error"
						}`,
					);
				}

				logger.info(
					`PHASE 2 |  Blockchain transfer successful: ${withdrawalResult.transactionHash}`,
				);

				// PHASE 3: Confirm balance deduction (already done in processWithdrawal, but verify)
				logger.info(`PHASE 3 | Confirming balance deduction and finalizing...`);

				// Update claim with completion details
				await ClaimRequest.findByIdAndUpdate(claimRequestId, {
					status: "completed",
					processedAt: new Date(),
					transactionHash: withdrawalResult.transactionHash,
					blockNumber: withdrawalResult.blockNumber,
					blockchainVerificationPassed: true,
				});

				// Release balance lock after successful withdrawal
				await User.findByIdAndUpdate(user._id, {
					$set: {
						balanceLocked: false,
						balanceLockedAt: null,
						balanceLockedBy: null,
						balanceLockedAmount: 0,
					},
				});

				logger.info(
					`TWO-PHASE COMMIT SUCCESS | Claim ${claimRequestId} completed`,
					{
						transactionHash: withdrawalResult.transactionHash,
						newBalance: withdrawalResult.newBalance,
						balanceUnlocked: true,
					},
				);

				// Audit log successful approval and withdrawal
				await auditLogService.logClaimApproval({
					claimRequestId: claimRequestId.toString(),
					adminUsername,
					adminUserId,
					username: claimRequest.username,
					amountNTE: claimRequest.amountNTE,
					usdValue: claimRequest.usdValue,
					success: true,
					error: null,
					transactionHash: withdrawalResult.transactionHash,
					clientIP: data.clientIP,
					userAgent: data.userAgent,
				});

				// Audit log withdrawal processing
				await auditLogService.logWithdrawalProcessing({
					withdrawalRequestId: withdrawalRequest._id.toString(),
					claimRequestId: claimRequestId.toString(),
					username: claimRequest.username,
					amountNTE: claimRequest.amountNTE,
					toAddress: claimRequest.walletAddress,
					status: "completed",
					transactionHash: withdrawalResult.transactionHash,
					blockNumber: withdrawalResult.blockNumber,
					error: null,
				});

				// Send success notification to user
				try {
					const userEmailBody = `
						<div style="color: #e1e4eb; line-height: 1.8;">
							<h2 style="color: #10b981; margin-bottom: 20px; font-size: 24px;">✓ Claim Completed Successfully!</h2>
							
							<p style="margin-bottom: 15px;">Hello <strong>${
								claimRequest.username
							}</strong>,</p>
							
							<p style="margin-bottom: 20px;">Great news! Your claim request has been successfully processed and the funds have been transferred to your wallet.</p>
							
							<div style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.15), rgba(16, 185, 129, 0.05)); border-left: 4px solid #10b981; padding: 20px; margin: 25px 0; border-radius: 8px;">
								<h3 style="color: #10b981; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Transaction Details</h3>
								<table style="width: 100%; border-collapse: collapse;">
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Amount:</td>
										<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600; font-size: 18px;">${claimRequest.amountNTE.toFixed(
											6,
										)} NTE</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">USD Value:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right;">$${claimRequest.usdValue.toFixed(
											2,
										)}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Wallet Address:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 11px;">${claimRequest.walletAddress.substring(
											0,
											10,
										)}...${claimRequest.walletAddress.substring(
											claimRequest.walletAddress.length - 8,
										)}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Transaction Hash:</td>
										<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${
											withdrawalResult.transactionHash
										}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Block Number:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right;">#${
											withdrawalResult.blockNumber || "Pending"
										}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">New Balance:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${withdrawalResult.newBalance.toFixed(
											6,
										)} NTE</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
										<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600;">✓ Completed</td>
									</tr>
								</table>
							</div>
							
							<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
								<p style="margin: 0; color: #3b82f6;"><strong>ℹ️ Verify on Blockchain</strong></p>
								<p style="margin: 10px 0 0 0; color: #e1e4eb;">You can verify this transaction on the BSC blockchain explorer using the transaction hash above.</p>
							</div>
							
							<p style="margin-top: 25px; color: #9aa3b2; font-size: 14px;">The funds should appear in your wallet within a few minutes. If you have any questions, please contact our support team.</p>
							
							<div style="text-align: center; margin-top: 30px;">
								<a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">View Transactions</a>
								<a href="https://bscscan.com/tx/${
									withdrawalResult.transactionHash
								}" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View on BSCScan</a>
							</div>
						</div>
					`;

					const userEmailHtml = generateEmailFromTemplate(userEmailBody);

					systemNotificationService.sendToUser({
						username: claimRequest.username,
						email: user.email,
						subject: "Claim Completed Successfully - NodeMeta",
						title: "Claim Completed",
						body: `Your claim of ${claimRequest.amountNTE.toFixed(
							6,
						)} NTE has been completed. TX: ${withdrawalResult.transactionHash}`,
						html: userEmailHtml,
						pushPayload: {
							icon: "/icons/icon-192x192.png",
							badge: "/icons/badge-72x72.png",
							data: {
								url: "/dashboard/transactions",
								transactionHash: withdrawalResult.transactionHash,
								claimRequestId: claimRequest._id.toString(),
							},
						},
						sendEmail: true,
						sendPush: true,
					});
				} catch (error) {
					logger.error("Failed to send success notification to user:", {
						errorMessage: error.message,
						errorStack: error.stack,
						username: claimRequest.username,
						claimRequestId: claimRequest._id.toString(),
					});
				}

				// Send success notification to admins
				try {
					const adminEmailBody = `
						<div style="color: #e1e4eb; line-height: 1.8;">
							<h2 style="color: #10b981; margin-bottom: 20px; font-size: 24px;">✓ Claim Processed Successfully</h2>
							
							<p style="margin-bottom: 20px;">A claim request has been successfully processed and funds have been transferred.</p>
							
							<div style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.15), rgba(16, 185, 129, 0.05)); border-left: 4px solid #10b981; padding: 20px; margin: 25px 0; border-radius: 8px;">
								<h3 style="color: #10b981; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Transaction Details</h3>
								<table style="width: 100%; border-collapse: collapse;">
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Claim ID:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 12px;">${claimRequest._id.toString()}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Username:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
											claimRequest.username
										}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Amount:</td>
										<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600; font-size: 18px;">${claimRequest.amountNTE.toFixed(
											6,
										)} NTE</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">USD Value:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right;">$${claimRequest.usdValue.toFixed(
											2,
										)}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Wallet Address:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 11px;">${claimRequest.walletAddress.substring(
											0,
											10,
										)}...${claimRequest.walletAddress.substring(
											claimRequest.walletAddress.length - 8,
										)}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Transaction Hash:</td>
										<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${
											withdrawalResult.transactionHash
										}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Block Number:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right;">#${
											withdrawalResult.blockNumber || "Pending"
										}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Approved By:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right;">${
											autoApproval ? "Auto-Approval System" : adminUsername
										}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">New User Balance:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${withdrawalResult.newBalance.toFixed(
											6,
										)} NTE</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Claim Type:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; text-transform: capitalize;">${
											claimRequest.withdrawalType || "auto"
										}</td>
									</tr>
								</table>
							</div>
							
							${
								!autoApproval && claimRequest.queuedForApproval
									? `
							<div style="background: rgba(255, 165, 0, 0.1); border-left: 4px solid #ffa500; padding: 15px; margin: 20px 0; border-radius: 8px;">
								<p style="margin: 0; color: #ffa500;"><strong>⚠️ Manual Override</strong></p>
								<p style="margin: 10px 0 0 0; color: #e1e4eb;">This claim was manually processed by ${adminUsername}. It was previously queued for auto-approval at position ${claimRequest.queuePosition}.</p>
							</div>
							`
									: ""
							}
							
							<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
								<p style="margin: 0; color: #3b82f6;"><strong>ℹ️ Blockchain Verification</strong></p>
								<p style="margin: 10px 0 0 0; color: #e1e4eb;">Transaction has been confirmed on the BSC blockchain. View the transaction on BSCScan for full details.</p>
							</div>
							
							<div style="text-align: center; margin-top: 30px;">
								<a href="https://node-meta.com/dashboard/admin/claim-request" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">View All Claims</a>
								<a href="https://bscscan.com/tx/${
									withdrawalResult.transactionHash
								}" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View on BSCScan</a>
							</div>
						</div>
					`;

					const adminEmailHtml = generateEmailFromTemplate(adminEmailBody);

					systemNotificationService.sendToAdmin({
						subject: `Claim Completed: ${claimRequest.amountNTE.toFixed(
							2,
						)} NTE - ${claimRequest.username}`,
						title: "Claim Completed",
						body: `${
							claimRequest.username
						} claim of ${claimRequest.amountNTE.toFixed(
							6,
						)} NTE completed. TX: ${withdrawalResult.transactionHash}`,
						html: adminEmailHtml,
						pushPayload: {
							icon: "/icons/icon-192x192.png",
							badge: "/icons/badge-72x72.png",
							data: {
								url: "/admin/withdrawals",
								transactionHash: withdrawalResult.transactionHash,
								claimRequestId: claimRequest._id.toString(),
								withdrawalRequestId: withdrawalRequest._id.toString(),
							},
						},
						sendEmail: true,
						sendPush: true,
					});
				} catch (error) {
					logger.error("Failed to send success notification to admins:", {
						errorMessage: error.message,
						errorStack: error.stack,
						claimRequestId: claimRequest._id.toString(),
					});
				}

				// Build response message
				let responseMessage =
					"Claim approved and withdrawal processed successfully";
				if (!autoApproval && claimRequest.queuedForApproval) {
					responseMessage += ` (manual override - was queued at position ${claimRequest.queuePosition})`;
				}

				return {
					success: true,
					message: responseMessage,
					claimRequestId,
					withdrawalRequestId: withdrawalRequest._id,
					transactionHash: withdrawalResult.transactionHash,
					blockNumber: withdrawalResult.blockNumber,
					newBalance: withdrawalResult.newBalance,
					wasQueuedForAutoApproval:
						!autoApproval && claimRequest.queuedForApproval,
				};
			} catch (error) {
				logger.error(
					`TWO-PHASE COMMIT ROLLBACK | Blockchain transfer failed for claim ${claimRequestId}:`,
					{
						error: error.message,
						stack: error.stack,
						username: claimRequest.username,
						amount: claimRequest.amountNTE,
					},
				);

				// Rollback with session support and retry mechanism
				try {
					const currentUserBalance = await withSession(
						User.findById(user._id).select(
							"walletBalance balanceSnapshot balanceChecksum",
						),
						session,
					);

					// Verify balance integrity using snapshot and checksum
					const balanceModified =
						currentUserBalance.walletBalance <
						(balanceLockResult.balanceSnapshot ||
							balanceLockResult.walletBalance);

					// Check if balance was already deducted
					if (balanceModified) {
						logger.warn(
							`ROLLBACK | Balance was deducted, restoring ${claimRequest.amountNTE} NTE to user ${user.username}`,
						);

						// Restore balance atomically WITH SESSION for ACID guarantee
						const rollbackResult = await withSession(
							User.findOneAndUpdate(
								{ _id: user._id },
								{
									$inc: { walletBalance: claimRequest.amountNTE, __v: 1 },
									$set: {
										balanceLocked: false,
										balanceLockedAt: null,
										balanceLockedBy: null,
										balanceLockedAmount: 0,
										balanceSnapshot: null,
										balanceChecksum: null,
										balanceLockNonce: null,
									},
								},
								{ new: true },
							),
							session, // Use transaction session
						);

						if (rollbackResult) {
							logger.info(
								` ROLLBACK SUCCESS | Restored ${claimRequest.amountNTE} NTE. New balance: ${rollbackResult.walletBalance}`,
							);

							// Log rollback in audit trail
							await auditLogService.logSecurityViolation({
								type: "WITHDRAWAL_ROLLBACK",
								username: claimRequest.username,
								description: `Balance restored after blockchain transfer failure: ${error.message}`,
								severity: "HIGH",
								clientIP: data.clientIP,
								userAgent: data.userAgent,
								metadata: {
									claimRequestId,
									amountRestored: claimRequest.amountNTE,
									previousBalance: currentUserBalance.walletBalance,
									newBalance: rollbackResult.walletBalance,
									blockchainError: error.message,
								},
							});
						}
					} else {
						// Balance not deducted yet, just unlock WITH SESSION
						await withSession(
							User.findByIdAndUpdate(user._id, {
								$set: {
									balanceLocked: false,
									balanceLockedAt: null,
									balanceLockedBy: null,
									balanceLockedAmount: 0,
									balanceSnapshot: null,
									balanceChecksum: null,
									balanceLockNonce: null,
								},
							}),
							session,
						);
						logger.info(
							`ROLLBACK | Balance not deducted, lock released for user ${user.username}`,
						);
					}
				} catch (rollbackError) {
					logger.error(`ROLLBACK FAILED for user ${user._id}:`, {
						rollbackError: rollbackError.message,
						originalError: error.message,
						sessionActive: !!session,
					});

					// Use PERSISTENT database-backed rollback queue
					const { RollbackQueue } = require("./rollbackQueue.model");

					try {
						const rollbackEntry = await RollbackQueue.create({
							userId: user._id,
							username: user.username,
							claimRequestId: claimRequestId,
							withdrawalRequestId: withdrawalRequest._id,
							amountNTE: claimRequest.amountNTE,
							originalError: error.message,
							rollbackError: rollbackError.message,
							status: "pending",
							priority: 10,
							nextRetryAt: new Date(Date.now() + 60000),
							metadata: {
								clientIP: data.clientIP,
								userAgent: data.userAgent,
								balanceSnapshot: balanceLockResult.balanceSnapshot,
								balanceChecksum: balanceLockResult.balanceChecksum,
								lockNonce: balanceLockResult.balanceLockNonce,
								originalApprovalBy: adminUsername,
							},
						});

						logger.info(
							` Rollback queued in persistent database (ID: ${rollbackEntry._id})`,
						);

						// Also try in-memory queue as backup
						try {
							const { enqueueJob } = require("../../../queues/job.queue");
							await enqueueJob(
								"rollback-retry",
								{
									rollbackQueueId: rollbackEntry._id.toString(),
									userId: user._id.toString(),
									username: user.username,
									amountNTE: claimRequest.amountNTE,
								},
								{ priority: 10, attempts: 5 },
							);
							logger.info("Rollback also queued in memory for immediate retry");
						} catch (queueError) {
							logger.warn(
								"In-memory queue failed (OK - persistent DB queue active):",
								queueError.message,
							);
						}
					} catch (queueError) {
						logger.error(
							"FATAL: Failed to queue rollback in persistent database:",
							queueError.message,
						);
					}

					await auditLogService.logSecurityViolation({
						type: "CRITICAL_ROLLBACK_FAILURE",
						username: claimRequest.username,
						description: `Failed to rollback balance after blockchain failure. Queued in persistent database for automated retry. Manual verification required!`,
						severity: "CRITICAL",
						clientIP: data.clientIP,
						userAgent: data.userAgent,
						metadata: {
							claimRequestId,
							userId: user._id,
							amountNTE: claimRequest.amountNTE,
							blockchainError: error.message,
							rollbackError: rollbackError.message,
							persistentQueueActive: true,
							balanceSnapshot: balanceLockResult.balanceSnapshot,
						},
					});
				}

				await ClaimRequest.findByIdAndUpdate(claimRequestId, {
					status: "failed",
					processedAt: new Date(),
					withdrawalError: error.message,
					$unset: {
						processingLock: "",
						processingLockedAt: "",
						processingLockedBy: "",
					},
				});

				await auditLogService.logWithdrawalProcessing({
					withdrawalRequestId: withdrawalRequest._id.toString(),
					claimRequestId: claimRequestId.toString(),
					username: claimRequest.username,
					amountNTE: claimRequest.amountNTE,
					toAddress: claimRequest.walletAddress,
					status: "failed",
					transactionHash: null,
					blockNumber: null,
					error: error.message,
					idempotencyKey,
					balanceUnlocked: true,
				});
				await auditLogService.logSecurityViolation({
					type: "WITHDRAWAL_PROCESSING_FAILED",
					username: claimRequest.username,
					description: `Withdrawal processing failed for claim ${claimRequestId}: ${error.message}`,
					severity: "HIGH",
					clientIP: data.clientIP,
					userAgent: data.userAgent,
					metadata: {
						claimRequestId,
						withdrawalRequestId: withdrawalRequest._id.toString(),
						error: error.message,
						idempotencyKey,
					},
				});

				// Send failure notification to user
				try {
					const userEmailBody = `
						<div style="color: #e1e4eb; line-height: 1.8;">
							<h2 style="color: #ef4444; margin-bottom: 20px; font-size: 24px;">⚠️ Claim Processing Failed</h2>
							
							<p style="margin-bottom: 15px;">Hello <strong>${
								claimRequest.username
							}</strong>,</p>
							
							<p style="margin-bottom: 20px;">We encountered an issue while processing your claim request. Your balance has been restored and no funds were deducted.</p>
							
							<div style="background: linear-gradient(135deg, rgba(239, 68, 68, 0.15), rgba(239, 68, 68, 0.05)); border-left: 4px solid #ef4444; padding: 20px; margin: 25px 0; border-radius: 8px;">
								<h3 style="color: #ef4444; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Request Details</h3>
								<table style="width: 100%; border-collapse: collapse;">
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Amount:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600; font-size: 18px;">${claimRequest.amountNTE.toFixed(
											6,
										)} NTE</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">USD Value:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right;">$${claimRequest.usdValue.toFixed(
											2,
										)}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Wallet Address:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 11px;">${claimRequest.walletAddress.substring(
											0,
											10,
										)}...${claimRequest.walletAddress.substring(
											claimRequest.walletAddress.length - 8,
										)}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
										<td style="padding: 8px 0; color: #ef4444; text-align: right; font-weight: 600;">✗ Failed</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Balance Status:</td>
										<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600;">✓ Restored</td>
									</tr>
								</table>
							</div>
							
							<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
								<p style="margin: 0; color: #3b82f6;"><strong>ℹ️ What Happened?</strong></p>
								<p style="margin: 10px 0 0 0; color: #e1e4eb;">There was a technical issue during the blockchain transfer process. Your balance has been fully restored and you can try submitting a new claim request.</p>
							</div>
							
							<div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
								<p style="margin: 0; color: #eab308;"><strong>💡 Next Steps</strong></p>
								<p style="margin: 10px 0 0 0; color: #e1e4eb;">Please verify your wallet address is correct and try submitting a new claim request. If the issue persists, please contact our support team with your request ID.</p>
							</div>
							
							<div style="text-align: center; margin-top: 30px;">
								<a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">Try Again</a>
								<a href="https://node-meta.com" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">Contact Support</a>
							</div>
						</div>
					`;

					const userEmailHtml = generateEmailFromTemplate(userEmailBody);

					systemNotificationService.sendToUser({
						username: claimRequest.username,
						email: user.email,
						subject: "Claim Processing Failed - NodeMeta",
						title: "Claim Failed",
						body: `Your claim request of ${claimRequest.amountNTE.toFixed(
							6,
						)} NTE failed. Your balance has been restored.`,
						html: userEmailHtml,
						pushPayload: {
							icon: "/icons/icon-192x192.png",
							badge: "/icons/badge-72x72.png",
							data: {
								url: "/dashboard/claim",
								claimRequestId: claimRequest._id.toString(),
								status: "failed",
							},
						},
						sendEmail: true,
						sendPush: true,
					});
				} catch (notificationError) {
					logger.error("Failed to send failure notification to user:", {
						errorMessage: notificationError.message,
						errorStack: notificationError.stack,
						username: claimRequest.username,
						claimRequestId: claimRequest._id.toString(),
					});
				}

				// Send failure notification to admins
				try {
					const adminEmailBody = `
						<div style="color: #e1e4eb; line-height: 1.8;">
							<h2 style="color: #ef4444; margin-bottom: 20px; font-size: 24px;">⚠️ Claim Processing Failed</h2>
							
							<p style="margin-bottom: 20px;">A claim request failed during blockchain transfer. The user's balance has been restored.</p>
							
							<div style="background: linear-gradient(135deg, rgba(239, 68, 68, 0.15), rgba(239, 68, 68, 0.05)); border-left: 4px solid #ef4444; padding: 20px; margin: 25px 0; border-radius: 8px;">
								<h3 style="color: #ef4444; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Failure Details</h3>
								<table style="width: 100%; border-collapse: collapse;">
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Claim ID:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 12px;">${claimRequest._id.toString()}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Withdrawal ID:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 12px;">${withdrawalRequest._id.toString()}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Username:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
											claimRequest.username
										}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Amount:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600; font-size: 18px;">${claimRequest.amountNTE.toFixed(
											6,
										)} NTE</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">USD Value:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right;">$${claimRequest.usdValue.toFixed(
											2,
										)}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Wallet Address:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 11px;">${claimRequest.walletAddress.substring(
											0,
											10,
										)}...${claimRequest.walletAddress.substring(
											claimRequest.walletAddress.length - 8,
										)}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Approved By:</td>
										<td style="padding: 8px 0; color: #fff; text-align: right;">${
											autoApproval ? "Auto-Approval System" : adminUsername
										}</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
										<td style="padding: 8px 0; color: #ef4444; text-align: right; font-weight: 600;">✗ Failed</td>
									</tr>
									<tr>
										<td style="padding: 8px 0; color: #9aa3b2;">Balance Restored:</td>
										<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600;">✓ Yes</td>
									</tr>
								</table>
							</div>
							
							<div style="background: rgba(239, 68, 68, 0.1); border-left: 4px solid #ef4444; padding: 15px; margin: 20px 0; border-radius: 8px;">
								<p style="margin: 0; color: #ef4444;"><strong>⚠️ Error Details</strong></p>
								<p style="margin: 10px 0 0 0; color: #e1e4eb; font-family: monospace; font-size: 13px; word-break: break-word;">${
									error.message
								}</p>
							</div>
							
							<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
								<p style="margin: 0; color: #3b82f6;"><strong>ℹ️ Action Taken</strong></p>
								<p style="margin: 10px 0 0 0; color: #e1e4eb;">The user's balance has been automatically restored. The withdrawal can be retried or investigated for technical issues.</p>
							</div>
							
							<div style="text-align: center; margin-top: 30px;">
								<a href="https://node-meta.com/dashboard/admin/claim-request" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">View All Claims</a>
								<a href="https://node-meta.com/dashboard/admin/claim-request" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View Claim Details</a>
							</div>
						</div>
					`;

					const adminEmailHtml = generateEmailFromTemplate(adminEmailBody);

					systemNotificationService.sendToAdmin({
						subject: `Claim Failed: ${claimRequest.amountNTE.toFixed(
							2,
						)} NTE - ${claimRequest.username}`,
						title: "Claim Failed",
						body: `${
							claimRequest.username
						} claim of ${claimRequest.amountNTE.toFixed(
							6,
						)} NTE failed. Balance restored. Error: ${error.message}`,
						html: adminEmailHtml,
						pushPayload: {
							icon: "/icons/icon-192x192.png",
							badge: "/icons/badge-72x72.png",
							data: {
								url: "/admin/withdrawals",
								claimRequestId: claimRequest._id.toString(),
								withdrawalRequestId: withdrawalRequest._id.toString(),
								status: "failed",
							},
						},
						sendEmail: true,
						sendPush: true,
					});
				} catch (notificationError) {
					logger.error("Failed to send failure notification to admins:", {
						errorMessage: notificationError.message,
						errorStack: notificationError.stack,
						claimRequestId: claimRequest._id.toString(),
					});
				}

				return {
					success: false,
					message: "Claim approved but claim failed",
					error: error.message,
					claimRequestId,
					withdrawalRequestId: withdrawalRequest._id,
				};
			}
		} catch (error) {
			if (session) {
				await session.abortTransaction();
			}
			logger.error(`Claim approval failed ${claimRequestId}:`, {
				error: error.message,
				stack: error.stack,
				username: claimRequest?.username,
				amount: claimRequest?.amountNTE,
			});

			// Release balance lock if it was acquired
			if (user && user._id) {
				try {
					const userCheck = await User.findById(user._id).select(
						"balanceLocked balanceLockedBy",
					);
					if (
						userCheck?.balanceLocked &&
						userCheck?.balanceLockedBy?.includes(`claim_${claimRequestId}`)
					) {
						await User.findByIdAndUpdate(user._id, {
							$set: {
								balanceLocked: false,
								balanceLockedAt: null,
								balanceLockedBy: null,
								balanceLockedAmount: 0,
							},
						});
						logger.info(
							`BALANCE UNLOCK | Released on error for user ${user._id}`,
						);
					}
				} catch (unlockError) {
					logger.error(
						`Failed to check/release balance lock on error for user ${user._id}:`,
						unlockError,
					);
				}
			}

			// Release processing lock on error
			if (claimRequest) {
				try {
					await ClaimRequest.findByIdAndUpdate(claimRequestId, {
						$unset: {
							processingLock: "",
							processingLockedAt: "",
							processingLockedBy: "",
						},
					});
					logger.info(
						`PROCESSING LOCK | Released on error for claim ${claimRequestId}`,
					);
				} catch (unlockError) {
					logger.error(
						`Failed to release processing lock on error:`,
						unlockError,
					);
				}
			}

			// Comprehensive audit log for approval failure
			await auditLogService.logClaimApproval({
				claimRequestId: claimRequestId?.toString(),
				adminUsername,
				adminUserId,
				username: claimRequest?.username,
				amountNTE: claimRequest?.amountNTE,
				usdValue: claimRequest?.usdValue,
				success: false,
				phase: "failed",
				error: error.message,
				transactionHash: null,
				clientIP: data.clientIP,
				userAgent: data.userAgent,
			});

			throw error;
		} finally {
			if (session) {
				session.endSession();
			}
		}
	},

	/**
	 * Reject claim request
	 */
	async rejectClaimRequest(data) {
		const {
			claimRequestId,
			masterPassword,
			rejectionReason,
			notes,
			adminUsername,
		} = data;

		try {
			// 1. Verify admin master password
			const adminConfig = await AdminConfig.findOne({
				type: "master_password",
			});

			if (!adminConfig || !adminConfig.passwordHash) {
				throw new ApiError(500, "Admin master password not configured");
			}

			const isPasswordValid = await bcrypt.compare(
				masterPassword.trim(),
				adminConfig.passwordHash,
			);

			// Log master password attempt
			await auditLogService.logMasterPasswordAttempt({
				adminUsername,
				adminUserId: data.adminUserId,
				action: "claim_rejection",
				success: isPasswordValid,
				clientIP: data.clientIP,
				userAgent: data.userAgent,
			});

			if (!isPasswordValid) {
				logger.warn("Failed claim rejection - invalid master password", {
					adminUsername,
					claimRequestId,
				});
				throw new ApiError(401, "Invalid master password");
			}

			// 2. Get claim request and check processing lock
			const claimRequest = await ClaimRequest.findById(claimRequestId);

			if (!claimRequest) {
				throw new ApiError(404, "Claim request not found");
			}

			if (claimRequest.status !== "pending") {
				throw new ApiError(400, `Claim request already ${claimRequest.status}`);
			}

			// Check if claim is currently being processed by queue
			if (claimRequest.processingLock === true) {
				const lockedBy = claimRequest.processingLockedBy || "another admin";
				const lockedAt = claimRequest.processingLockedAt;
				const lockAge = lockedAt
					? Math.floor((Date.now() - new Date(lockedAt).getTime()) / 1000)
					: 0;

				throw new ApiError(
					409,
					`Cannot reject - this claim is currently being processed by ${lockedBy}${
						lockAge > 0 ? ` (${lockAge}s ago)` : ""
					}. Please wait or use queue management to skip it first.`,
				);
			}

			// Log if manually rejecting a queued claim
			if (claimRequest.queuedForApproval && claimRequest.queuePosition) {
				logger.warn(
					`MANUAL REJECTION | Admin ${adminUsername} rejecting claim ${claimRequestId} that was queued for auto-approval (position ${claimRequest.queuePosition})`,
				);
			}

			// 3. Validate rejection reason
			if (!rejectionReason || rejectionReason.trim().length < 10) {
				throw new ApiError(
					400,
					"Rejection reason must be at least 10 characters",
				);
			}

			// 4. Update claim request to rejected and clear queue status
			await ClaimRequest.findByIdAndUpdate(claimRequestId, {
				status: "rejected",
				rejectedAt: new Date(),
				rejectedBy: adminUsername,
				rejectionReason: rejectionReason.trim(),
				notes: notes || null,
				processedAt: new Date(),
				// Clear queue status
				queuedForApproval: false,
				queuePosition: null,
				$unset: {
					processingLock: "",
					processingLockedAt: "",
					processingLockedBy: "",
				},
			});

			logger.info(
				`Claim request ${claimRequestId} rejected by ${adminUsername}: ${rejectionReason}`,
			);

			// Audit log rejection
			await auditLogService.logClaimRejection({
				claimRequestId: claimRequestId.toString(),
				adminUsername,
				adminUserId: data.adminUserId,
				username: claimRequest.username,
				amountNTE: claimRequest.amountNTE,
				rejectionReason: rejectionReason.trim(),
				success: true,
				error: null,
				clientIP: data.clientIP,
				userAgent: data.userAgent,
			});

			// Get user for email notification
			const user = await User.findOne({
				username: claimRequest.username,
			}).select("email");

			// Send rejection notification to user
			try {
				const userEmailBody = `
					<div style="color: #e1e4eb; line-height: 1.8;">
						<h2 style="color: #f59e0b; margin-bottom: 20px; font-size: 24px;">⚠️ Claim Request Rejected</h2>
						
						<p style="margin-bottom: 15px;">Hello <strong>${
							claimRequest.username
						}</strong>,</p>
						
						<p style="margin-bottom: 20px;">We regret to inform you that your claim request has been reviewed and rejected by our admin team.</p>
						
						<div style="background: linear-gradient(135deg, rgba(245, 158, 11, 0.15), rgba(245, 158, 11, 0.05)); border-left: 4px solid #f59e0b; padding: 20px; margin: 25px 0; border-radius: 8px;">
							<h3 style="color: #f59e0b; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Request Details</h3>
							<table style="width: 100%; border-collapse: collapse;">
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Amount:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600; font-size: 18px;">${claimRequest.amountNTE.toFixed(
										6,
									)} NTE</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">USD Value:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">$${claimRequest.usdValue.toFixed(
										2,
									)}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Wallet Address:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 11px;">${claimRequest.walletAddress.substring(
										0,
										10,
									)}...${claimRequest.walletAddress.substring(
										claimRequest.walletAddress.length - 8,
									)}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Request Date:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${new Date(
										claimRequest.createdAt,
									).toLocaleDateString()}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
									<td style="padding: 8px 0; color: #f59e0b; text-align: right; font-weight: 600;">✗ Rejected</td>
								</tr>
							</table>
						</div>
						
						<div style="background: rgba(239, 68, 68, 0.1); border-left: 4px solid #ef4444; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #ef4444;"><strong>⚠️ Rejection Reason</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">${rejectionReason.trim()}</p>
						</div>
						
						${
							notes
								? `
						<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #3b82f6;"><strong>ℹ️ Additional Notes</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">${notes}</p>
						</div>
						`
								: ""
						}
						
						<div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #eab308;"><strong>💡 What's Next?</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">Please review the rejection reason and address any issues. You may submit a new withdrawal request once you've resolved the concerns mentioned above. If you have questions, please contact our support team.</p>
						</div>
						
						<div style="text-align: center; margin-top: 30px;">
							<a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">Submit New Request</a>
							<a href="https://node-meta.com" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">Contact Support</a>
						</div>
					</div>
				`;

				const userEmailHtml = generateEmailFromTemplate(userEmailBody);

				systemNotificationService.sendToUser({
					username: claimRequest.username,
					email: user?.email,
					subject: "Claim Request Rejected - NodeMeta",
					title: "Request Rejected",
					body: `Your claim request of ${claimRequest.amountNTE.toFixed(
						6,
					)} NTE has been rejected. Reason: ${rejectionReason.trim()}`,
					html: userEmailHtml,
					pushPayload: {
						icon: "/icons/icon-192x192.png",
						badge: "/icons/badge-72x72.png",
						data: {
							url: "/dashboard/claim",
							claimRequestId: claimRequest._id.toString(),
							status: "rejected",
						},
					},
					sendEmail: true,
					sendPush: true,
				});
			} catch (notificationError) {
				logger.error("Failed to send rejection notification to user:", {
					errorMessage: notificationError.message,
					errorStack: notificationError.stack,
					username: claimRequest.username,
					claimRequestId: claimRequest._id.toString(),
				});
				// Don't throw - notification failure shouldn't break the rejection
			}

			// Send rejection notification to admins
			try {
				const adminEmailBody = `
					<div style="color: #e1e4eb; line-height: 1.8;">
						<h2 style="color: #f59e0b; margin-bottom: 20px; font-size: 24px;">✓ Claim Request Rejected</h2>
						
						<p style="margin-bottom: 20px;">A claim request has been rejected by <strong>${adminUsername}</strong>.</p>
						
						<div style="background: linear-gradient(135deg, rgba(245, 158, 11, 0.15), rgba(245, 158, 11, 0.05)); border-left: 4px solid #f59e0b; padding: 20px; margin: 25px 0; border-radius: 8px;">
							<h3 style="color: #f59e0b; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Rejection Details</h3>
							<table style="width: 100%; border-collapse: collapse;">
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Claim ID:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 12px;">${claimRequest._id.toString()}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Username:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
										claimRequest.username
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Amount:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600; font-size: 18px;">${claimRequest.amountNTE.toFixed(
										6,
									)} NTE</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">USD Value:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">$${claimRequest.usdValue.toFixed(
										2,
									)}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Wallet Address:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 11px;">${claimRequest.walletAddress.substring(
										0,
										10,
									)}...${claimRequest.walletAddress.substring(
										claimRequest.walletAddress.length - 8,
									)}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Request Date:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${new Date(
										claimRequest.createdAt,
									).toLocaleDateString()}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Rejected By:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${adminUsername}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Rejected At:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${new Date().toLocaleString()}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
									<td style="padding: 8px 0; color: #f59e0b; text-align: right; font-weight: 600;">✗ Rejected</td>
								</tr>
							</table>
						</div>
						
						<div style="background: rgba(239, 68, 68, 0.1); border-left: 4px solid #ef4444; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #ef4444;"><strong>⚠️ Rejection Reason</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">${rejectionReason.trim()}</p>
						</div>
						
						${
							notes
								? `
						<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #3b82f6;"><strong>ℹ️ Additional Notes</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">${notes}</p>
						</div>
						`
								: ""
						}
						
						${
							claimRequest.queuedForApproval
								? `
						<div style="background: rgba(255, 165, 0, 0.1); border-left: 4px solid #ffa500; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #ffa500;"><strong>⚠️ Manual Rejection</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">This claim was manually rejected by ${adminUsername}. It was previously queued for auto-approval at position ${claimRequest.queuePosition}.</p>
						</div>
						`
								: ""
						}
						
						<div style="text-align: center; margin-top: 30px;">
							<a href="https://node-meta.com/dashboard/admin/claim-request" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">View All Claims</a>
							<a href="https://node-meta.com/dashboard/admin/claim-request" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View Claim Details</a>
						</div>
					</div>
				`;

				const adminEmailHtml = generateEmailFromTemplate(adminEmailBody);

				systemNotificationService.sendToAdmin({
					subject: `Claim Rejected: ${claimRequest.amountNTE.toFixed(
						2,
					)} NTE - ${claimRequest.username}`,
					title: "Claim Rejected",
					body: `${
						claimRequest.username
					} claim of ${claimRequest.amountNTE.toFixed(
						6,
					)} NTE rejected by ${adminUsername}. Reason: ${rejectionReason.trim()}`,
					html: adminEmailHtml,
					pushPayload: {
						icon: "/icons/icon-192x192.png",
						badge: "/icons/badge-72x72.png",
						data: {
							url: "/admin/claims",
							claimRequestId: claimRequest._id.toString(),
							status: "rejected",
							rejectedBy: adminUsername,
						},
					},
					sendEmail: true,
					sendPush: true,
				});
			} catch (notificationError) {
				logger.error("Failed to send rejection notification to admins:", {
					errorMessage: notificationError.message,
					errorStack: notificationError.stack,
					claimRequestId: claimRequest._id.toString(),
				});
				// Don't throw - notification failure shouldn't break the rejection
			}

			// Build response message
			let responseMessage = "Claim request rejected successfully";
			if (claimRequest.queuedForApproval) {
				responseMessage += ` (was queued at position ${claimRequest.queuePosition})`;
			}

			return {
				success: true,
				message: responseMessage,
				claimRequestId,
				wasQueuedForAutoApproval: claimRequest.queuedForApproval || false,
			};
		} catch (error) {
			logger.error(`Claim rejection failed:`, error);

			// Audit log failed rejection
			const claimRequest =
				await ClaimRequest.findById(claimRequestId).select(
					"username amountNTE",
				);

			await auditLogService.logClaimRejection({
				claimRequestId: claimRequestId.toString(),
				adminUsername,
				adminUserId: data.adminUserId,
				username: claimRequest?.username || "unknown",
				amountNTE: claimRequest?.amountNTE || 0,
				rejectionReason: rejectionReason?.trim() || "unknown",
				success: false,
				error: error.message,
				clientIP: data.clientIP,
				userAgent: data.userAgent,
			});

			throw error;
		}
	},

	/**
	 * Get all claim requests with filters
	 */
	async getClaimRequests(filters = {}) {
		try {
			const {
				page = 1,
				limit = 50,
				status,
				username,
				startDate,
				endDate,
			} = filters;

			const query = {};

			if (status && status !== "all") {
				query.status = status;
			}

			if (username) {
				query.username = { $regex: username, $options: "i" };
			}

			if (startDate || endDate) {
				query.requestedAt = {};
				if (startDate) {
					query.requestedAt.$gte = new Date(startDate);
				}
				if (endDate) {
					query.requestedAt.$lte = new Date(endDate);
				}
			}

			const total = await ClaimRequest.countDocuments(query);

			const claims = await ClaimRequest.find(query)
				.sort({ requestedAt: -1 })
				.skip((page - 1) * limit)
				.limit(limit)
				.lean();

			// Get statistics
			const stats = await ClaimRequest.aggregate([
				{
					$group: {
						_id: "$status",
						count: { $sum: 1 },
						totalAmount: { $sum: "$amountNTE" },
						totalUSD: { $sum: "$usdValue" },
					},
				},
			]);

			return {
				claims,
				pagination: {
					page,
					limit,
					total,
					totalPages: Math.ceil(total / limit),
				},
				stats: stats.reduce((acc, stat) => {
					acc[stat._id] = {
						count: stat.count,
						totalAmount: stat.totalAmount,
						totalUSD: stat.totalUSD,
					};
					return acc;
				}, {}),
			};
		} catch (error) {
			logger.error(`Error getting claim requests:`, error);
			throw error;
		}
	},

	/**
	 * Get claim request by ID
	 */
	async getClaimRequestById(claimRequestId) {
		try {
			const claimRequest = await ClaimRequest.findById(claimRequestId).lean();
			return claimRequest;
		} catch (error) {
			logger.error(`Error getting claim request ${claimRequestId}:`, error);
			throw error;
		}
	},

	/**
	 * Get number of claims ahead in queue
	 */
	async getClaimsAheadInQueue(queuedAt) {
		try {
			const claimsAhead = await ClaimRequest.countDocuments({
				status: "pending",
				queuedForApproval: true,
				queuedAt: { $lt: queuedAt },
			});
			return claimsAhead;
		} catch (error) {
			logger.error(`Error counting claims ahead in queue:`, error);
			throw error;
		}
	},

	// ==================== ADMIN QUEUE MANAGEMENT ====================

	/**
	 * Get queue statistics for admin dashboard
	 */
	async getQueueStats() {
		try {
			const modelStats = await ClaimRequest.getQueueStats();

			// Get recent processing history
			const last24Hours = new Date(Date.now() - TIME_CONSTANTS.ONE_DAY_MS);
			const recentProcessed = await ClaimRequest.aggregate([
				{
					$match: {
						processedAt: { $gte: last24Hours },
						status: { $in: ["approved", "rejected", "completed"] },
					},
				},
				{
					$group: {
						_id: "$status",
						count: { $sum: 1 },
						totalNTE: { $sum: "$amountNTE" },
						totalUSD: { $sum: "$usdValue" },
					},
				},
			]);

			// Get average processing time
			const avgProcessingTime = await ClaimRequest.aggregate([
				{
					$match: {
						processedAt: { $exists: true, $ne: null },
						requestedAt: { $exists: true },
						status: { $in: ["approved", "completed"] },
					},
				},
				{
					$project: {
						processingTime: { $subtract: ["$processedAt", "$requestedAt"] },
					},
				},
				{
					$group: {
						_id: null,
						avgTime: { $avg: "$processingTime" },
						minTime: { $min: "$processingTime" },
						maxTime: { $max: "$processingTime" },
					},
				},
			]);

			return {
				...modelStats,
				recent24Hours: recentProcessed.reduce((acc, item) => {
					acc[item._id] = {
						count: item.count,
						totalNTE: item.totalNTE,
						totalUSD: item.totalUSD,
					};
					return acc;
				}, {}),
				processingTimes: avgProcessingTime[0] || {
					avgTime: 0,
					minTime: 0,
					maxTime: 0,
				},
			};
		} catch (error) {
			logger.error("Error getting queue stats:", error);
			throw error;
		}
	},

	/**
	 * Get pending claims in queue for admin
	 */
	async getPendingQueue(options = {}) {
		try {
			const {
				page = 1,
				limit = 50,
				sortBy = "queuedAt",
				sortOrder = 1,
			} = options;

			const query = {
				status: "pending",
				queuedForApproval: true,
			};

			const claims = await ClaimRequest.find(query)
				.sort({ [sortBy]: sortOrder })
				.skip((page - 1) * limit)
				.limit(limit)
				.lean();

			const total = await ClaimRequest.countDocuments(query);

			return {
				claims,
				pagination: {
					page,
					limit,
					total,
					totalPages: Math.ceil(total / limit),
				},
			};
		} catch (error) {
			logger.error("Error getting pending queue:", error);
			throw error;
		}
	},

	/**
	 * Get queue processing history
	 */
	async getQueueHistory(options = {}) {
		try {
			const { page = 1, limit = 50, startDate, endDate, status } = options;

			const query = {
				queuedForApproval: true,
				status: { $ne: "pending" },
			};

			if (startDate || endDate) {
				query.processedAt = {};
				if (startDate) query.processedAt.$gte = new Date(startDate);
				if (endDate) query.processedAt.$lte = new Date(endDate);
			}

			if (status) {
				query.status = status;
			}

			const claims = await ClaimRequest.find(query)
				.sort({ processedAt: -1 })
				.skip((page - 1) * limit)
				.limit(limit)
				.lean();

			const total = await ClaimRequest.countDocuments(query);

			return {
				claims,
				pagination: {
					page,
					limit,
					total,
					totalPages: Math.ceil(total / limit),
				},
			};
		} catch (error) {
			logger.error("Error getting queue history:", error);
			throw error;
		}
	},

	/**
	 * Get failed auto-approval claims
	 */
	async getFailedClaims(options = {}) {
		try {
			const { page = 1, limit = 50 } = options;

			const query = {
				status: "pending",
				queuedForApproval: true,
				$or: [
					{ queueStatus: "failed" },
					{ queueStatus: "manual_required" },
					{ autoApprovalAttempts: { $gt: 0 } },
				],
			};

			const claims = await ClaimRequest.find(query)
				.sort({ lastAutoApprovalAttempt: -1 })
				.skip((page - 1) * limit)
				.limit(limit)
				.lean();

			const total = await ClaimRequest.countDocuments(query);

			return {
				claims,
				pagination: {
					page,
					limit,
					total,
					totalPages: Math.ceil(total / limit),
				},
			};
		} catch (error) {
			logger.error("Error getting failed claims:", error);
			throw error;
		}
	},

	/**
	 * Get claims requiring manual review
	 */
	async getManualReviewClaims(options = {}) {
		try {
			const { page = 1, limit = 50 } = options;

			const query = {
				status: "pending",
				requiresManualReview: true,
			};

			const claims = await ClaimRequest.find(query)
				.sort({ manualReviewRequestedAt: -1, queuedAt: 1 })
				.skip((page - 1) * limit)
				.limit(limit)
				.lean();

			const total = await ClaimRequest.countDocuments(query);

			return {
				claims,
				pagination: {
					page,
					limit,
					total,
					totalPages: Math.ceil(total / limit),
				},
			};
		} catch (error) {
			logger.error("Error getting manual review claims:", error);
			throw error;
		}
	},

	/**
	 * Retry a failed claim
	 */
	async retryFailedClaim(claimRequestId, adminUsername) {
		try {
			const claim = await ClaimRequest.findById(claimRequestId);

			if (!claim) {
				throw new ApiError(404, "Claim request not found");
			}

			if (claim.status !== "pending") {
				throw new ApiError(400, "Can only retry pending claims");
			}

			await ClaimRequest.findByIdAndUpdate(claimRequestId, {
				$set: {
					queueStatus: "waiting",
					skipAutoApproval: false,
					requiresManualReview: false,
					manualReviewReason: null,
					autoApprovalError: null,
					scheduledProcessingTime: new Date(), // Process immediately
				},
				$inc: { maxAutoApprovalAttempts: 1 }, // Give one more attempt
			});

			logger.info(
				`Claim ${claimRequestId} queued for retry by ${adminUsername}`,
			);

			await auditLogService.logAdminAction({
				adminUsername,
				action: "RETRY_FAILED_CLAIM",
				targetId: claimRequestId,
				description: `Queued claim for retry`,
			});

			return { success: true, message: "Claim queued for retry" };
		} catch (error) {
			logger.error(`Error retrying claim ${claimRequestId}:`, error);
			throw error;
		}
	},

	/**
	 * Skip a claim from auto-approval
	 */
	async skipClaim(claimRequestId, adminUsername, reason) {
		try {
			const claim = await ClaimRequest.findById(claimRequestId);

			if (!claim) {
				throw new ApiError(404, "Claim request not found");
			}

			if (claim.status !== "pending") {
				throw new ApiError(400, "Can only skip pending claims");
			}

			await ClaimRequest.findByIdAndUpdate(claimRequestId, {
				$set: {
					queueStatus: "skipped",
					skipAutoApproval: true,
					requiresManualReview: true,
					manualReviewReason: reason || "Skipped by admin",
					manualReviewRequestedBy: adminUsername,
					manualReviewRequestedAt: new Date(),
				},
			});

			logger.info(
				`Claim ${claimRequestId} skipped by ${adminUsername}: ${reason}`,
			);

			await auditLogService.logAdminAction({
				adminUsername,
				action: "SKIP_CLAIM",
				targetId: claimRequestId,
				description: `Skipped claim from auto-approval: ${reason}`,
			});

			return { success: true, message: "Claim skipped from auto-approval" };
		} catch (error) {
			logger.error(`Error skipping claim ${claimRequestId}:`, error);
			throw error;
		}
	},

	/**
	 * Prioritize a claim in queue
	 */
	async prioritizeClaim(
		claimRequestId,
		adminUsername,
		priority = 10,
		clientIP = null,
		userAgent = null,
	) {
		try {
			const { AdminPriorityChange } = require("./adminPriorityChange.model");

			const claim = await ClaimRequest.findById(claimRequestId);

			if (!claim) {
				throw new ApiError(404, "Claim request not found");
			}

			if (claim.status !== "pending") {
				throw new ApiError(400, "Can only prioritize pending claims");
			}

			if (priority < 1 || priority > 50) {
				throw new ApiError(400, "Priority must be between 1 and 50");
			}

			const priorityChange = priority - (claim.queuePriority || 0);
			const currentPriority = claim.queuePriority || 0;

			const dailyStats =
				await AdminPriorityChange.getDailyPriorityBoost(adminUsername);
			const MAX_DAILY_CHANGES = 10; // Max 10 priority changes per admin per day
			const MAX_DAILY_BOOST = 100; // Max total priority boost of 100 per day

			if (dailyStats.changeCount >= MAX_DAILY_CHANGES) {
				logger.warn(
					`Admin ${adminUsername} exceeded daily priority change limit (${dailyStats.changeCount}/${MAX_DAILY_CHANGES})`,
				);
				throw new ApiError(
					429,
					`Daily priority change limit exceeded (${MAX_DAILY_CHANGES} changes per day). Contact senior admin for override.`,
				);
			}

			if (
				dailyStats.totalBoost + Math.max(0, priorityChange) >
				MAX_DAILY_BOOST
			) {
				logger.warn(
					`Admin ${adminUsername} exceeded daily priority boost limit (${
						dailyStats.totalBoost + priorityChange
					}/${MAX_DAILY_BOOST})`,
				);
				throw new ApiError(
					429,
					`Daily priority boost limit exceeded (${MAX_DAILY_BOOST} total boost per day). Contact senior admin.`,
				);
			}

			if (priority > 20) {
				logger.warn(`High priority change (${priority}) by ${adminUsername}`);
			}

			// Update claim priority
			await ClaimRequest.findByIdAndUpdate(claimRequestId, {
				$set: {
					queuePriority: priority,
					queueStatus: "ready",
				},
			});

			await AdminPriorityChange.create({
				adminUsername,
				adminUserId: adminUsername, // Should be actual user ID in production
				claimRequestId,
				priorityBefore: currentPriority,
				priorityAfter: priority,
				priorityChange,
				reason: `Priority adjusted from ${currentPriority} to ${priority}`,
				ipAddress: clientIP,
				userAgent,
			});

			logger.info(
				`Claim ${claimRequestId} prioritized by ${adminUsername}: ${currentPriority} to ${priority}`,
			);

			await auditLogService.logAdminAction({
				adminUsername,
				action: "PRIORITIZE_CLAIM",
				targetId: claimRequestId,
				description: `Changed priority from ${currentPriority} to ${priority}`,
				severity: priority > 20 ? "HIGH" : "MEDIUM",
			});

			// Refresh queue positions
			await ClaimRequest.updateQueuePositions();

			return {
				success: true,
				message: "Claim prioritized successfully",
				dailyStats: {
					changesRemaining: MAX_DAILY_CHANGES - dailyStats.changeCount - 1,
					boostRemaining:
						MAX_DAILY_BOOST -
						dailyStats.totalBoost -
						Math.max(0, priorityChange),
				},
			};
		} catch (error) {
			logger.error(`Error prioritizing claim ${claimRequestId}:`, error);
			throw error;
		}
	},

	/**
	 * Mark claim for manual review
	 */
	async markForManualReview(claimRequestId, adminUsername, reason) {
		try {
			const claim = await ClaimRequest.findById(claimRequestId);

			if (!claim) {
				throw new ApiError(404, "Claim request not found");
			}

			if (claim.status !== "pending") {
				throw new ApiError(400, "Can only mark pending claims for review");
			}

			await ClaimRequest.findByIdAndUpdate(claimRequestId, {
				$set: {
					requiresManualReview: true,
					manualReviewReason: reason,
					manualReviewRequestedBy: adminUsername,
					manualReviewRequestedAt: new Date(),
					queueStatus: "manual_required",
				},
			});

			logger.info(
				`Claim ${claimRequestId} marked for manual review by ${adminUsername}: ${reason}`,
			);

			await auditLogService.logAdminAction({
				adminUsername,
				action: "MARK_MANUAL_REVIEW",
				targetId: claimRequestId,
				description: reason,
			});

			return { success: true, message: "Claim marked for manual review" };
		} catch (error) {
			logger.error(`Error marking claim for review ${claimRequestId}:`, error);
			throw error;
		}
	},

	/**
	 * Clear manual review requirement
	 */
	async clearManualReview(claimRequestId, adminUsername) {
		try {
			const claim = await ClaimRequest.findById(claimRequestId);

			if (!claim) {
				throw new ApiError(404, "Claim request not found");
			}

			if (claim.status !== "pending") {
				throw new ApiError(400, "Can only clear review on pending claims");
			}

			await ClaimRequest.findByIdAndUpdate(claimRequestId, {
				$set: {
					requiresManualReview: false,
					manualReviewReason: null,
					skipAutoApproval: false,
					queueStatus: "waiting",
					scheduledProcessingTime: new Date(), // Process immediately
				},
			});

			logger.info(
				`Manual review cleared for ${claimRequestId} by ${adminUsername}`,
			);

			await auditLogService.logAdminAction({
				adminUsername,
				action: "CLEAR_MANUAL_REVIEW",
				targetId: claimRequestId,
				description: "Cleared manual review requirement",
			});

			return { success: true, message: "Manual review cleared" };
		} catch (error) {
			logger.error(`Error clearing manual review ${claimRequestId}:`, error);
			throw error;
		}
	},

	/**
	 * Refresh all queue positions
	 */
	async refreshQueuePositions() {
		try {
			const updatedCount = await ClaimRequest.updateQueuePositions();
			logger.info(`Queue positions refreshed for ${updatedCount} claims`);
			return { success: true, updatedCount };
		} catch (error) {
			logger.error("Error refreshing queue positions:", error);
			throw error;
		}
	},

	/**
	 * Get user's claims
	 */
	async getUserClaims(username, options = {}) {
		try {
			const { page = 1, limit = 20, status } = options;

			const query = { username };
			if (status) query.status = status;

			const claims = await ClaimRequest.find(query)
				.select(
					"-ipAddress -userAgent -autoApprovalErrorHistory -preApprovalVerification",
				)
				.sort({ requestedAt: -1 })
				.skip((page - 1) * limit)
				.limit(limit)
				.lean();

			const total = await ClaimRequest.countDocuments(query);

			// Add real-time queue position for pending claims
			for (const claim of claims) {
				if (claim.status === "pending" && claim.queuedForApproval) {
					const claimsAhead = await this.getClaimsAheadInQueue(claim.queuedAt);
					claim.currentQueuePosition = claimsAhead + 1;
					claim.estimatedWaitMinutes = Math.ceil(
						1 + (claim.currentQueuePosition - 1) * 0.5,
					);
				}
			}

			return {
				claims,
				pagination: {
					page,
					limit,
					total,
					totalPages: Math.ceil(total / limit),
				},
			};
		} catch (error) {
			logger.error(`Error getting user claims for ${username}:`, error);
			throw error;
		}
	},
};

module.exports = { claimRequestService };
