const mongoose = require("mongoose");

const rollbackQueueSchema = new mongoose.Schema(
	{
		userId: {
			type: mongoose.Schema.Types.ObjectId,
			required: true,
			ref: "User",
			index: true,
		},
		username: {
			type: String,
			required: true,
			trim: true,
			index: true,
		},
		claimRequestId: {
			type: mongoose.Schema.Types.ObjectId,
			required: true,
			ref: "ClaimRequest",
			index: true,
		},
		withdrawalRequestId: {
			type: mongoose.Schema.Types.ObjectId,
			required: true,
			ref: "WithdrawalRequest",
		},
		amountNTE: {
			type: Number,
			required: true,
		},
		originalError: {
			type: String,
			required: true,
			trim: true,
		},
		rollbackError: {
			type: String,
			trim: true,
		},
		status: {
			type: String,
			required: true,
			enum: [
				"pending",
				"processing",
				"completed",
				"failed",
				"manual_intervention_required",
			],
			default: "pending",
			index: true,
		},
		attempts: {
			type: Number,
			required: true,
			default: 0,
		},
		maxAttempts: {
			type: Number,
			required: true,
			default: 5,
		},
		lastAttemptAt: {
			type: Date,
			default: null,
		},
		nextRetryAt: {
			type: Date,
			required: true,
			index: true,
			default: Date.now,
		},
		completedAt: {
			type: Date,
			default: null,
		},
		failedAt: {
			type: Date,
			default: null,
		},
		priority: {
			type: Number,
			required: true,
			default: 10,
			index: true,
		},
		attemptHistory: [
			{
				attemptNumber: Number,
				attemptedAt: Date,
				error: String,
				balanceBeforeAttempt: Number,
				balanceAfterAttempt: Number,
			},
		],
		metadata: {
			clientIP: String,
			userAgent: String,
			balanceSnapshot: Number,
			balanceChecksum: String,
			lockNonce: String,
			originalApprovalBy: String,
		},
		manualInterventionNotes: {
			type: String,
			trim: true,
		},
		resolvedBy: {
			type: String,
			trim: true,
		},
		resolvedAt: {
			type: Date,
			default: null,
		},
	},
	{
		timestamps: true,
		collection: "rollback_queue",
	},
);

rollbackQueueSchema.index({ status: 1, nextRetryAt: 1, priority: -1 });
rollbackQueueSchema.index({ userId: 1, status: 1 });
rollbackQueueSchema.index({ createdAt: -1 });

rollbackQueueSchema.statics.getNextPending = async function () {
	return await this.findOneAndUpdate(
		{
			status: "pending",
			nextRetryAt: { $lte: new Date() },
			attempts: { $lt: this.schema.path("maxAttempts").options.default || 5 },
		},
		{
			$set: {
				status: "processing",
				lastAttemptAt: new Date(),
			},
			$inc: { attempts: 1 },
		},
		{
			new: true,
			sort: { priority: -1, nextRetryAt: 1 },
		},
	);
};

rollbackQueueSchema.statics.markCompleted = async function (
	rollbackId,
	balanceAfter,
) {
	return await this.findByIdAndUpdate(
		rollbackId,
		{
			$set: {
				status: "completed",
				completedAt: new Date(),
			},
			$push: {
				attemptHistory: {
					attemptNumber: "$attempts",
					attemptedAt: new Date(),
					error: null,
					balanceAfterAttempt: balanceAfter,
				},
			},
		},
		{ new: true },
	);
};

rollbackQueueSchema.statics.markFailed = async function (
	rollbackId,
	error,
	nextRetryDelay = 60000,
) {
	const rollback = await this.findById(rollbackId);

	if (!rollback) {
		throw new Error(`Rollback ${rollbackId} not found`);
	}

	const update = {
		$set: {
			status:
				rollback.attempts >= rollback.maxAttempts
					? "manual_intervention_required"
					: "pending",
			rollbackError: error,
			nextRetryAt: new Date(Date.now() + nextRetryDelay),
		},
		$push: {
			attemptHistory: {
				attemptNumber: rollback.attempts,
				attemptedAt: new Date(),
				error: error,
			},
		},
	};

	if (rollback.attempts >= rollback.maxAttempts) {
		update.$set.failedAt = new Date();
	}

	return await this.findByIdAndUpdate(rollbackId, update, { new: true });
};

const RollbackQueue = mongoose.model("RollbackQueue", rollbackQueueSchema);

module.exports = { RollbackQueue };
