const {
	getTokenValue,
	getTotalTaxEarned,
	getTotalUsers,
	updateTokenValue,
	updateTotalTaxEarned,
	updateTotalUsers,
	getAllConfigs,
	clearAllCaches,
	initializeDefaultConfigs,
} = require("./globalConfig.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");
const { fetchAllTokenPrices } = require("./tokenPriceFetcher.service");

const globalConfigController = {
	// Get all global configurations
	async getAllConfigs(req, res, next) {
		try {
			const configs = await getAllConfigs();
			res.json(
				new ApiResponse(
					200,
					configs,
					"Global configurations retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get specific configuration by key
	async getConfigByKey(req, res, next) {
		try {
			const { key } = req.params;
			let value;

			switch (key) {
				case "nteTokenPrice":
					value = await getTokenValue();
					break;
				case "totalTaxEarned":
					value = await getTotalTaxEarned();
					break;
				case "totalUsers":
					value = await getTotalUsers();
					break;
				default:
					throw new ApiError(404, `Configuration key '${key}' not found`);
			}

			res.json(
				new ApiResponse(
					200,
					{ key, value },
					`Configuration '${key}' retrieved successfully`,
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Update configuration (admin only)
	async updateConfig(req, res, next) {
		try {
			const { key } = req.params;
			const { value, source = "manual" } = req.body;

			if (typeof value !== "number") {
				throw new ApiError(400, "Value must be a number");
			}

			if (value < 0) {
				throw new ApiError(400, "Value cannot be negative");
			}

			let config;

			switch (key) {
				case "nteTokenPrice":
					config = await updateTokenValue(value, source);
					break;
				case "totalTaxEarned":
					config = await updateTotalTaxEarned(value, source);
					break;
				case "totalUsers":
					config = await updateTotalUsers(value, source);
					break;
				default:
					throw new ApiError(404, `Configuration key '${key}' not found`);
			}

			res.json(
				new ApiResponse(
					200,
					config,
					`Configuration '${key}' updated successfully`,
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Clear all caches
	async clearCaches(req, res, next) {
		try {
			clearAllCaches();
			res.json(
				new ApiResponse(
					200,
					null,
					"All global configuration caches cleared successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Initialize default configurations
	async initializeDefaults(req, res, next) {
		try {
			await initializeDefaultConfigs();
			res.json(
				new ApiResponse(
					200,
					null,
					"Default configurations initialized successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// fetchAllTokenPrices
	async fetchAllTokenPrices(req, res, next) {
		try {
			const prices = await fetchAllTokenPrices();
			res.json(
				new ApiResponse(200, prices, "Token prices fetched successfully"),
			);
		} catch (err) {
			next(err);
		}
	},
};

module.exports = { globalConfigController };
