const cron = require('node-cron');
const { updateTokenValue } = require('./globalConfig.service');
const { fetchTokenPrice } = require('./tokenPriceFetcher.service');
const { getAdminOverrides } = require('../admin/adminOverride.service');
const { logger } = require('../../core/logger/logger');

let isUpdating = false;

// Update token price from blockchain/API with admin override check
async function updateTokenPriceJob() {
  if (isUpdating) {
    logger.debug('Token price update already in progress, skipping');
    return;
  }

  isUpdating = true;

  try {
    logger.info('Starting token price update job');

    let finalValue = null;
    let source = null;

    // Priority 1: Check admin overrides first
    try {
      const overrides = await getAdminOverrides();
      if (overrides && overrides.tokenValue) {
        finalValue = overrides.tokenValue.value;
        source = 'admin_override';
        logger.info(`Using admin override token value: $${finalValue.toFixed(6)} (set by ${overrides.tokenValue.setBy})`);
      }
    } catch (error) {
      logger.warn('Failed to check admin overrides:', error.message);
    }

    // Priority 2: If no admin override, fetch from blockchain/API
    if (!finalValue) {
      const priceData = await fetchTokenPrice();
      if (priceData && priceData.price > 0) {
        finalValue = priceData.price;
        source = priceData.source;
        logger.info(`Fetched token price from ${source}: $${finalValue.toFixed(6)}`);
      } else {
        logger.warn('Failed to fetch token price from blockchain/API');
      }
    }

    // Update cache with the determined value
    if (finalValue && finalValue > 0) {
      await updateTokenValue(finalValue, source);
      logger.info(`Token price updated successfully: $${finalValue.toFixed(6)} from ${source}`);
    } else {
      logger.warn('No valid token price available, keeping cached value');
    }
  } catch (error) {
    logger.error('Error in token price update job:', error);
  } finally {
    isUpdating = false;
  }
}

// Initialize cron jobs
function initializeGlobalConfigCronJobs() {
  // Update token price every 3 minutes
  const tokenPriceJob = cron.schedule('*/3 * * * *', updateTokenPriceJob, {
    scheduled: false,
    timezone: 'UTC',
  });

  logger.info('Global config cron jobs initialized: - Token price update: Every 3 minutes');

  // Start all jobs
  tokenPriceJob.start();

  // Run immediately on startup
  setTimeout(() => {
    logger.info('Running initial token price update');
    updateTokenPriceJob().catch((error) => {
      logger.error('Error in initial token price update:', error);
    });
  }, 5000); // Wait 5 seconds after startup

  return {
    tokenPriceJob,
    stop: () => {
      tokenPriceJob.stop();
      logger.info('Global config cron jobs stopped');
    },
  };
}

module.exports = {
  initializeGlobalConfigCronJobs,
  updateTokenPriceJob,
};
