const { cacheService } = require("../../core/cache/cache.service");
const { logger } = require("../../core/logger/logger");
const { getAdminOverrides } = require("../admin/adminOverride.service");

// Cache keys
const CACHE_KEYS = {
	TOKEN_PRICE: "global:nteTokenPrice",
	TOTAL_TAX: "global:totalTaxEarned",
	TOTAL_USERS: "global:totalUsers",
	DAYS_PER_YEAR: "global:daysPerYear",
	USERS_WITH_ACTIVE_PACKAGES: "global:usersWithActivePackages",
	PACKAGE_USER_COUNTS: "global:packageUserCounts",
};

// Cache TTL in seconds
const CACHE_TTL = {
	TOKEN_PRICE: 180, // 3 minutes
	GENERAL: 600, // 10 minutes
	USERS_WITH_ACTIVE_PACKAGES: 86400, // 24 hours
	PACKAGE_USER_COUNTS: 86400, // 24 hours
};

// In-memory cache for package user counts (24-hour TTL)
let packageUserCountsCache = {
	data: {},
	timestamp: null,
	ttl: 86400000, // 24 hours in milliseconds
};

// In-memory cache for users with active unlocked packages (24-hour TTL)
let usersWithActivePackagesCache = {
	data: [], // Array of usernames with active unlocked packages
	timestamp: null,
	ttl: 86400000, // 24 hours in milliseconds
};

// Default configuration values (fallback)
const DEFAULT_CONFIG = {
	nteTokenPrice: 0.001,
	totalTaxEarned: 50000,
	totalUsers: 1000,
	daysPerYear: 365,
};

// Get NTE token price from cache with admin override priority
async function getTokenValue() {
	try {
		let tokenValue = null;

		// Priority 1: Check admin overrides
		try {
			const overrides = await getAdminOverrides();
			if (overrides && overrides.tokenValue) {
				tokenValue = overrides.tokenValue.value;
			}
		} catch (error) {
			logger.warn(
				"Failed to check admin overrides for token value:",
				error.message,
			);
		}

		// Priority 2: Get from cache
		if (tokenValue === null) {
			const cached = await cacheService.get(CACHE_KEYS.TOKEN_PRICE);
			if (cached !== null) {
				tokenValue = cached;
			}
		}

		// Priority 3: Use default
		if (tokenValue === null) {
			tokenValue = DEFAULT_CONFIG.nteTokenPrice;
		}

		// Validate token price before use
		const parsedPrice = parseFloat(tokenValue);

		if (!Number.isFinite(parsedPrice) || parsedPrice <= 0) {
			logger.error(
				`Invalid token price detected: ${tokenValue}, using default`,
			);
			return DEFAULT_CONFIG.nteTokenPrice;
		}

		// Token price should be within reasonable range
		const MIN_TOKEN_PRICE = 0.00001; // $0.00001
		const MAX_TOKEN_PRICE = 1; // $1

		if (parsedPrice < MIN_TOKEN_PRICE || parsedPrice > MAX_TOKEN_PRICE) {
			logger.error(
				`Token price out of acceptable range: $${parsedPrice}. Min: $${MIN_TOKEN_PRICE}, Max: $${MAX_TOKEN_PRICE}`,
			);
			return DEFAULT_CONFIG.nteTokenPrice;
		}

		return parsedPrice;
	} catch (error) {
		logger.error("Error getting token value:", error);
		return DEFAULT_CONFIG.nteTokenPrice;
	}
}

// Get total tax earned from cache with admin override priority
async function getTotalTaxEarned() {
	try {
		// Priority 1: Check admin overrides first
		try {
			const overrides = await getAdminOverrides();
			if (overrides && overrides.totalTax) {
				return overrides.totalTax.value;
			}
		} catch (error) {
			logger.warn(
				"Failed to check admin overrides for total tax:",
				error.message,
			);
		}

		// Priority 2: Get from cache
		const cached = await cacheService.get(CACHE_KEYS.TOTAL_TAX);
		if (cached !== null) {
			return cached;
		}

		const defaultValue = DEFAULT_CONFIG.totalTaxEarned;
		await cacheService.set(
			CACHE_KEYS.TOTAL_TAX,
			defaultValue,
			CACHE_TTL.GENERAL,
		);
		return defaultValue;
	} catch (error) {
		logger.error("Error getting total tax earned:", error);
		return DEFAULT_CONFIG.totalTaxEarned;
	}
}

// Get total users from cache
async function getTotalUsers() {
	try {
		const cached = await cacheService.get(CACHE_KEYS.TOTAL_USERS);
		if (cached !== null) {
			return cached;
		}

		const defaultValue = DEFAULT_CONFIG.totalUsers;
		await cacheService.set(
			CACHE_KEYS.TOTAL_USERS,
			defaultValue,
			CACHE_TTL.GENERAL,
		);
		return defaultValue;
	} catch (error) {
		logger.error("Error getting total users:", error);
		return DEFAULT_CONFIG.totalUsers;
	}
}

// Update token value in cache
async function updateTokenValue(value, source = "manual") {
	try {
		await cacheService.set(
			CACHE_KEYS.TOKEN_PRICE,
			value,
			CACHE_TTL.TOKEN_PRICE,
		);
		logger.info(`Token value updated to ${value} from source: ${source}`);
		return { key: "nteTokenPrice", value, source, timestamp: Date.now() };
	} catch (error) {
		logger.error("Error updating token value:", error);
		throw error;
	}
}

// Update total tax earned in cache
async function updateTotalTaxEarned(value, source = "system") {
	try {
		await cacheService.set(CACHE_KEYS.TOTAL_TAX, value, CACHE_TTL.GENERAL);
		logger.info(`Total tax earned updated to ${value}`);
		return { key: "totalTaxEarned", value, source, timestamp: Date.now() };
	} catch (error) {
		logger.error("Error updating total tax earned:", error);
		throw error;
	}
}

// Update total users in cache
async function updateTotalUsers(value, source = "system") {
	try {
		await cacheService.set(CACHE_KEYS.TOTAL_USERS, value, CACHE_TTL.GENERAL);
		logger.info(`Total users updated to ${value}`);
		return { key: "totalUsers", value, source, timestamp: Date.now() };
	} catch (error) {
		logger.error("Error updating total users:", error);
		throw error;
	}
}

// Get all global configurations
async function getAllConfigs() {
	try {
		return {
			nteTokenPrice: await getTokenValue(),
			totalTaxEarned: await getTotalTaxEarned(),
			totalUsers: await getTotalUsers(),
			daysPerYear: DEFAULT_CONFIG.daysPerYear,
		};
	} catch (error) {
		logger.error("Error getting all configs:", error);
		return DEFAULT_CONFIG;
	}
}

// Clear all caches
async function clearAllCaches() {
	try {
		await cacheService.delete(CACHE_KEYS.TOKEN_PRICE);
		await cacheService.delete(CACHE_KEYS.TOTAL_TAX);
		await cacheService.delete(CACHE_KEYS.TOTAL_USERS);
		await cacheService.delete(CACHE_KEYS.DAYS_PER_YEAR);
		logger.info("All global config caches cleared");
	} catch (error) {
		logger.error("Error clearing caches:", error);
	}
}

// Initialize default configs in cache
async function initializeDefaultConfigs() {
	try {
		await cacheService.set(
			CACHE_KEYS.TOKEN_PRICE,
			DEFAULT_CONFIG.nteTokenPrice,
			CACHE_TTL.TOKEN_PRICE,
		);
		await cacheService.set(
			CACHE_KEYS.TOTAL_TAX,
			DEFAULT_CONFIG.totalTaxEarned,
			CACHE_TTL.GENERAL,
		);
		await cacheService.set(
			CACHE_KEYS.TOTAL_USERS,
			DEFAULT_CONFIG.totalUsers,
			CACHE_TTL.GENERAL,
		);
		await cacheService.set(
			CACHE_KEYS.DAYS_PER_YEAR,
			DEFAULT_CONFIG.daysPerYear,
			CACHE_TTL.GENERAL,
		);
		logger.info("Default global configurations initialized in cache");
	} catch (error) {
		logger.error("Error initializing default configs:", error);
	}
}

// Check if users with active packages cache is valid
async function isUsersWithActivePackagesCacheValid() {
	try {
		const cached = await cacheService.get(
			CACHE_KEYS.USERS_WITH_ACTIVE_PACKAGES,
		);
		return cached !== null && Array.isArray(cached) && cached.length > 0;
	} catch (error) {
		logger.warn(
			"Error checking users with active packages cache validity:",
			error,
		);
		return false;
	}
}

// Update users with active packages cache (24-hour TTL)
async function updateUsersWithActivePackagesCache() {
	try {
		logger.info("Updating cache of users with active unlocked packages...");

		const UserPackage = require("../purchase/userPackage.model");

		// Get all users with active unlocked packages
		const usersWithActivePackages = await UserPackage.aggregate([
			{
				$match: {
					status: "active",
				},
			},
			{
				$lookup: {
					from: "products",
					localField: "packageId",
					foreignField: "_id",
					as: "product",
				},
			},
			{
				$unwind: "$product",
			},
			{
				$match: {
					"product.packageType": "unlocked",
				},
			},
			{
				$group: {
					_id: "$username",
				},
			},
		]);

		// Extract usernames into an array
		const usernamesArray = usersWithActivePackages.map((user) => user._id);

		// Update in-memory cache
		usersWithActivePackagesCache = {
			data: usernamesArray,
			timestamp: Date.now(),
			ttl: 86400000, // 24 hours
		};

		// Also store in cache service
		await cacheService.set(
			CACHE_KEYS.USERS_WITH_ACTIVE_PACKAGES,
			usernamesArray,
			CACHE_TTL.USERS_WITH_ACTIVE_PACKAGES,
		);

		logger.info(
			`Users with active packages cache updated with ${usernamesArray.length} users`,
		);

		return usernamesArray;
	} catch (error) {
		logger.error("Error updating users with active packages cache:", error);
		throw error;
	}
}

// Get users with active packages from cache
async function getUsersWithActivePackages() {
	try {
		const cached = await cacheService.get(
			CACHE_KEYS.USERS_WITH_ACTIVE_PACKAGES,
		);
		if (cached !== null && Array.isArray(cached)) {
			return cached;
		}
		// If cache miss, update it
		return await updateUsersWithActivePackagesCache();
	} catch (error) {
		logger.warn("Error getting users with active packages:", error);
		return [];
	}
}

// Clear users with active packages cache
async function clearUsersWithActivePackagesCache() {
	try {
		await cacheService.delete(CACHE_KEYS.USERS_WITH_ACTIVE_PACKAGES);
		logger.info("Users with active packages cache cleared");
	} catch (error) {
		logger.error("Error clearing users with active packages cache:", error);
	}
}

// Check if a user has active unlocked packages
function hasActiveUnlockedPackageSync(username) {
	if (
		!usersWithActivePackagesCache.data ||
		usersWithActivePackagesCache.data.length === 0
	) {
		logger.warn(
			`Users with active packages cache is empty for user ${username}, may result in missed referral bonuses`,
		);
		return false;
	}
	return usersWithActivePackagesCache.data.includes(username);
}

// Check if package user counts cache is valid
function isPackageUserCountsCacheValid() {
	if (!packageUserCountsCache.data || !packageUserCountsCache.timestamp) {
		return false;
	}
	const now = Date.now();
	const cacheAge = now - packageUserCountsCache.timestamp;
	return cacheAge < packageUserCountsCache.ttl;
}

// Get package user counts from in-memory cache
function getPackageUserCountsSync() {
	return packageUserCountsCache.data || {};
}

// Get total users for a specific package
function getTotalUsersForPackageSync(packageId) {
	const counts = getPackageUserCountsSync();
	const count = counts[packageId] || 0;

	if (count === undefined || count === null) {
		logger.warn(
			`Package user count not found in cache for package: ${packageId}`,
		);
		return 0;
	}

	return count;
}

// Set package user counts
async function setPackageUserCounts(counts) {
	try {
		packageUserCountsCache = {
			data: counts,
			timestamp: Date.now(),
			ttl: 86400000, // 24 hours
		};

		// Also store in cache service for persistence
		await cacheService.set(
			CACHE_KEYS.PACKAGE_USER_COUNTS,
			counts,
			CACHE_TTL.PACKAGE_USER_COUNTS,
		);

		logger.info(
			`Package user counts cache updated: ${
				Object.keys(counts).length
			} packages`,
		);
		return counts;
	} catch (error) {
		logger.error("Error setting package user counts:", error);
		throw error;
	}
}

// Update package user counts from database
async function updatePackageUserCountsCache() {
	try {
		logger.info("Updating package user counts cache...");

		const UserPackage = require("../purchase/userPackage.model");

		// Aggregate to count active users per package
		const packageCounts = await UserPackage.aggregate([
			{
				$match: {
					status: "active",
				},
			},
			{
				$group: {
					_id: "$packageId",
					count: { $sum: 1 },
				},
			},
		]);

		const countsObject = {};
		packageCounts.forEach((item) => {
			countsObject[item._id.toString()] = item.count;
		});
		await setPackageUserCounts(countsObject);

		logger.info(
			`Package user counts cache updated with ${
				Object.keys(countsObject).length
			} packages`,
		);

		return countsObject;
	} catch (error) {
		logger.error("Error updating package user counts cache:", error);
		throw error;
	}
}

// Get package user counts (with fallback to database if cache is invalid)
async function getPackageUserCounts() {
	try {
		// Check if in-memory cache is valid
		if (isPackageUserCountsCacheValid()) {
			return getPackageUserCountsSync();
		}

		// Try to get from cache service
		const cached = await cacheService.get(CACHE_KEYS.PACKAGE_USER_COUNTS);
		if (cached !== null && typeof cached === "object") {
			packageUserCountsCache = {
				data: cached,
				timestamp: Date.now(),
				ttl: 86400000,
			};
			return cached;
		}

		// If cache miss, update from database
		logger.info("Package user counts cache expired, updating from database...");
		return await updatePackageUserCountsCache();
	} catch (error) {
		logger.warn("Error getting package user counts:", error);
		return getPackageUserCountsSync();
	}
}

// Clear package user counts cache
async function clearPackageUserCountsCache() {
	try {
		packageUserCountsCache = {
			data: {},
			timestamp: null,
			ttl: 86400000,
		};
		await cacheService.delete(CACHE_KEYS.PACKAGE_USER_COUNTS);
		logger.info("Package user counts cache cleared");
	} catch (error) {
		logger.error("Error clearing package user counts cache:", error);
	}
}

module.exports = {
	getTokenValue,
	getTotalTaxEarned,
	getTotalUsers,
	updateTokenValue,
	updateTotalTaxEarned,
	updateTotalUsers,
	getAllConfigs,
	clearAllCaches,
	initializeDefaultConfigs,
	isUsersWithActivePackagesCacheValid,
	updateUsersWithActivePackagesCache,
	getUsersWithActivePackages,
	clearUsersWithActivePackagesCache,
	hasActiveUnlockedPackageSync,
	getPackageUserCountsSync,
	isPackageUserCountsCacheValid,
	getTotalUsersForPackageSync,
	setPackageUserCounts,
	updatePackageUserCountsCache,
	getPackageUserCounts,
	clearPackageUserCountsCache,
	DEFAULT_CONFIG,
};
