const { ethers } = require("ethers");
const { logger } = require("../../core/logger/logger");
const { getTokenValue } = require("./globalConfig.service");

// Configuration
const BSC_RPC = process.env.BSC_RPC_URL || "https://bsc-dataseed.binance.org/";
const TOKEN_CONTRACT = process.env.TOKEN_CONTRACT_ADDRESS;
const TOKEN_BNB_PAIR = process.env.TOKEN_BNB_PAIR_ADDRESS;
const CHAINLINK_BNB_USD = "0x0567F2323251f0Aab15c8dFb1967E4e8A7D42aeE";
const MULTICALL_ADDRESS = "0x1Ee38d535d541c55C9dae27B12edf090C608E6Fb";

const PAIR_ABI = [
	"function getReserves() view returns (uint112 reserve0, uint112 reserve1, uint32 blockTimestampLast)",
	"function token0() view returns (address)",
	"function token1() view returns (address)",
];

const MULTICALL_ABI = [
	"function aggregate(tuple(address target, bytes callData)[] calls) public view returns (uint256 blockNumber, bytes[] returnData)",
];

const CHAINLINK_ABI = [
	"function latestAnswer() external view returns (int256)",
];

// Contract addresses for additional tokens
const BNB_ADDRESS = "0xbb4CdB9CBd36B01bD1cBaEBF2De08d9173bc095c"; // WBNB
const USDT_ADDRESS = "0x55d398326f99059fF775485246999027B3197955"; // USDT

// Fetch from DexScreener API
async function fetchDexScreenerPrice(contractAddress) {
	try {
		const response = await fetch(
			`https://api.dexscreener.com/latest/dex/tokens/${contractAddress}`,
			{ timeout: 5000 },
		);

		if (!response.ok) {
			throw new Error(`DexScreener API error: ${response.status}`);
		}

		const data = await response.json();
		if (!data.pairs?.length) return null;

		const bscPairs = data.pairs.filter(
			(pair) => pair.chainId === "bsc" && pair.liquidity?.usd > 0,
		);

		if (!bscPairs.length) return null;

		const selectedPair = bscPairs.reduce((max, pair) =>
			pair.liquidity.usd > max.liquidity.usd ? pair : max,
		);

		return {
			priceUsd: parseFloat(selectedPair.priceUsd || 0),
			timestamp: Date.now(),
			liquidity: parseFloat(selectedPair.liquidity?.usd || 0),
			volume24h: parseFloat(selectedPair.volume?.h24 || 0),
			priceChange24h: parseFloat(selectedPair.priceChange?.h24 || 0),
		};
	} catch (error) {
		logger.error("DexScreener fetch error:", error.message);
		return null;
	}
}

// Fetch from blockchain directly
async function fetchBlockchainPrice() {
	if (!TOKEN_CONTRACT || !TOKEN_BNB_PAIR) {
		logger.warn(
			"Missing TOKEN_CONTRACT or TOKEN_BNB_PAIR environment variables",
		);
		return null;
	}

	try {
		const provider = new ethers.JsonRpcProvider(BSC_RPC);
		const multicall = new ethers.Contract(
			MULTICALL_ADDRESS,
			MULTICALL_ABI,
			provider,
		);
		const pairInterface = new ethers.Interface(PAIR_ABI);

		// Batch blockchain calls
		const calls = [
			{
				target: TOKEN_BNB_PAIR,
				callData: pairInterface.encodeFunctionData("getReserves"),
			},
			{
				target: TOKEN_BNB_PAIR,
				callData: pairInterface.encodeFunctionData("token0"),
			},
			{
				target: TOKEN_BNB_PAIR,
				callData: pairInterface.encodeFunctionData("token1"),
			},
		];

		const [, returnData] = await multicall.aggregate(calls);

		const [reserve0, reserve1] = pairInterface.decodeFunctionResult(
			"getReserves",
			returnData[0],
		);
		const token0 = pairInterface.decodeFunctionResult(
			"token0",
			returnData[1],
		)[0];
		const token1 = pairInterface.decodeFunctionResult(
			"token1",
			returnData[2],
		)[0];

		let nteReserve, bnbReserve;
		if (token0.toLowerCase() === TOKEN_CONTRACT.toLowerCase()) {
			nteReserve = reserve0;
			bnbReserve = reserve1;
		} else if (token1.toLowerCase() === TOKEN_CONTRACT.toLowerCase()) {
			nteReserve = reserve1;
			bnbReserve = reserve0;
		} else {
			throw new Error("NTE token not found in pair");
		}

		// Get BNB/USD price from Chainlink
		const chainlink = new ethers.Contract(
			CHAINLINK_BNB_USD,
			CHAINLINK_ABI,
			provider,
		);
		const bnbPriceUSD = Number(await chainlink.latestAnswer()) / 1e8;

		// Calculate NTE price
		const nteReserveNum = Number(ethers.formatUnits(nteReserve, 18));
		const bnbReserveNum = Number(ethers.formatUnits(bnbReserve, 18));
		const priceInBNB = bnbReserveNum / nteReserveNum;
		const priceUSD = priceInBNB * bnbPriceUSD;

		return {
			priceUsd: priceUSD,
			priceInBNB,
			source: "blockchain",
		};
	} catch (error) {
		logger.error("Blockchain fetch error:", error.message);
		return null;
	}
}

// Main function to get token price with fallback strategy
async function fetchTokenPrice() {
	try {
		// Priority 1: Try blockchain first (most accurate)
		const blockchainPrice = await fetchBlockchainPrice();
		if (blockchainPrice && blockchainPrice.priceUsd > 0) {
			logger.info(
				`Token price from blockchain: $${blockchainPrice.priceUsd.toFixed(6)}`,
			);
			return {
				price: blockchainPrice.priceUsd,
				source: "blockchain",
				timestamp: Date.now(),
			};
		}

		// Priority 2: Try DexScreener API
		const dexScreenerPrice = await fetchDexScreenerPrice(TOKEN_CONTRACT);
		if (dexScreenerPrice && dexScreenerPrice.priceUsd > 0) {
			logger.info(
				`Token price from DexScreener: $${dexScreenerPrice.priceUsd.toFixed(
					6,
				)}`,
			);
			return {
				price: dexScreenerPrice.priceUsd,
				source: "dexscreener",
				timestamp: Date.now(),
				volume24h: dexScreenerPrice.volume24h,
				liquidity: dexScreenerPrice.liquidity,
				priceChange24h: dexScreenerPrice.priceChange24h,
			};
		}

		// Priority 3: Return null if both fail (caller should use cached/fallback)
		logger.warn("All token price sources failed, using fallback");
		return null;
	} catch (error) {
		logger.error("Error fetching token price:", error);
		return null;
	}
}

async function fetchTokenPriceDetails() {
	try {
		const price = await getTokenValue();

		// Priority 2: Try DexScreener API
		const dexScreenerPrice = await fetchDexScreenerPrice(TOKEN_CONTRACT);
		if (dexScreenerPrice && dexScreenerPrice.priceUsd > 0) {
			return {
				price: price,
				timestamp: Date.now(),
				volume24h: dexScreenerPrice.volume24h,
				liquidity: dexScreenerPrice.liquidity,
				priceChange24h: dexScreenerPrice.priceChange24h,
			};
		}

		// Priority 3: Return null if both fail (caller should use cached/fallback)
		logger.warn("All token price sources failed, using fallback");
		return {
			price: price,
			timestamp: Date.now(),
		};
	} catch (error) {
		logger.error("Error fetching token price:", error);
		return null;
	}
}

// Fetch prices for all three tokens: NTE, BNB, USDT
async function fetchAllTokenPrices() {
	try {
		const [ntePrice, bnbPrice, usdtPrice] = await Promise.all([
			fetchTokenPriceDetails(),
			fetchDexScreenerPrice(BNB_ADDRESS),
			fetchDexScreenerPrice(USDT_ADDRESS),
		]);

		return {
			NTE: ntePrice,
			BNB: bnbPrice,
			USDT: usdtPrice,
			timestamp: Date.now(),
		};
	} catch (error) {
		logger.error("Error fetching all token prices:", error);
		return null;
	}
}

module.exports = {
	fetchTokenPrice,
	fetchBlockchainPrice,
	fetchDexScreenerPrice,
	fetchAllTokenPrices,
};
