const { DbeProgram } = require("./dbeProgram.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");

class DbeProgramService {
	/**
	 * Get all DBE programs with optional filters
	 */
	async getAllPrograms({ activeOnly = false } = {}) {
		try {
			let pipeline = [];

			// Filter only active programs
			if (activeOnly) {
				pipeline.push({ $match: { isActive: true } });
			}

			// Lookup attendees to calculate current users and available spots
			pipeline.push(
				{
					$lookup: {
						from: "dbe_program_attendees",
						let: { programId: "$_id" },
						pipeline: [
							{
								$match: {
									$expr: {
										$and: [
											{ $eq: ["$programId", "$$programId"] },
											{ $eq: ["$status", "active"] },
										],
									},
								},
							},
						],
						as: "attendees",
					},
				},
				{
					$addFields: {
						currentUsers: { $size: "$attendees" },
						availableSpots: {
							$cond: [
								{ $ifNull: ["$maxUsers", false] },
								{
									$max: [{ $subtract: ["$maxUsers", { $size: "$attendees" }] }, 0],
								},
								0,
							],
						},
					},
				},
				{ $project: { attendees: 0 } }
			);

			// If no pipeline stages, use find() instead of aggregate()
			let programs;
			if (pipeline.length === 0) {
				programs = await DbeProgram.find();
			} else {
				programs = await DbeProgram.aggregate(pipeline);
			}

			logger.info(`Retrieved ${programs.length} DBE programs`);
			return programs;
		} catch (error) {
			logger.error("Error getting all DBE programs:", error);
			throw new ApiError(500, "Failed to retrieve DBE programs");
		}
	}

	/**
	 * Get a single DBE program by ID
	 */
	async getProgramById(id) {
		try {
			const program = await DbeProgram.findById(id);

			if (!program) {
				throw new ApiError(404, "DBE program not found");
			}

			// Get attendee count
			const pipeline = [
				{ $match: { _id: program._id } },
				{
					$lookup: {
						from: "dbe_program_attendees",
						let: { programId: "$_id" },
						pipeline: [
							{
								$match: {
									$expr: {
										$and: [
											{ $eq: ["$programId", "$$programId"] },
											{ $eq: ["$status", "active"] },
										],
									},
								},
							},
						],
						as: "attendees",
					},
				},
				{
					$addFields: {
						currentUsers: { $size: "$attendees" },
						availableSpots: {
							$cond: [
								{ $ifNull: ["$maxUsers", false] },
								{
									$max: [{ $subtract: ["$maxUsers", { $size: "$attendees" }] }, 0],
								},
								0,
							],
						},
					},
				},
				{ $project: { attendees: 0 } },
			];

			const result = await DbeProgram.aggregate(pipeline);
			const enrichedProgram = result[0] || program;

			logger.info(`Retrieved DBE program: ${id}`);
			return enrichedProgram;
		} catch (error) {
			if (error instanceof ApiError) {
				throw error;
			}
			logger.error("Error getting DBE program by ID:", error);
			throw new ApiError(500, "Failed to retrieve DBE program");
		}
	}

	/**
	 * Create a new DBE program
	 */
	async createProgram(programData) {
		try {
			// Validate required fields
			if (!programData.name || programData.name.trim() === "") {
				throw new ApiError(400, "Program name is required");
			}

			if (
				programData.minimumBuyAmount === undefined ||
				programData.minimumBuyAmount < 0
			) {
				throw new ApiError(400, "Minimum buy amount must be a non-negative number");
			}

			if (!programData.durationDays || programData.durationDays <= 0) {
				throw new ApiError(400, "Duration must be a positive number");
			}

			if (!programData.price) {
				throw new ApiError(400, "Price configuration is required");
			}

			const price = programData.price;
			if (price.fixedUSD === undefined && price.startUSD === undefined) {
				throw new ApiError(400, "Price must have either fixedUSD or startUSD");
			}

			if (
				!programData.userBonusCalculation ||
				programData.userBonusCalculation.bonusPercent === undefined
			) {
				throw new ApiError(400, "User bonus calculation with bonusPercent is required");
			}

			if (!programData.reserveWalletAddress) {
				throw new ApiError(400, "Reserve wallet address is required");
			}

			// Check for duplicate name
			const existingProgram = await DbeProgram.findOne({
				name: programData.name.trim(),
			});

			if (existingProgram) {
				throw new ApiError(
					409,
					`DBE program with name "${programData.name}" already exists`
				);
			}

			// Create program
			const program = new DbeProgram({
				...programData,
				name: programData.name.trim(),
				packageType: programData.packageType || 'dbe',
			});

			await program.save();

			logger.info(`Created DBE program: ${program._id}`);
			return program;
		} catch (error) {
			if (error instanceof ApiError) {
				throw error;
			}
			logger.error("Error creating DBE program:", error);
			throw new ApiError(500, "Failed to create DBE program");
		}
	}

	/**
	 * Update an existing DBE program
	 */
	async updateProgram(id, updateData) {
		try {
			const program = await DbeProgram.findById(id);

			if (!program) {
				throw new ApiError(404, "DBE program not found");
			}

			// Validate fields if provided
			if (updateData.name !== undefined) {
				if (updateData.name.trim() === "") {
					throw new ApiError(400, "Program name cannot be empty");
				}

				// Check for duplicate name (excluding current program)
				const existingProgram = await DbeProgram.findOne({
					name: updateData.name.trim(),
					_id: { $ne: id },
				});

				if (existingProgram) {
					throw new ApiError(
						409,
						`DBE program with name "${updateData.name}" already exists`
					);
				}

				updateData.name = updateData.name.trim();
			}

			if (
				updateData.minimumBuyAmount !== undefined &&
				updateData.minimumBuyAmount < 0
			) {
				throw new ApiError(400, "Minimum buy amount must be a non-negative number");
			}

			if (updateData.durationDays !== undefined && updateData.durationDays <= 0) {
				throw new ApiError(400, "Duration must be a positive number");
			}

			if (updateData.price) {
				const price = updateData.price;
				if (price.fixedUSD === undefined && price.startUSD === undefined) {
					throw new ApiError(400, "Price must have either fixedUSD or startUSD");
				}
			}

			if (
				updateData.userBonusCalculation !== undefined &&
				updateData.userBonusCalculation.bonusPercent === undefined
			) {
				throw new ApiError(400, "User bonus calculation with bonusPercent is required");
			}

			// Update program
			Object.assign(program, updateData);
			await program.save();

			logger.info(`Updated DBE program: ${id}`);
			return program;
		} catch (error) {
			if (error instanceof ApiError) {
				throw error;
			}
			logger.error("Error updating DBE program:", error);
			throw new ApiError(500, "Failed to update DBE program");
		}
	}

	/**
	 * Toggle program active status
	 */
	async toggleProgramStatus(id, isActive) {
		try {
			const program = await DbeProgram.findById(id);

			if (!program) {
				throw new ApiError(404, "DBE program not found");
			}

			program.isActive = isActive;
			await program.save();

			logger.info(`Toggled DBE program status: ${id} -> ${isActive}`);
			return program;
		} catch (error) {
			if (error instanceof ApiError) {
				throw error;
			}
			logger.error("Error toggling DBE program status:", error);
			throw new ApiError(500, "Failed to toggle DBE program status");
		}
	}

	/**
	 * Delete a DBE program
	 */
	async deleteProgram(id) {
		try {
			const program = await DbeProgram.findById(id);

			if (!program) {
				throw new ApiError(404, "DBE program not found");
			}

			await program.deleteOne();

			logger.info(`Deleted DBE program: ${id}`);
			return { message: "DBE program deleted successfully" };
		} catch (error) {
			if (error instanceof ApiError) {
				throw error;
			}
			logger.error("Error deleting DBE program:", error);
			throw new ApiError(500, "Failed to delete DBE program");
		}
	}
}

module.exports = new DbeProgramService();
