const dbeProgramAttendeeService = require('./dbeProgramAttendee.service');
const { ApiError } = require('../../utils/ApiError');
const { logger } = require('../../core/logger/logger');

/**
 * Get all DBE program attendees for a user
 */
const getUserAttendances = async (req, res) => {
    try {
        const { username } = req.params;
        
        // Parse query parameters properly
        const filters = {
            includeCompliance: req.query.includeCompliance === 'true',
            searchTerm: req.query.searchTerm,
            status: req.query.status,
            startDate: req.query.startDate,
            endDate: req.query.endDate,
            sortBy: req.query.sortBy || 'createdAt',
            sortOrder: req.query.sortOrder || 'desc',
            page: parseInt(req.query.page) || 1,
            limit: parseInt(req.query.limit) || 10,
        };

        const result = await dbeProgramAttendeeService.getUserAttendances(username, filters);

        res.status(200).json({
            success: true,
            data: result,
        });
    } catch (error) {
        if (error instanceof ApiError) {
            res.status(error.statusCode).json({
                success: false,
                message: error.message,
            });
        } else {
            logger.error('Error in getUserAttendances:', error);
            res.status(500).json({
                success: false,
                message: 'Internal server error',
                error: error.message,
            });
        }
    }
};

/**
 * Get all attendees for a specific DBE program
 */
const getProgramAttendees = async (req, res) => {
    try {
        const { programId } = req.params;

        const attendances = await dbeProgramAttendeeService.getProgramAttendees(programId);

        res.status(200).json({
            success: true,
            data: attendances,
        });
    } catch (error) {
        if (error instanceof ApiError) {
            res.status(error.statusCode).json({
                success: false,
                message: error.message,
            });
        } else {
            logger.error('Error in getProgramAttendees:', error);
            res.status(500).json({
                success: false,
                message: 'Internal server error',
            });
        }
    }
};

/**
 * Update attendance status
 */
const updateAttendanceStatus = async (req, res) => {
    try {
        const { attendanceId } = req.params;
        const { status } = req.body;

        if (!status) {
            throw new ApiError(400, 'Status is required');
        }

        const attendance = await dbeProgramAttendeeService.updateAttendanceStatus(attendanceId, status);

        res.status(200).json({
            success: true,
            data: attendance,
            message: 'Attendance status updated successfully',
        });
    } catch (error) {
        if (error instanceof ApiError) {
            res.status(error.statusCode).json({
                success: false,
                message: error.message,
            });
        } else {
            logger.error('Error in updateAttendanceStatus:', error);
            res.status(500).json({
                success: false,
                message: 'Internal server error',
            });
        }
    }
};

/**
 * Get attendance by ID
 */
const getAttendanceById = async (req, res) => {
    try {
        const { attendanceId } = req.params;

        const attendance = await dbeProgramAttendeeService.getAttendanceById(attendanceId);

        res.status(200).json({
            success: true,
            data: attendance,
        });
    } catch (error) {
        if (error instanceof ApiError) {
            res.status(error.statusCode).json({
                success: false,
                message: error.message,
            });
        } else {
            logger.error('Error in getAttendanceById:', error);
            res.status(500).json({
                success: false,
                message: 'Internal server error',
            });
        }
    }
};

/**
 * Get all active attendances
 */
const getActiveAttendances = async (req, res) => {
    try {
        const attendances = await dbeProgramAttendeeService.getActiveAttendances();

        res.status(200).json({
            success: true,
            data: attendances,
        });
    } catch (error) {
        if (error instanceof ApiError) {
            res.status(error.statusCode).json({
                success: false,
                message: error.message,
            });
        } else {
            logger.error('Error in getActiveAttendances:', error);
            res.status(500).json({
                success: false,
                message: 'Internal server error',
            });
        }
    }
};

/**
 * Get all DBE attendances with advanced filtering (admin)
 */
const getAllAttendances = async (req, res) => {
    try {
        const filters = req.query;

        const result = await dbeProgramAttendeeService.getAllAttendances(filters);

        res.status(200).json({
            success: true,
            data: result,
        });
    } catch (error) {
        if (error instanceof ApiError) {
            res.status(error.statusCode).json({
                success: false,
                message: error.message,
            });
        } else {
            logger.error('Error in getAllAttendances:', error);
            res.status(500).json({
                success: false,
                message: 'Internal server error',
            });
        }
    }
};

module.exports = {
    getUserAttendances,
    getProgramAttendees,
    updateAttendanceStatus,
    getAttendanceById,
    getActiveAttendances,
    getAllAttendances,
};