const { DbeProgramAttendee } = require("./dbeProgramAttendee.model");
const {
	PancakeswapTransaction,
} = require("../pancakeswapTransactions/pancakeswapTransaction.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");

class DbeProgramAttendeeService {
	/**
	 * Get all DBE program attendees for a user with filters
	 */
	async getUserAttendances(username, filters = {}) {
		try {
			if (!username) {
				throw new ApiError(400, "Username is required");
			}

			// Build aggregation pipeline
			const pipeline = [
				{
					$match: { username: username },
				},
				{
					$lookup: {
						from: "dbe_programs",
						localField: "programId",
						foreignField: "_id",
						as: "program",
					},
				},
				{
					$unwind: {
						path: "$program",
						preserveNullAndEmptyArrays: true,
					},
				},
			];

			// Build projection dynamically
			const projection = {
				_id: 1,
				username: 1,
				programId: 1,
				programName: { $ifNull: ["$program.name", "Unknown Program"] },
				durationDays: { $ifNull: ["$program.durationDays", 30] },
				userBonusPercent: {
					$ifNull: ["$program.userBonusCalculation.bonusPercent", 0],
				},
				referralBonus: { $ifNull: ["$program.referralBonus", 0] },
				minimumBuyAmount: { $ifNull: ["$program.minimumBuyAmount", 100] },
				joiningFee: { $ifNull: ["$program.fee", 0] },
				feeUSD: 1,
				feeNTE: 1,
				tokenPriceAtJoining: 1,
				startDate: 1,
				endDate: 1,
				status: 1,
				disqualificationReason: 1,
				disqualifiedAt: 1,
				blockchainTransaction: 1,
				metadata: 1,
				createdAt: 1,
				updatedAt: 1,
			};

			// Include compliance data if requested
			if (filters.includeCompliance === true) {
				projection.dailyLogs = 1;
			}

			pipeline.push({ $project: projection });

			// Add filtering
			const filterConditions = [];

			// Search filter (program name)
			if (filters.searchTerm) {
				filterConditions.push({
					$match: {
						programName: { $regex: filters.searchTerm, $options: "i" },
					},
				});
			}

			// Status filter
			if (filters.status) {
				filterConditions.push({
					$match: {
						status: filters.status,
					},
				});
			}

			// Date range filter
			if (filters.startDate || filters.endDate) {
				const dateMatch = {};
				if (filters.startDate) {
					dateMatch.$gte = new Date(filters.startDate);
				}
				if (filters.endDate) {
					const endDate = new Date(filters.endDate);
					endDate.setHours(23, 59, 59, 999);
					dateMatch.$lte = endDate;
				}
				filterConditions.push({
					$match: {
						createdAt: dateMatch,
					},
				});
			}

			// Add filter conditions to pipeline
			pipeline.push(...filterConditions);

			// Add sorting
			const sortField = filters.sortBy || "createdAt";
			const sortOrder = filters.sortOrder === "desc" ? -1 : 1;
			pipeline.push({
				$sort: { [sortField]: sortOrder },
			});

			// Get total count
			const countPipeline = [...pipeline, { $count: "total" }];
			const countResult = await DbeProgramAttendee.aggregate(countPipeline);
			const total = countResult.length > 0 ? countResult[0].total : 0;

			// Add pagination
			const page = parseInt(filters.page) || 1;
			const limit = parseInt(filters.limit) || 10;
			const skip = (page - 1) * limit;

			pipeline.push({ $skip: skip });
			pipeline.push({ $limit: limit });

			// Execute query
			const attendances = await DbeProgramAttendee.aggregate(pipeline);

			logger.info(
				`Retrieved ${attendances.length} DBE attendances for user ${username}`,
			);
			return {
				attendances,
				total,
				page,
				limit,
				totalPages: Math.ceil(total / limit),
			};
		} catch (error) {
			logger.error("Error getting user DBE attendances:", {
				error: error.message,
				stack: error.stack,
				username,
				filters,
			});
			throw new ApiError(500, `Failed to retrieve DBE attendances: ${error.message}`);
		}
	}

	/**
	 * Get all attendees for a specific DBE program
	 */
	async getProgramAttendees(programId) {
		try {
			const attendances = await DbeProgramAttendee.aggregate([
				{
					$match: { programId: programId },
				},
				{
					$lookup: {
						from: "dbe_programs",
						localField: "programId",
						foreignField: "_id",
						as: "program",
					},
				},
				{
					$unwind: {
						path: "$program",
						preserveNullAndEmptyArrays: true,
					},
				},
				{
					$project: {
						_id: 1,
						username: 1,
						programId: 1,
						programName: { $ifNull: ["$program.name", "Unknown Program"] },
						durationDays: { $ifNull: ["$program.durationDays", 30] },
						userBonusPercent: {
							$ifNull: ["$program.userBonusCalculation.bonusPercent", 0],
						},
						referralBonus: { $ifNull: ["$program.referralBonus", 0] },
						minimumBuyAmount: { $ifNull: ["$program.minimumBuyAmount", 100] },
						joiningFee: { $ifNull: ["$program.fee", 0] },
						feeUSD: 1,
						feeNTE: 1,
						tokenPriceAtJoining: 1,
						startDate: 1,
						endDate: 1,
						status: 1,
						createdAt: 1,
						updatedAt: 1,
					},
				},
				{
					$sort: { createdAt: -1 },
				},
			]);

			logger.info(
				`Retrieved ${attendances.length} attendees for program ${programId}`,
			);
			return attendances;
		} catch (error) {
			logger.error("Error getting program attendees:", error);
			throw new ApiError(500, "Failed to retrieve program attendees");
		}
	}

	/**
	 * Update attendance status
	 */
	async updateAttendanceStatus(attendanceId, status) {
		try {
			const attendance = await DbeProgramAttendee.findById(attendanceId);

			if (!attendance) {
				throw new ApiError(404, "DBE attendance not found");
			}

			attendance.status = status;
			attendance.updatedAt = new Date();
			await attendance.save();

			logger.info(
				`Updated DBE attendance status: ${attendanceId} -> ${status}`,
			);
			return attendance;
		} catch (error) {
			if (error instanceof ApiError) {
				throw error;
			}
			logger.error("Error updating attendance status:", error);
			throw new ApiError(500, "Failed to update attendance status");
		}
	}

	/**
	 * Get attendance by ID
	 */
	async getAttendanceById(attendanceId) {
		try {
			const attendance = await DbeProgramAttendee.aggregate([
				{
					$match: { _id: attendanceId },
				},
				{
					$lookup: {
						from: "dbe_programs",
						localField: "programId",
						foreignField: "_id",
						as: "program",
					},
				},
				{
					$unwind: {
						path: "$program",
						preserveNullAndEmptyArrays: true,
					},
				},
				{
					$project: {
						_id: 1,
						username: 1,
						programId: 1,
						programName: { $ifNull: ["$program.name", "Unknown Program"] },
						durationDays: { $ifNull: ["$program.durationDays", 30] },
						userBonusPercent: {
							$ifNull: ["$program.userBonusCalculation.bonusPercent", 0],
						},
						referralBonus: { $ifNull: ["$program.referralBonus", 0] },
						minimumBuyAmount: { $ifNull: ["$program.minimumBuyAmount", 100] },
						joiningFee: { $ifNull: ["$program.fee", 0] },
						startDate: 1,
						endDate: 1,
						status: 1,
						createdAt: 1,
						updatedAt: 1,
					},
				},
			]);

			if (!attendance[0]) {
				throw new ApiError(404, "DBE attendance not found");
			}

			logger.info(`Retrieved DBE attendance: ${attendanceId}`);
			return attendance[0];
		} catch (error) {
			if (error instanceof ApiError) {
				throw error;
			}
			logger.error("Error getting attendance by ID:", error);
			throw new ApiError(500, "Failed to retrieve DBE attendance");
		}
	}

	/**
	 * Get all active attendances (for cron jobs, etc.)
	 */
	async getActiveAttendances() {
		try {
			const attendances = await DbeProgramAttendee.aggregate([
				{
					$match: { status: "active" },
				},
				{
					$lookup: {
						from: "dbe_programs",
						localField: "programId",
						foreignField: "_id",
						as: "program",
					},
				},
				{
					$unwind: {
						path: "$program",
						preserveNullAndEmptyArrays: true,
					},
				},
				{
					$project: {
						_id: 1,
						username: 1,
						programId: 1,
						address: 1,
						programName: { $ifNull: ["$program.name", "Unknown Program"] },
						durationDays: { $ifNull: ["$program.durationDays", 30] },
						userBonusPercent: {
							$ifNull: ["$program.userBonusCalculation.bonusPercent", 0],
						},
						referralBonus: { $ifNull: ["$program.referralBonus", 0] },
						minimumBuyAmount: { $ifNull: ["$program.minimumBuyAmount", 100] },
						joiningFee: { $ifNull: ["$program.fee", 0] },
						feeUSD: 1,
						feeNTE: 1,
						tokenPriceAtJoining: 1,
						startDate: 1,
						endDate: 1,
						blockchainTransactionHash: 1,
						status: 1,
						dailyLogs: 1,
						createdAt: 1,
						updatedAt: 1,
					},
				},
				{
					$sort: { createdAt: -1 },
				},
			]);

			logger.info(`Retrieved ${attendances.length} active DBE attendances`);
			return attendances;
		} catch (error) {
			logger.error("Error getting active attendances:", error);
			throw new ApiError(500, "Failed to retrieve active attendances");
		}
	}

	/**
	 * Get all DBE attendances with advanced filtering for admin
	 */
	async getAllAttendances(filters = {}) {
		try {
			const mongoose = require("mongoose");
			const matchConditions = {};

			if (filters.programSearch) {
				// This will be handled after the lookup
			}

			if (filters.usernameSearch) {
				matchConditions.username = {
					$regex: filters.usernameSearch,
					$options: "i",
				};
			}

			if (filters.programId) {
				matchConditions.programId = new mongoose.Types.ObjectId(filters.programId);
			}

			if (filters.status) {
				matchConditions.status = filters.status;
			}

			if (filters.startDate) {
				matchConditions.startDate = { $gte: new Date(filters.startDate) };
			}

			if (filters.endDate) {
				const endDate = new Date(filters.endDate);
				endDate.setHours(23, 59, 59, 999);
				matchConditions.startDate = {
					...matchConditions.startDate,
					$lte: endDate,
				};
			}

			// Build aggregation pipeline
			const pipeline = [];

			// Add match conditions if any
			if (Object.keys(matchConditions).length > 0) {
				pipeline.push({ $match: matchConditions });
			}

			// Join with programs
			pipeline.push({
				$lookup: {
					from: "dbe_programs",
					localField: "programId",
					foreignField: "_id",
					as: "program",
				},
			});

			pipeline.push({
				$unwind: {
					path: "$program",
					preserveNullAndEmptyArrays: true,
				},
			});

			// Filter by program name if search provided
			if (filters.programSearch) {
				pipeline.push({
					$match: {
						"program.name": { $regex: filters.programSearch, $options: "i" },
					},
				});
			}

			// Add compliance data if requested
			if (filters.includeCompliance) {
				// Calculate compliance status from existing dailyLogs
				pipeline.push({
					$addFields: {
						complianceStatus: {
							$cond: {
								if: { $gt: [{ $size: { $ifNull: ["$dailyLogs", []] } }, 0] },
								then: {
									$cond: {
										if: {
											$gt: [
												{
													$size: {
														$filter: {
															input: { $ifNull: ["$dailyLogs", []] },
															as: "log",
															cond: { $eq: ["$$log.isCompliant", true] },
														},
													},
												},
												0,
											],
										},
										then: "Compliant",
										else: "Disqualified",
									},
								},
								else: "No Data",
							},
						},
						daysCompliant: {
							$size: {
								$filter: {
									input: { $ifNull: ["$dailyLogs", []] },
									as: "log",
									cond: { $eq: ["$$log.isCompliant", true] },
								},
							},
						},
						lastComplianceCheck: {
							$cond: {
								if: { $gt: [{ $size: { $ifNull: ["$dailyLogs", []] } }, 0] },
								then: { $max: "$dailyLogs.checkedAt" },
								else: null,
							},
						},
					},
				});
			}

			// Project fields
			pipeline.push({
				$project: {
					_id: 1,
					username: 1,
					programId: 1,
					programName: { $ifNull: ["$program.name", "Unknown Program"] },
					durationDays: { $ifNull: ["$program.durationDays", 30] },
					userBonusPercent: {
						$ifNull: ["$program.userBonusCalculation.bonusPercent", 0],
					},
					referralBonus: { $ifNull: ["$program.referralBonus", 0] },
					minimumBuyAmount: { $ifNull: ["$program.minimumBuyAmount", 100] },
					joiningFee: { $ifNull: ["$program.fee", 0] },
					feeUSD: 1,
					feeNTE: 1,
					tokenPriceAtJoining: 1,
					startDate: 1,
					endDate: 1,
					status: 1,
					disqualificationReason: 1,
					disqualifiedAt: 1,
					blockchainTransaction: 1,
					metadata: 1,
					createdAt: 1,
					updatedAt: 1,
					...(filters.includeCompliance && {
						dailyLogs: 1,
						complianceStatus: 1,
						daysCompliant: 1,
						lastComplianceCheck: 1,
					}),
				},
			});

			// Sorting
			const sortField = filters.sortBy || "createdAt";
			const sortOrder = filters.sortOrder === "desc" ? -1 : 1;
			pipeline.push({
				$sort: { [sortField]: sortOrder },
			});

			// Get total count
			const countPipeline = [...pipeline, { $count: "total" }];
			const countResult = await DbeProgramAttendee.aggregate(countPipeline);
			const total = countResult.length > 0 ? countResult[0].total : 0;

			// Pagination
			const page = parseInt(filters.page) || 1;
			const limit = parseInt(filters.limit) || 10;
			const skip = (page - 1) * limit;

			pipeline.push({ $skip: skip });
			pipeline.push({ $limit: limit });

			// Execute query
			const attendances = await DbeProgramAttendee.aggregate(pipeline);

			logger.info(
				`Retrieved ${attendances.length} DBE attendances (total: ${total})`,
			);
			return {
				attendances,
				total,
				page,
				limit,
				totalPages: Math.ceil(total / limit),
			};
		} catch (error) {
			logger.error("Error getting all DBE attendances:", error);
			throw new ApiError(500, "Failed to retrieve DBE attendances");
		}
	}

	/**
	 * Perform daily DBE verification for all active attendances
	 */
	async performDailyDBEVerification() {
		try {
			logger.info("[DBE] Starting verification");

			const activeAttendances = await this.getActiveAttendances();
			logger.info(`[DBE] Found ${activeAttendances.length} active attendances`);

			if (activeAttendances.length === 0) {
				logger.info("[DBE] No active attendances");
				return { success: true, processed: 0, disqualified: 0, date: null };
			}

			const now = new Date();
			const yesterday = new Date(now);
			yesterday.setDate(yesterday.getDate() - 1);
			yesterday.setHours(0, 0, 0, 0);
			const today = new Date(yesterday);
			today.setDate(today.getDate() + 1);

			const yesterdayStart = yesterday.getTime();
			const yesterdayEnd = today.getTime();

			const activeUsernames = activeAttendances
				.map((attendance) => attendance.username)
				.filter((username) => username && username.trim() !== "");

			logger.info(`[DBE] Processing ${activeUsernames.length} usernames`);

			if (activeUsernames.length === 0) {
				logger.info("[DBE] No valid usernames");
				return {
					success: true,
					processed: 0,
					disqualified: 0,
					date: yesterday.toISOString().split("T")[0],
				};
			}

			const BATCH_SIZE = 5;
			let allYesterdayTransactions = [];

			for (let i = 0; i < activeUsernames.length; i += BATCH_SIZE) {
				const batchUsernames = activeUsernames.slice(i, i + BATCH_SIZE);

				const batchTransactions = await PancakeswapTransaction.find({
					processed: true,
					timestamp: {
						$gte: new Date(yesterdayStart),
						$lt: new Date(yesterdayEnd),
					},
					$or: [
						{ usernameFrom: { $in: batchUsernames } },
						{ usernameTo: { $in: batchUsernames } },
					],
				}).sort({ timestamp: -1 });

				allYesterdayTransactions.push(...batchTransactions);
			}

			logger.info(
				`[DBE] Found ${allYesterdayTransactions.length} transactions`,
			);

			if (allYesterdayTransactions.length === 0) {
				logger.info("[DBE] No transactions - all users disqualified");
			}

			const transactionsByUsername = new Map();

			for (const tx of allYesterdayTransactions) {
				if (tx.usernameFrom && activeUsernames.includes(tx.usernameFrom)) {
					if (!transactionsByUsername.has(tx.usernameFrom)) {
						transactionsByUsername.set(tx.usernameFrom, []);
					}
					transactionsByUsername.get(tx.usernameFrom).push(tx);
				}

				if (
					tx.usernameTo &&
					activeUsernames.includes(tx.usernameTo) &&
					tx.usernameTo !== tx.usernameFrom
				) {
					if (!transactionsByUsername.has(tx.usernameTo)) {
						transactionsByUsername.set(tx.usernameTo, []);
					}
					transactionsByUsername.get(tx.usernameTo).push(tx);
				}
			}

			logger.info(
				`[DBE] Grouped transactions for ${transactionsByUsername.size} users`,
			);

			let processedCount = 0;
			let disqualifiedCount = 0;
			const startTime = Date.now();
			const bulkOps = [];

			for (const attendance of activeAttendances) {
				try {
					const {
						_id,
						username,
						programId,
						minimumBuyAmount,
						status,
						dailyLogs,
						startDate,
						endDate,
						blockchainTransactionHash,
					} = attendance;

					const now = new Date();
					if (now > new Date(endDate)) {
						bulkOps.push({
							updateOne: {
								filter: { _id },
								update: {
									$set: {
										status: "completed",
										completedAt: new Date(),
										updatedAt: new Date(),
									},
								},
							},
						});
						logger.info(
							`[DBE] ${username} program ended, status set to completed`,
						);
						continue;
					}
					if (now < new Date(startDate)) {
						logger.info(
							`[DBE] Skipping ${username} - program not started yet (start: ${startDate})`,
						);
						continue;
					}

					if (status !== "active") continue;
					if (!username || username.trim() === "") continue;

					const yesterdayDate = yesterday.toISOString().split("T")[0];
					const existingLog = dailyLogs?.find(
						(log) => log.date === yesterdayDate,
					);

					if (existingLog) {
						logger.info(
							`[DBE] Skipping ${username} - already processed for ${yesterdayDate}`,
						);
						continue;
					}

					logger.info(
						`[DBE] Step 1: Starting verification for ${username} (program: ${programId})`,
					);

					const userTransactions = transactionsByUsername.get(username) || [];
					logger.info(
						`[DBE] Step 2: Found ${userTransactions.length} transactions for ${username}`,
					);

					// Filter transactions to only those after the program start date
					const filteredTransactions = userTransactions.filter(
						(tx) => new Date(tx.timestamp) >= new Date(startDate),
					);
					logger.info(
						`[DBE] Step 2.5: After filtering by startDate (${startDate}), ${filteredTransactions.length} transactions remain`,
					);

					let totalBuyNTE = 0;
					let totalBuyUSD = 0;
					let totalSellNTE = 0;
					let totalSellUSD = 0;
					let totalTransferOutNTE = 0;
					let totalTransferInNTE = 0;
					let hasSell = false;
					let hasTransferOut = false;
					let hasBuy = false;

					logger.info(`[DBE] Step 3: Analyzing transactions for ${username}`);

					for (const tx of filteredTransactions) {
						// Skip the transaction used to buy the package
						if (tx.transactionHash === blockchainTransactionHash) continue;

						const amount = parseFloat(tx.valueFormatted) || 0;
						const amountUSD = parseFloat(tx.valueUSD) || 0;
						const isOutgoing = tx.usernameFrom === username;
						const isIncoming = tx.usernameTo === username;

						if (tx.type === "buy" && isIncoming) {
							totalBuyNTE += amount;
							totalBuyUSD += amountUSD;
							hasBuy = true;
							logger.info(`[DBE]   - ${username}: Buy +${amount} NTE`);
						} else if (tx.type === "sell" && isOutgoing) {
							totalSellNTE += amount;
							totalSellUSD += amountUSD;
							hasSell = true;
							logger.info(
								`[DBE]   - ${username}: Sell -${amount} NTE (DISQUALIFY)`,
							);
						} else if (isOutgoing) {
							totalTransferOutNTE += amount;
							hasTransferOut = true;
							logger.info(
								`[DBE]   - ${username}: Transfer out -${amount} NTE (DISQUALIFY)`,
							);
						} else if (isIncoming) {
							totalTransferInNTE += amount;
							logger.info(
								`[DBE]   - ${username}: Transfer in +${amount} NTE (ignored)`,
							);
						}
					}

					logger.info(`[DBE] Step 4: Compliance check for ${username}`);
					logger.info(
						`[DBE]   - Total buy: ${totalBuyNTE} NTE / ${totalBuyUSD} USD, Required: ${minimumBuyAmount} USD`,
					);
					logger.info(
						`[DBE]   - Has sell: ${hasSell}, Has transfer out: ${hasTransferOut}, Has buy: ${hasBuy}`,
					);

					let isCompliant = true;
					let disqualificationReason = null;

					if (hasSell || hasTransferOut) {
						isCompliant = false;
						disqualificationReason = hasSell
							? "sold_tokens"
							: "transferred_out";
						logger.info(
							`[DBE]   - ${username}: FAILED - ${disqualificationReason}`,
						);
					} else if (hasBuy && totalBuyUSD < minimumBuyAmount) {
						isCompliant = false;
						disqualificationReason = "insufficient_buy";
						logger.info(
							`[DBE]   - ${username}: FAILED - insufficient buy amount`,
						);
					} else if (!hasBuy) {
						isCompliant = false;
						disqualificationReason = "no_buy";
						logger.info(`[DBE]   - ${username}: FAILED - no buy activity`);
					} else {
						logger.info(`[DBE]   - ${username}: PASSED - compliant`);
					}

					const dailyLog = {
						date: yesterday.toISOString().split("T")[0],
						totalBuyNTE,
						totalBuyUSD,
						totalSellNTE,
						totalSellUSD,
						totalTransferOutNTE,
						totalTransferInNTE,
						minimumRequired: minimumBuyAmount,
						isCompliant,
						disqualificationReason,
						transactionSummary: {
							hasBuy,
							hasSell,
							hasTransferOut,
							hasTransferIn: totalTransferInNTE > 0,
							totalTransactions: filteredTransactions.length,
						},
						checkedAt: new Date().toISOString(),
					};

					const updateData = {
						$set: {
							updatedAt: new Date(),
							...(isCompliant
								? {}
								: {
										status: "disqualified",
										disqualifiedAt: new Date(),
										disqualificationReason: disqualificationReason,
								  }),
						},
						$push: { dailyLogs: dailyLog },
					};

					logger.info(`[DBE] Step 5: Updating database for ${username}`);

					if (!isCompliant) {
						disqualifiedCount++;
						logger.info(
							`[DBE]   - ${username}: Status set to DISQUALIFIED (${disqualificationReason})`,
						);
					} else {
						logger.info(
							`[DBE]   - ${username}: Status remains ACTIVE (compliant)`,
						);
					}

					logger.info(
						`[DBE]   - ${username}: Daily log added for ${dailyLog.date}`,
					);
					logger.info(`[DBE] Step 6: Completed verification for ${username}\n`);

					bulkOps.push({
						updateOne: {
							filter: { _id },
							update: updateData,
						},
					});

					processedCount++;

					if (bulkOps.length >= 100) {
						await DbeProgramAttendee.bulkWrite(bulkOps);
						bulkOps.length = 0;
					}
				} catch (error) {
					logger.error(`[DBE] Error processing ${attendance._id}:`, error);
					continue;
				}
			}

			if (bulkOps.length > 0) {
				await DbeProgramAttendee.bulkWrite(bulkOps);
			}

			const endTime = Date.now();
			const duration = endTime - startTime;

			logger.info(
				`[DBE] Completed: ${processedCount} processed, ${disqualifiedCount} disqualified`,
			);

			return {
				success: true,
				processed: processedCount,
				disqualified: disqualifiedCount,
				date: yesterday.toISOString().split("T")[0],
				duration,
				avgTimePerUser: processedCount > 0 ? duration / processedCount : 0,
			};
		} catch (error) {
			logger.error("[DBE] Critical error:", error);
			return {
				success: false,
				error: error.message,
			};
		}
	}
}

module.exports = new DbeProgramAttendeeService();
