const { failedPurchaseService } = require("./failedPurchase.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");

const failedPurchaseController = {
	// Get all failed purchases with search, filters and pagination
	async listFailedPurchases(req, res, next) {
		try {
			const {
				page = 1,
				limit = 20,
				search = "",
				status = "",
				selectedPackage = "",
				sortBy = "submittedAt",
				sortOrder = "desc",
			} = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				search,
				status,
				selectedPackage,
				sortBy,
				sortOrder,
			};

			const result = await failedPurchaseService.listFailedPurchases(options);
			res.json(
				new ApiResponse(200, result, "Failed purchases retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get failed purchase by ID
	async getFailedPurchaseById(req, res, next) {
		try {
			const { id } = req.params;
			const failedPurchase = await failedPurchaseService.getFailedPurchaseById(
				id,
			);

			res.json(
				new ApiResponse(
					200,
					{ failedPurchase },
					"Failed purchase retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get failed purchase by transaction hash
	async getFailedPurchaseByTxHash(req, res, next) {
		try {
			const { txHash } = req.params;
			const failedPurchase =
				await failedPurchaseService.getFailedPurchaseByTxHash(txHash);

			res.json(
				new ApiResponse(
					200,
					{ failedPurchase },
					"Failed purchase retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get failed purchases by username
	async getFailedPurchasesByUsername(req, res, next) {
		try {
			const { username } = req.params;
			const { page = 1, limit = 20, status = "" } = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				status,
			};

			const result = await failedPurchaseService.getFailedPurchasesByUsername(
				username,
				options,
			);
			res.json(
				new ApiResponse(200, result, "Failed purchases retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get current user's own failed purchases
	async getMyFailedPurchases(req, res, next) {
		try {
			const { page = 1, limit = 20, status = "" } = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				status,
			};

			const result = await failedPurchaseService.getFailedPurchasesByUsername(
				req.user.username,
				options,
			);
			res.json(
				new ApiResponse(
					200,
					result,
					"Your failed purchases retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Create a new failed purchase - authenticated users can submit
	async createFailedPurchase(req, res, next) {
		try {
			const purchaseData = {
				username: req.user.username,
				transactionHash: req.body.transactionHash,
				fromAddress: req.body.fromAddress,
				selectedPackage: req.body.selectedPackage,
				quantity: Number(req.body.quantity),
				priceUSD: Number(req.body.priceUSD),
				priceNTE: Number(req.body.priceNTE),
				errorMessage: req.body.errorMessage,
				status: "pending",
			};

			const failedPurchase = await failedPurchaseService.createFailedPurchase(
				purchaseData,
			);

			res
				.status(201)
				.json(
					new ApiResponse(
						201,
						{ failedPurchase },
						"Failed purchase submitted successfully",
					),
				);
		} catch (err) {
			next(err);
		}
	},

	// Update failed purchase status - admin only
	async updateFailedPurchaseStatus(req, res, next) {
		try {
			const { id } = req.params;
			const { status, adminMessage } = req.body;

			const failedPurchase =
				await failedPurchaseService.updateFailedPurchaseStatus(id, {
					status,
					adminMessage,
				});

			res.json(
				new ApiResponse(
					200,
					{ failedPurchase },
					"Failed purchase status updated successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Update failed purchase (full update) - admin only
	async updateFailedPurchase(req, res, next) {
		try {
			const { id } = req.params;
			const updateData = {
				username: req.body.username,
				transactionHash: req.body.transactionHash,
				fromAddress: req.body.fromAddress,
				selectedPackage: req.body.selectedPackage,
				quantity: req.body.quantity,
				priceUSD: req.body.priceUSD,
				priceNTE: req.body.priceNTE,
				errorMessage: req.body.errorMessage,
				status: req.body.status,
				adminMessage: req.body.adminMessage,
			};

			const failedPurchase = await failedPurchaseService.updateFailedPurchase(
				id,
				updateData,
			);

			res.json(
				new ApiResponse(
					200,
					{ failedPurchase },
					"Failed purchase updated successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Delete a failed purchase - admin only
	async deleteFailedPurchase(req, res, next) {
		try {
			const { id } = req.params;
			await failedPurchaseService.deleteFailedPurchase(id);

			res.json(
				new ApiResponse(200, {}, "Failed purchase deleted successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get statistics - admin only
	async getStatistics(req, res, next) {
		try {
			const stats = await failedPurchaseService.getStatistics();

			res.json(
				new ApiResponse(
					200,
					{ statistics: stats },
					"Statistics retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Validate manual purchase before approval - admin only
	async validateManualPurchase(req, res, next) {
		try {
			const { id } = req.params;

			const validationResult =
				await failedPurchaseService.validateManualPurchase(id);

			res.json(
				new ApiResponse(
					200,
					validationResult,
					validationResult.isValid
						? "Manual purchase validation successful"
						: "Manual purchase validation failed",
				),
			);
		} catch (err) {
			next(err);
		}
	},
};

module.exports = { failedPurchaseController };
