const { FailedPurchase } = require("./failedPurchase.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const systemNotificationService = require("../../utils/system-notification.util");
const generateEmailFromTemplate = require("../../templates/generateEmailFromTemplate");

const failedPurchaseService = {
	// Get failed purchases list with search, filters and pagination
	async listFailedPurchases(options) {
		const {
			page = 1,
			limit = 20,
			search = "",
			status = "",
			selectedPackage = "",
			sortBy = "submittedAt",
			sortOrder = "desc",
		} = options;

		const skip = (page - 1) * limit;
		const filter = {};

		if (search) {
			filter.$or = [
				{ username: { $regex: search, $options: "i" } },
				{ transactionHash: { $regex: search, $options: "i" } },
				{ fromAddress: { $regex: search, $options: "i" } },
				{ errorMessage: { $regex: search, $options: "i" } },
			];
		}

		if (status) {
			filter.status = status;
		}

		if (selectedPackage) {
			filter.selectedPackage = selectedPackage;
		}

		const sortOptions = {};
		sortOptions[sortBy] = sortOrder === "asc" ? 1 : -1;

		const [failedPurchases, totalFailedPurchases] = await Promise.all([
			FailedPurchase.find(filter)
				.sort(sortOptions)
				.skip(skip)
				.limit(limit)
				.populate("selectedPackage", "name packageType price.fixedUSD")
				.lean(),
			FailedPurchase.countDocuments(filter),
		]);

		return {
			failedPurchases,
			pagination: {
				page,
				limit,
				total: totalFailedPurchases,
				totalPages: Math.ceil(totalFailedPurchases / limit),
			},
		};
	},

	// Find failed purchase by ID
	async getFailedPurchaseById(id) {
		const failedPurchase = await FailedPurchase.findById(id)
			.populate("selectedPackage", "name packageType price.fixedUSD lockPeriod")
			.lean();
		if (!failedPurchase) {
			throw new ApiError(404, "Failed purchase not found");
		}
		return failedPurchase;
	},

	// Find failed purchase by transaction hash
	async getFailedPurchaseByTxHash(transactionHash) {
		const failedPurchase = await FailedPurchase.findOne({ transactionHash })
			.populate("selectedPackage", "name packageType price.fixedUSD lockPeriod")
			.lean();
		if (!failedPurchase) {
			throw new ApiError(404, "Failed purchase not found");
		}
		return failedPurchase;
	},

	// Get failed purchases by username
	async getFailedPurchasesByUsername(username, options = {}) {
		const { page = 1, limit = 20, status = "" } = options;

		const skip = (page - 1) * limit;
		const filter = { username };

		if (status) {
			filter.status = status;
		}

		const [failedPurchases, total] = await Promise.all([
			FailedPurchase.find(filter)
				.sort({ submittedAt: -1 })
				.skip(skip)
				.limit(limit)
				.populate("selectedPackage", "name packageType price.fixedUSD")
				.lean(),
			FailedPurchase.countDocuments(filter),
		]);

		return {
			failedPurchases,
			pagination: {
				page,
				limit,
				total,
				totalPages: Math.ceil(total / limit),
			},
		};
	},

	// Create a new failed purchase record
	async createFailedPurchase(purchaseData) {
		const {
			username,
			transactionHash,
			fromAddress,
			selectedPackage,
			quantity,
			priceUSD,
			priceNTE,
			errorMessage,
			status = "pending",
		} = purchaseData;

		if (!username) {
			throw new ApiError(401, "Username is required");
		}

		if (!transactionHash || !fromAddress || !selectedPackage) {
			throw new ApiError(
				400,
				"Missing required fields: transactionHash, fromAddress, selectedPackage",
			);
		}

		if (!quantity || !priceUSD || !priceNTE || !errorMessage) {
			throw new ApiError(
				400,
				"Missing required fields: quantity, priceUSD, priceNTE, errorMessage",
			);
		}

		// Check if transaction hash already exists
		const existingPurchase = await FailedPurchase.findOne({ transactionHash });
		if (existingPurchase) {
			throw new ApiError(409, "Transaction hash already exists");
		}

		const failedPurchase = new FailedPurchase({
			username,
			transactionHash,
			fromAddress,
			selectedPackage,
			quantity,
			priceUSD,
			priceNTE,
			errorMessage,
			status,
			submittedAt: new Date(),
		});

		await failedPurchase.save();

		// Populate package details for notifications
		await failedPurchase.populate(
			"selectedPackage",
			"name packageType price.fixedUSD",
		);

		// Get user for email notification
		const { User } = require("../users/user.model");
		const user = await User.findOne({ username }).select("email");

		logger.info("Failed purchase created", {
			id: failedPurchase._id,
			username,
			transactionHash,
			selectedPackage,
		});

		// Send notification to user
		if (user?.email) {
			try {
				const userEmailBody = `
					<div style="color: #e1e4eb; line-height: 1.8;">
						<h2 style="color: #f59e0b; margin-bottom: 20px; font-size: 24px;">⚠️ Purchase Transaction Issue</h2>
						
						<p style="margin-bottom: 15px;">Hello <strong>${username}</strong>,</p>
						
						<p style="margin-bottom: 20px;">We noticed that your recent purchase transaction encountered an issue and could not be processed automatically. Don't worry - we've recorded the details and will review it shortly.</p>
						
						<div style="background: linear-gradient(135deg, rgba(245, 158, 11, 0.15), rgba(245, 158, 11, 0.05)); border-left: 4px solid #f59e0b; padding: 20px; margin: 25px 0; border-radius: 8px;">
							<h3 style="color: #f59e0b; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Transaction Details</h3>
							<table style="width: 100%; border-collapse: collapse;">
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Package:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
										failedPurchase.selectedPackage?.name || "N/A"
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Quantity:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${quantity}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Amount:</td>
									<td style="padding: 8px 0; color: #f59e0b; text-align: right; font-weight: 600; font-size: 18px;">$${priceUSD?.toFixed(
										2,
									)}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">NTE Value:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${priceNTE?.toFixed(
										6,
									)} NTE</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Transaction Hash:</td>
									<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${transactionHash}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
									<td style="padding: 8px 0; color: #f59e0b; text-align: right; font-weight: 600;">⚠️ Under Review</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Submitted:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${new Date().toUTCString()}</td>
								</tr>
							</table>
						</div>
						
						<div style="background: rgba(239, 68, 68, 0.1); border-left: 4px solid #ef4444; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #ef4444;"><strong>⚠️ Issue Detected</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb; font-size: 14px;">${errorMessage}</p>
						</div>
						
						<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #3b82f6;"><strong>📋 What Happens Next?</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">Our team will review your transaction and verify it on the blockchain. If everything checks out, we'll manually process your purchase and you'll receive a confirmation email. This usually takes 24-48 hours.</p>
						</div>
						
						<div style="background: rgba(16, 185, 129, 0.1); border-left: 4px solid #10b981; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #10b981;"><strong>💡 Important</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">Please do not make another purchase with the same transaction. We have your details and will process them accordingly. You'll receive an email notification once your purchase is resolved.</p>
						</div>
						
						<div style="text-align: center; margin-top: 30px;">
							<a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">View Dashboard</a>
							<a href="https://bscscan.com/tx/${transactionHash}" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View on BSCScan</a>
						</div>
					</div>
				`;

				const userEmailHtml = generateEmailFromTemplate(userEmailBody);

				systemNotificationService.sendToUser({
					username,
					email: user.email,
					subject: "Purchase Transaction Under Review - NodeMeta",
					title: "Transaction Issue Detected",
					body: `Your purchase transaction of $${priceUSD?.toFixed(
						2,
					)} is under review. We'll process it manually within 24-48 hours.`,
					html: userEmailHtml,
					pushPayload: {
						icon: "/icons/icon-192x192.png",
						badge: "/icons/badge-72x72.png",
						data: {
							url: "/dashboard",
							failedPurchaseId: failedPurchase._id.toString(),
							status: "pending",
						},
					},
					sendEmail: true,
					sendPush: true,
				});
			} catch (notificationError) {
				logger.error("Failed to send failed purchase notification to user:", {
					errorMessage: notificationError.message,
					errorStack: notificationError.stack,
					username,
					failedPurchaseId: failedPurchase._id.toString(),
				});
			}
		}

		// Send notification to admins
		try {
			const adminEmailBody = `
				<div style="color: #e1e4eb; line-height: 1.8;">
					<h2 style="color: #ef4444; margin-bottom: 20px; font-size: 24px;">🚨 New Failed Purchase Reported</h2>
					
					<p style="margin-bottom: 20px;">A new failed purchase has been submitted and requires manual review and processing.</p>
					
					<div style="background: linear-gradient(135deg, rgba(239, 68, 68, 0.15), rgba(239, 68, 68, 0.05)); border-left: 4px solid #ef4444; padding: 20px; margin: 25px 0; border-radius: 8px;">
						<h3 style="color: #ef4444; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Purchase Details</h3>
						<table style="width: 100%; border-collapse: collapse;">
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Purchase ID:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 12px;">${failedPurchase._id.toString()}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Username:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${username}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">User Email:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">${
									user?.email || "N/A"
								}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Package:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
									failedPurchase.selectedPackage?.name || "N/A"
								}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Quantity:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">${quantity}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">USD Value:</td>
								<td style="padding: 8px 0; color: #ef4444; text-align: right; font-weight: 600; font-size: 18px;">$${priceUSD?.toFixed(
									2,
								)}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">NTE Value:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">${priceNTE?.toFixed(
									6,
								)} NTE</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">From Address:</td>
								<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${fromAddress}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Transaction Hash:</td>
								<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${transactionHash}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
								<td style="padding: 8px 0; color: #f59e0b; text-align: right; font-weight: 600;">⚠️ Pending Review</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Submitted:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">${new Date().toUTCString()}</td>
							</tr>
						</table>
					</div>
					
					<div style="background: rgba(239, 68, 68, 0.1); border-left: 4px solid #ef4444; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #ef4444;"><strong>⚠️ Error Details</strong></p>
						<p style="margin: 10px 0 0 0; color: #e1e4eb; font-family: monospace; font-size: 13px; white-space: pre-wrap;">${errorMessage}</p>
					</div>
					
					<div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #eab308;"><strong>📋 Action Required</strong></p>
						<ul style="margin: 10px 0 0 0; padding-left: 20px; color: #e1e4eb;">
							<li>Verify the transaction on BSCScan</li>
							<li>Check if payment was received</li>
							<li>Validate purchase details match blockchain data</li>
							<li>Process manually or reject with reason</li>
							<li>User has been notified and is waiting for resolution</li>
						</ul>
					</div>
					
					<div style="text-align: center; margin-top: 30px;">
						<a href="https://node-meta.com/dashboard/admin" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">Review Purchase</a>
						<a href="https://bscscan.com/tx/${transactionHash}" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View on BSCScan</a>
					</div>
				</div>
			`;

			const adminEmailHtml = generateEmailFromTemplate(adminEmailBody);

			systemNotificationService.sendToAdmin({
				subject: `🚨 Failed Purchase Requires Review: $${priceUSD?.toFixed(
					2,
				)} - ${username}`,
				title: "Failed Purchase Reported",
				body: `${username} submitted a failed purchase of $${priceUSD?.toFixed(
					2,
				)} that requires manual review.`,
				html: adminEmailHtml,
				pushPayload: {
					icon: "/icons/icon-192x192.png",
					badge: "/icons/badge-72x72.png",
					data: {
						url: "/dashboard/admin/failed-purchases",
						failedPurchaseId: failedPurchase._id.toString(),
						status: "pending",
					},
				},
				sendEmail: true,
				sendPush: true,
			});
		} catch (notificationError) {
			logger.error("Failed to send failed purchase notification to admins:", {
				errorMessage: notificationError.message,
				errorStack: notificationError.stack,
				failedPurchaseId: failedPurchase._id.toString(),
			});
		}

		return failedPurchase.toObject();
	},

	// Update failed purchase status
	async updateFailedPurchaseStatus(id, statusData) {
		const { status, adminMessage } = statusData;

		if (!status) {
			throw new ApiError(400, "Status is required");
		}

		const validStatuses = [
			"pending",
			"processing",
			"resolved",
			"rejected",
			"processed",
		];
		if (!validStatuses.includes(status)) {
			throw new ApiError(
				400,
				`Invalid status. Must be one of: ${validStatuses.join(", ")}`,
			);
		}

		const failedPurchase = await FailedPurchase.findById(id);
		if (!failedPurchase) {
			throw new ApiError(404, "Failed purchase not found");
		}

		failedPurchase.status = status;

		if (adminMessage !== undefined) {
			failedPurchase.adminMessage = adminMessage;
			failedPurchase.adminMessageAt = new Date();
		}

		if (status === "resolved" || status === "rejected") {
			failedPurchase.processedAt = new Date();
		}

		await failedPurchase.save();

		// Populate package details for notifications
		await failedPurchase.populate(
			"selectedPackage",
			"name packageType price.fixedUSD",
		);

		// Get user for email notification
		const { User } = require("../users/user.model");
		const user = await User.findOne({
			username: failedPurchase.username,
		}).select("email");

		// Send notification to user based on status
		if (status === "processed") {
			// Send resolution notification to user
			try {
				const userEmailBody = `
					<div style="color: #e1e4eb; line-height: 1.8;">
						<h2 style="color: #10b981; margin-bottom: 20px; font-size: 24px;">✓ Purchase Issue Resolved!</h2>
						
						<p style="margin-bottom: 15px;">Hello <strong>${
							failedPurchase.username
						}</strong>,</p>
						
						<p style="margin-bottom: 20px;">Great news! Your previously failed purchase has been reviewed and successfully resolved by our team.</p>
						
						<div style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.15), rgba(16, 185, 129, 0.05)); border-left: 4px solid #10b981; padding: 20px; margin: 25px 0; border-radius: 8px;">
							<h3 style="color: #10b981; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Purchase Details</h3>
							<table style="width: 100%; border-collapse: collapse;">
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Package:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
										failedPurchase.selectedPackage?.name || "N/A"
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Quantity:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${
										failedPurchase.quantity
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Amount:</td>
									<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600; font-size: 18px;">$${failedPurchase.priceUSD?.toFixed(
										2,
									)}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">NTE Value:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${failedPurchase.priceNTE?.toFixed(
										6,
									)} NTE</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Transaction Hash:</td>
									<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${
										failedPurchase.transactionHash
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
									<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600;">✓ Resolved</td>
								</tr>
							</table>
						</div>
						
						${
							adminMessage
								? `
						<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #3b82f6;"><strong>ℹ️ Admin Message</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">${adminMessage}</p>
						</div>
						`
								: ""
						}
						
						<div style="background: rgba(16, 185, 129, 0.1); border-left: 4px solid #10b981; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #10b981;"><strong>✓ What Happened?</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">Your purchase has been manually processed and your account has been updated accordingly. You should now see the package in your dashboard.</p>
						</div>
						
						<div style="text-align: center; margin-top: 30px;">
							<a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">View My Packages</a>
							<a href="https://bscscan.com/tx/${
								failedPurchase.transactionHash
							}" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View on BSCScan</a>
						</div>
					</div>
				`;

				const userEmailHtml = generateEmailFromTemplate(userEmailBody);

				systemNotificationService.sendToUser({
					username: failedPurchase.username,
					email: user?.email,
					subject: "Purchase Issue Resolved - NodeMeta",
					title: "Purchase Resolved",
					body: `Your failed purchase of $${failedPurchase.priceUSD?.toFixed(
						2,
					)} has been resolved and processed successfully.`,
					html: userEmailHtml,
					pushPayload: {
						icon: "/icons/icon-192x192.png",
						badge: "/icons/badge-72x72.png",
						data: {
							url: "/dashboard/packages",
							failedPurchaseId: failedPurchase._id.toString(),
							status: "resolved",
						},
					},
					sendEmail: true,
					sendPush: true,
				});
			} catch (notificationError) {
				logger.error("Failed to send resolution notification to user:", {
					errorMessage: notificationError.message,
					errorStack: notificationError.stack,
					username: failedPurchase.username,
					failedPurchaseId: failedPurchase._id.toString(),
				});
			}

			// Send resolution notification to admins
			try {
				const adminEmailBody = `
					<div style="color: #e1e4eb; line-height: 1.8;">
						<h2 style="color: #10b981; margin-bottom: 20px; font-size: 24px;">✓ Failed Purchase Resolved</h2>
						
						<p style="margin-bottom: 20px;">A failed purchase has been marked as resolved and processed.</p>
						
						<div style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.15), rgba(16, 185, 129, 0.05)); border-left: 4px solid #10b981; padding: 20px; margin: 25px 0; border-radius: 8px;">
							<h3 style="color: #10b981; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Purchase Details</h3>
							<table style="width: 100%; border-collapse: collapse;">
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Purchase ID:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 12px;">${failedPurchase._id.toString()}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Username:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
										failedPurchase.username
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Package:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${
										failedPurchase.selectedPackage?.name || "N/A"
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Quantity:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${
										failedPurchase.quantity
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">USD Value:</td>
									<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600; font-size: 18px;">$${failedPurchase.priceUSD?.toFixed(
										2,
									)}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">NTE Value:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${failedPurchase.priceNTE?.toFixed(
										6,
									)} NTE</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Transaction Hash:</td>
									<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${
										failedPurchase.transactionHash
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Original Error:</td>
									<td style="padding: 8px 0; color: #9aa3b2; text-align: right; font-size: 12px;">${failedPurchase.errorMessage?.substring(
										0,
										50,
									)}...</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
									<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600;">✓ Resolved</td>
								</tr>
							</table>
						</div>
						
						${
							adminMessage
								? `
						<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #3b82f6;"><strong>ℹ️ Admin Message</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">${adminMessage}</p>
						</div>
						`
								: ""
						}
						
						<div style="text-align: center; margin-top: 30px;">
							<a href="https://node-meta.com/admin/failed-purchases" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">View All Failed Purchases</a>
							<a href="https://bscscan.com/tx/${
								failedPurchase.transactionHash
							}" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View on BSCScan</a>
						</div>
					</div>
				`;

				const adminEmailHtml = generateEmailFromTemplate(adminEmailBody);

				systemNotificationService.sendToAdmin({
					subject: `Failed Purchase Resolved: $${failedPurchase.priceUSD?.toFixed(
						2,
					)} - ${failedPurchase.username}`,
					title: "Purchase Resolved",
					body: `${
						failedPurchase.username
					} failed purchase of $${failedPurchase.priceUSD?.toFixed(
						2,
					)} has been resolved.`,
					html: adminEmailHtml,
					pushPayload: {
						icon: "/icons/icon-192x192.png",
						badge: "/icons/badge-72x72.png",
						data: {
							url: "/admin/failed-purchases",
							failedPurchaseId: failedPurchase._id.toString(),
							status: "resolved",
						},
					},
					sendEmail: true,
					sendPush: true,
				});
			} catch (notificationError) {
				logger.error("Failed to send resolution notification to admins:", {
					errorMessage: notificationError.message,
					errorStack: notificationError.stack,
					failedPurchaseId: failedPurchase._id.toString(),
				});
			}
		} else if (status === "rejected") {
			// Send rejection notification to user
			try {
				const userEmailBody = `
					<div style="color: #e1e4eb; line-height: 1.8;">
						<h2 style="color: #ef4444; margin-bottom: 20px; font-size: 24px;">✗ Purchase Request Rejected</h2>
						
						<p style="margin-bottom: 15px;">Hello <strong>${
							failedPurchase.username
						}</strong>,</p>
						
						<p style="margin-bottom: 20px;">After reviewing your failed purchase, our team has determined that it cannot be processed.</p>
						
						<div style="background: linear-gradient(135deg, rgba(239, 68, 68, 0.15), rgba(239, 68, 68, 0.05)); border-left: 4px solid #ef4444; padding: 20px; margin: 25px 0; border-radius: 8px;">
							<h3 style="color: #ef4444; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Purchase Details</h3>
							<table style="width: 100%; border-collapse: collapse;">
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Package:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
										failedPurchase.selectedPackage?.name || "N/A"
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Quantity:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${
										failedPurchase.quantity
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Amount:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600; font-size: 18px;">$${failedPurchase.priceUSD?.toFixed(
										2,
									)}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Transaction Hash:</td>
									<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${
										failedPurchase.transactionHash
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
									<td style="padding: 8px 0; color: #ef4444; text-align: right; font-weight: 600;">✗ Rejected</td>
								</tr>
							</table>
						</div>
						
						<div style="background: rgba(239, 68, 68, 0.1); border-left: 4px solid #ef4444; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #ef4444;"><strong>⚠️ Original Issue</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">${
								failedPurchase.errorMessage
							}</p>
						</div>
						
						${
							adminMessage
								? `
						<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #3b82f6;"><strong>ℹ️ Admin Message</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">${adminMessage}</p>
						</div>
						`
								: ""
						}
						
						<div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #eab308;"><strong>💡 Next Steps</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">If you believe this rejection was made in error, please contact our support team with your transaction details. Make sure to review the requirements for package purchases before trying again.</p>
						</div>
						
						<div style="text-align: center; margin-top: 30px;">
							<a href="https://node-meta.com" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">Contact Support</a>
							<a href="https://bscscan.com/tx/${
								failedPurchase.transactionHash
							}" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View on BSCScan</a>
						</div>
					</div>
				`;

				const userEmailHtml = generateEmailFromTemplate(userEmailBody);

				systemNotificationService.sendToUser({
					username: failedPurchase.username,
					email: user?.email,
					subject: "Purchase Request Rejected - NodeMeta",
					title: "Purchase Rejected",
					body: `Your failed purchase request of $${failedPurchase.priceUSD?.toFixed(
						2,
					)} has been rejected after review.`,
					html: userEmailHtml,
					pushPayload: {
						icon: "/icons/icon-192x192.png",
						badge: "/icons/badge-72x72.png",
						data: {
							url: "/support",
							failedPurchaseId: failedPurchase._id.toString(),
							status: "rejected",
						},
					},
					sendEmail: true,
					sendPush: true,
				});
			} catch (notificationError) {
				logger.error("Failed to send rejection notification to user:", {
					errorMessage: notificationError.message,
					errorStack: notificationError.stack,
					username: failedPurchase.username,
					failedPurchaseId: failedPurchase._id.toString(),
				});
			}

			// Send rejection notification to admins
			try {
				const adminEmailBody = `
					<div style="color: #e1e4eb; line-height: 1.8;">
						<h2 style="color: #ef4444; margin-bottom: 20px; font-size: 24px;">✗ Failed Purchase Rejected</h2>
						
						<p style="margin-bottom: 20px;">A failed purchase has been marked as rejected.</p>
						
						<div style="background: linear-gradient(135deg, rgba(239, 68, 68, 0.15), rgba(239, 68, 68, 0.05)); border-left: 4px solid #ef4444; padding: 20px; margin: 25px 0; border-radius: 8px;">
							<h3 style="color: #ef4444; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Purchase Details</h3>
							<table style="width: 100%; border-collapse: collapse;">
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Purchase ID:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 12px;">${failedPurchase._id.toString()}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Username:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
										failedPurchase.username
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Package:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right;">${
										failedPurchase.selectedPackage?.name || "N/A"
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">USD Value:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">$${failedPurchase.priceUSD?.toFixed(
										2,
									)}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Transaction Hash:</td>
									<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${
										failedPurchase.transactionHash
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
									<td style="padding: 8px 0; color: #ef4444; text-align: right; font-weight: 600;">✗ Rejected</td>
								</tr>
							</table>
						</div>
						
						${
							adminMessage
								? `
						<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #3b82f6;"><strong>ℹ️ Admin Message</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">${adminMessage}</p>
						</div>
						`
								: ""
						}
						
						<div style="text-align: center; margin-top: 30px;">
							<a href="https://node-meta.com/admin/failed-purchases" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View All Failed Purchases</a>
						</div>
					</div>
				`;

				const adminEmailHtml = generateEmailFromTemplate(adminEmailBody);

				systemNotificationService.sendToAdmin({
					subject: `Failed Purchase Rejected: $${failedPurchase.priceUSD?.toFixed(
						2,
					)} - ${failedPurchase.username}`,
					title: "Purchase Rejected",
					body: `${
						failedPurchase.username
					} failed purchase of $${failedPurchase.priceUSD?.toFixed(
						2,
					)} has been rejected.`,
					html: adminEmailHtml,
					pushPayload: {
						icon: "/icons/icon-192x192.png",
						badge: "/icons/badge-72x72.png",
						data: {
							url: "/admin/failed-purchases",
							failedPurchaseId: failedPurchase._id.toString(),
							status: "rejected",
						},
					},
					sendEmail: true,
					sendPush: true,
				});
			} catch (notificationError) {
				logger.error("Failed to send rejection notification to admins:", {
					errorMessage: notificationError.message,
					errorStack: notificationError.stack,
					failedPurchaseId: failedPurchase._id.toString(),
				});
			}
		} else if (status === "processing") {
			// Send processing notification to user
			try {
				const userEmailBody = `
					<div style="color: #e1e4eb; line-height: 1.8;">
						<h2 style="color: #3b82f6; margin-bottom: 20px; font-size: 24px;">🔄 Purchase Under Review</h2>
						
						<p style="margin-bottom: 15px;">Hello <strong>${
							failedPurchase.username
						}</strong>,</p>
						
						<p style="margin-bottom: 20px;">Your previously failed purchase is now being reviewed by our team. We'll notify you once the review is complete.</p>
						
						<div style="background: linear-gradient(135deg, rgba(59, 130, 246, 0.15), rgba(59, 130, 246, 0.05)); border-left: 4px solid #3b82f6; padding: 20px; margin: 25px 0; border-radius: 8px;">
							<h3 style="color: #3b82f6; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Purchase Details</h3>
							<table style="width: 100%; border-collapse: collapse;">
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Package:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
										failedPurchase.selectedPackage?.name || "N/A"
									}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Amount:</td>
									<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">$${failedPurchase.priceUSD?.toFixed(
										2,
									)}</td>
								</tr>
								<tr>
									<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
									<td style="padding: 8px 0; color: #3b82f6; text-align: right; font-weight: 600;">🔄 Processing</td>
								</tr>
							</table>
						</div>
						
						${
							adminMessage
								? `
						<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #3b82f6;"><strong>ℹ️ Admin Message</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">${adminMessage}</p>
						</div>
						`
								: ""
						}
						
						<div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
							<p style="margin: 0; color: #eab308;"><strong>⏳ What's Happening?</strong></p>
							<p style="margin: 10px 0 0 0; color: #e1e4eb;">Our team is verifying your transaction details and will either process your purchase or provide feedback. You'll receive another notification once the review is complete.</p>
						</div>
						
						<div style="text-align: center; margin-top: 30px;">
							<a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">Go to Dashboard</a>
						</div>
					</div>
				`;

				const userEmailHtml = generateEmailFromTemplate(userEmailBody);

				systemNotificationService.sendToUser({
					username: failedPurchase.username,
					email: user?.email,
					subject: "Purchase Under Review - NodeMeta",
					title: "Purchase Processing",
					body: `Your failed purchase of $${failedPurchase.priceUSD?.toFixed(
						2,
					)} is now being reviewed by our team.`,
					html: userEmailHtml,
					pushPayload: {
						icon: "/icons/icon-192x192.png",
						badge: "/icons/badge-72x72.png",
						data: {
							url: "/dashboard",
							failedPurchaseId: failedPurchase._id.toString(),
							status: "processing",
						},
					},
					sendEmail: true,
					sendPush: true,
				});
			} catch (notificationError) {
				logger.error("Failed to send processing notification to user:", {
					errorMessage: notificationError.message,
					errorStack: notificationError.stack,
					username: failedPurchase.username,
					failedPurchaseId: failedPurchase._id.toString(),
				});
			}
		}

		logger.info("Failed purchase status updated", {
			id: failedPurchase._id,
			status,
			hasAdminMessage: !!adminMessage,
		});

		return failedPurchase.toObject();
	},

	// Update failed purchase (full update)
	async updateFailedPurchase(id, updateData) {
		const failedPurchase = await FailedPurchase.findById(id);
		if (!failedPurchase) {
			throw new ApiError(404, "Failed purchase not found");
		}

		const {
			username,
			transactionHash,
			fromAddress,
			selectedPackage,
			quantity,
			priceUSD,
			priceNTE,
			errorMessage,
			status,
			adminMessage,
		} = updateData;

		// Check for duplicate transaction hash if it's being changed
		if (transactionHash && transactionHash !== failedPurchase.transactionHash) {
			const existingPurchase = await FailedPurchase.findOne({
				transactionHash,
			});
			if (existingPurchase) {
				throw new ApiError(409, "Transaction hash already exists");
			}
		}

		// Update fields if provided
		if (username !== undefined) failedPurchase.username = username;
		if (transactionHash !== undefined)
			failedPurchase.transactionHash = transactionHash;
		if (fromAddress !== undefined) failedPurchase.fromAddress = fromAddress;
		if (selectedPackage !== undefined)
			failedPurchase.selectedPackage = selectedPackage;
		if (quantity !== undefined) failedPurchase.quantity = quantity;
		if (priceUSD !== undefined) failedPurchase.priceUSD = priceUSD;
		if (priceNTE !== undefined) failedPurchase.priceNTE = priceNTE;
		if (errorMessage !== undefined) failedPurchase.errorMessage = errorMessage;

		if (status !== undefined) {
			const validStatuses = [
				"pending",
				"processing",
				"resolved",
				"rejected",
				"processed",
			];
			if (!validStatuses.includes(status)) {
				throw new ApiError(
					400,
					`Invalid status. Must be one of: ${validStatuses.join(", ")}`,
				);
			}
			failedPurchase.status = status;

			if (status === "resolved" || status === "rejected") {
				failedPurchase.processedAt = new Date();
			}
		}

		if (adminMessage !== undefined) {
			failedPurchase.adminMessage = adminMessage;
			failedPurchase.adminMessageAt = new Date();
		}

		await failedPurchase.save();

		logger.info("Failed purchase updated", {
			id: failedPurchase._id,
			username: failedPurchase.username,
		});

		return failedPurchase.toObject();
	},

	// Delete a failed purchase
	async deleteFailedPurchase(id) {
		const failedPurchase = await FailedPurchase.findById(id);
		if (!failedPurchase) {
			throw new ApiError(404, "Failed purchase not found");
		}

		await FailedPurchase.findByIdAndDelete(id);

		logger.info("Failed purchase deleted", { id });
	},

	// Get statistics
	async getStatistics() {
		const [total, pending, processing, resolved, rejected, totalUSD, totalNTE] =
			await Promise.all([
				FailedPurchase.countDocuments(),
				FailedPurchase.countDocuments({ status: "pending" }),
				FailedPurchase.countDocuments({ status: "processing" }),
				FailedPurchase.countDocuments({ status: "resolved" }),
				FailedPurchase.countDocuments({ status: "rejected" }),
				FailedPurchase.aggregate([
					{ $group: { _id: null, total: { $sum: "$priceUSD" } } },
				]).then((result) => result[0]?.total || 0),
				FailedPurchase.aggregate([
					{ $group: { _id: null, total: { $sum: "$priceNTE" } } },
				]).then((result) => result[0]?.total || 0),
			]);

		return {
			total,
			byStatus: {
				pending,
				processing,
				resolved,
				rejected,
			},
			totalValueUSD: totalUSD,
			totalValueNTE: totalNTE,
		};
	},

	// Validate manual purchase before approval
	async validateManualPurchase(id) {
		const { User } = require("../users/user.model");
		const UserPackage = require("../purchase/userPackage.model");
		const { BonusTransaction } = require("../bonuses/bonusTransaction.model");
		const { WalletTransaction } = require("../wallets/walletTransaction.model");
		const { createWalletHash } = require("../users/user.service");
		const { getTokenValue } = require("../config/globalConfig.service");
		const {
			getValidatedReserveWallet,
		} = require("../purchase/services/userPackage.service");
		const {
			verifyNTETransfer,
		} = require("../purchase/services/blockchainVerification.service");
		const { getApprovedReserveWallets } = require("../../utils/contractUtils");

		// Get failed purchase details
		const failedPurchase = await FailedPurchase.findById(id).populate(
			"selectedPackage",
			"name packageType price.fixedUSD price.minUSD price.maxUSD lockPeriod reserveWallets isActive reserveWalletAddress",
		);

		if (!failedPurchase) {
			throw new ApiError(404, "Failed purchase not found");
		}

		const {
			username,
			transactionHash,
			fromAddress,
			selectedPackage,
			quantity,
			priceUSD,
			priceNTE,
			status,
		} = failedPurchase;

		const errors = [];
		const warnings = [];
		const details = {
			userExists: false,
			userActive: false,
			walletMatches: false,
			hashNotUsed: false,
			blockchainVerified: false,
			priceValid: false,
			nteAmountValid: false,
			packageValid: false,
			statusValid: false,
		};

		let currentNtePrice = await getTokenValue();
		// 0. Check if already processed
		if (status === "resolved") {
			errors.push("This failed purchase has already been resolved");
		} else if (status === "rejected") {
			errors.push("This failed purchase has already been rejected");
		} else {
			details.statusValid = true;
		}

		// 1. Validate basic data
		if (!username || typeof username !== "string") {
			errors.push("Invalid or missing username");
		}

		if (!transactionHash || typeof transactionHash !== "string") {
			errors.push("Invalid or missing transaction hash");
		}

		if (!fromAddress || typeof fromAddress !== "string") {
			errors.push("Invalid or missing from address");
		}

		if (!quantity || quantity <= 0) {
			errors.push("Invalid quantity: must be greater than 0");
		}

		if (!priceUSD || priceUSD <= 0) {
			errors.push("Invalid USD price: must be greater than 0");
		}

		if (!priceNTE || priceNTE <= 0) {
			errors.push("Invalid NTE price: must be greater than 0");
		}

		// 2. Validate user exists and is active
		let user = null;
		if (username) {
			user = await User.findOne({ username });
			if (!user) {
				errors.push("Username not found in database");
			} else {
				details.userExists = true;

				if (!user.isActive) {
					errors.push("User account is not active");
				} else if (user.isBlocked) {
					errors.push("User account is blocked");
				} else {
					details.userActive = true;
				}

				// 3. Validate wallet address matches
				if (fromAddress) {
					const hashedFromAddress = createWalletHash(fromAddress);
					if (!user.walletHash) {
						errors.push("User has no wallet address registered");
					} else if (user.walletHash !== hashedFromAddress) {
						errors.push(
							"From address does not match user's registered wallet address",
						);
					} else {
						details.walletMatches = true;
					}
				}
			}
		}

		// 4. Validate transaction hash not already used
		if (transactionHash) {
			const [existingPackage, hashInBonus, hashInWallet] = await Promise.all([
				UserPackage.findOne({
					$or: [
						{ blockchainTransactionHash: transactionHash },
						{ "blockchainTransaction.hash": transactionHash },
					],
				}),
				BonusTransaction.findOne({
					"metadata.blockchainTransaction.hash": transactionHash,
				}),
				WalletTransaction.findOne({
					"metadata.blockchainTransaction.hash": transactionHash,
				}),
			]);

			if (existingPackage) {
				errors.push(
					`Transaction hash already used for package purchase (Package ID: ${existingPackage._id})`,
				);
			} else if (hashInBonus) {
				errors.push("Transaction hash already used in bonus transactions");
			} else if (hashInWallet) {
				errors.push("Transaction hash already used in wallet transactions");
			} else {
				details.hashNotUsed = true;
			}
		}

		// 5. Validate package
		let targetWalletAddress = null;

		if (!selectedPackage) {
			errors.push("Selected package not found in database");
		} else {
			details.packageValid = true;

			if (!selectedPackage.isActive) {
				warnings.push(
					"Package is currently inactive - approval may still proceed",
				);
			}

			// Get current NTE price for validations
			try {
				if (!currentNtePrice || currentNtePrice <= 0) {
					errors.push("Could not retrieve current NTE price");
				}
			} catch (priceError) {
				errors.push(`Failed to get NTE price: ${priceError.message}`);
			}

			// 6. Validate price based on package type
			if (priceUSD > 0 && quantity > 0 && currentNtePrice > 0) {
				const pricePerUnit = priceUSD / quantity;

				if (selectedPackage.packageType === "unlocked") {
					// Fixed price package validation
					const expectedPriceUSD = selectedPackage.price?.fixedUSD;

					if (!expectedPriceUSD || expectedPriceUSD <= 0) {
						errors.push("Package has invalid fixed USD price configuration");
					} else {
						const totalExpectedUSD = expectedPriceUSD * quantity;
						const priceDifference = Math.abs(priceUSD - totalExpectedUSD);
						const priceTolerancePercent =
							(priceDifference / totalExpectedUSD) * 100;

						if (priceTolerancePercent > 5) {
							errors.push(
								`USD price deviation too high: ${priceTolerancePercent.toFixed(
									2,
								)}% (expected: $${totalExpectedUSD.toFixed(
									2,
								)}, received: $${priceUSD?.toFixed(2)})`,
							);
						} else if (priceTolerancePercent > 1) {
							warnings.push(
								`USD price deviation: ${priceTolerancePercent.toFixed(
									2,
								)}% (within 5% tolerance)`,
							);
							details.priceValid = true;
						} else {
							details.priceValid = true;
						}
					}
				} else {
					// Flexible/staking package validation (min/max range)
					const minUSD = selectedPackage.price?.minUSD;
					const maxUSD = selectedPackage.price?.maxUSD;

					if (!minUSD || !maxUSD || minUSD <= 0 || maxUSD <= 0) {
						errors.push(
							"Package has invalid flexible price range configuration",
						);
					} else if (pricePerUnit < minUSD) {
						errors.push(
							`Price per unit ($${pricePerUnit.toFixed(
								2,
							)}) is below minimum ($${minUSD.toFixed(2)})`,
						);
					} else if (pricePerUnit > maxUSD) {
						errors.push(
							`Price per unit ($${pricePerUnit.toFixed(
								2,
							)}) exceeds maximum ($${maxUSD.toFixed(2)})`,
						);
					} else {
						details.priceValid = true;
					}
				}

				// 7. Validate NTE amount
				const expectedNTE = priceUSD / currentNtePrice;
				const nteDifference = Math.abs(priceNTE - expectedNTE);
				const nteTolerancePercent = (nteDifference / expectedNTE) * 100;

				if (nteTolerancePercent > 10) {
					errors.push(
						`NTE amount deviation too high: ${nteTolerancePercent.toFixed(
							2,
						)}% (expected: ${expectedNTE.toFixed(
							4,
						)} NTE at $${currentNtePrice.toFixed(
							4,
						)}/NTE, received: ${priceNTE?.toFixed(4)} NTE)`,
					);
				} else if (nteTolerancePercent > 5) {
					warnings.push(
						`NTE amount deviation: ${nteTolerancePercent.toFixed(
							2,
						)}% (within 10% tolerance, may be due to price fluctuation)`,
					);
					details.nteAmountValid = true;
				} else {
					details.nteAmountValid = true;
				}
			}

			// 8. Get target wallet for blockchain verification
			try {
				targetWalletAddress = getValidatedReserveWallet(selectedPackage);
				if (!targetWalletAddress) {
					errors.push("Package has no valid reserve wallet configured");
				}
			} catch (walletError) {
				errors.push(`Failed to get reserve wallet: ${walletError.message}`);
			}
		}

		// 9. Blockchain verification (only if we have all required data)
		if (transactionHash && fromAddress && priceNTE > 0) {
			try {
				const {
					getApprovedReserveWallets,
					isApprovedReserveWallet,
				} = require("../../utils/contractUtils");
				const approvedWallets = getApprovedReserveWallets();

				logger.info("Verifying blockchain transaction", {
					transactionHash,
					fromAddress,
					expectedWallet: targetWalletAddress,
					approvedWallets,
					expectedAmount: priceNTE,
				});

				// First, get the actual transaction details from blockchain
				const rpcUrl = "https://bsc-dataseed1.binance.org/";
				const TOKEN_CONTRACT_ADDRESS = process.env.TOKEN_CONTRACT_ADDRESS;

				// Get transaction receipt
				const receiptResponse = await fetch(rpcUrl, {
					method: "POST",
					headers: { "Content-Type": "application/json" },
					body: JSON.stringify({
						jsonrpc: "2.0",
						method: "eth_getTransactionReceipt",
						params: [transactionHash],
						id: 1,
					}),
				});
				const receiptData = await receiptResponse.json();

				if (!receiptData.result) {
					errors.push("Transaction not found or not confirmed on blockchain");
					details.blockchainDetails = { error: "Transaction not found" };
				} else {
					const receipt = receiptData.result;

					if (receipt.status !== "0x1") {
						errors.push("Transaction failed on blockchain");
						details.blockchainDetails = {
							error: "Transaction failed",
							status: receipt.status,
						};
					} else {
						// Get transaction details
						const txResponse = await fetch(rpcUrl, {
							method: "POST",
							headers: { "Content-Type": "application/json" },
							body: JSON.stringify({
								jsonrpc: "2.0",
								method: "eth_getTransactionByHash",
								params: [transactionHash],
								id: 2,
							}),
						});
						const txData = await txResponse.json();
						const transaction = txData.result;

						if (!transaction) {
							errors.push("Transaction details not found");
						} else {
							// Verify sender
							const actualSender = transaction.from.toLowerCase();
							const expectedSender = fromAddress.toLowerCase();

							if (actualSender !== expectedSender) {
								errors.push(
									`Sender mismatch: expected ${fromAddress}, got ${transaction.from}`,
								);
							}

							// Parse recipient from transaction input data
							let actualRecipient = null;
							if (transaction.input && transaction.input.length >= 74) {
								const dataWithoutSelector = transaction.input.slice(10);
								const recipientHex = dataWithoutSelector.slice(0, 64);
								actualRecipient = "0x" + recipientHex.slice(24);
							}

							// Find Transfer event to get actual amount
							const transferEventTopic =
								"0xddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef";
							const transferLog = receipt.logs.find(
								(log) =>
									log.topics[0] === transferEventTopic &&
									log.address.toLowerCase() ===
										TOKEN_CONTRACT_ADDRESS?.toLowerCase(),
							);

							let actualAmount = null;
							let actualToFromLog = null;

							if (transferLog && transferLog.topics.length >= 3) {
								// Get recipient from Transfer event (more reliable)
								actualToFromLog = "0x" + transferLog.topics[2].slice(26);
								// Get amount from data
								actualAmount = parseInt(transferLog.data, 16) / 1e18;
							}

							// Use recipient from Transfer log if available, otherwise from input data
							const finalRecipient = actualToFromLog || actualRecipient;

							// Check if recipient is ANY approved reserve wallet
							const isRecipientApproved =
								finalRecipient && isApprovedReserveWallet(finalRecipient);
							const recipientMatchesPackage =
								finalRecipient &&
								targetWalletAddress &&
								finalRecipient.toLowerCase() ===
									targetWalletAddress.toLowerCase();

							details.blockchainDetails = {
								blockNumber: parseInt(receipt.blockNumber, 16),
								recipientValid: isRecipientApproved,
								actualAmount: actualAmount,
								amountDeviation: actualAmount
									? (
											(Math.abs(actualAmount - priceNTE) / priceNTE) *
											100
									  ).toFixed(2) + "%"
									: null,
							};

							// Validation logic
							if (!finalRecipient) {
								errors.push(
									"Could not determine transaction recipient from blockchain data",
								);
							} else if (isRecipientApproved) {
								// Recipient is an approved wallet - this is valid
								details.blockchainVerified = true;

								if (!recipientMatchesPackage) {
									warnings.push(
										"Transaction sent to a different approved wallet than package's configured wallet",
									);
								}

								// Verify amount
								if (actualAmount !== null) {
									const amountDiff = Math.abs(actualAmount - priceNTE);
									const amountTolerancePercent = (amountDiff / priceNTE) * 100;

									if (amountTolerancePercent > 5) {
										warnings.push(
											`Amount deviation: ${amountTolerancePercent.toFixed(2)}%`,
										);
									}
								}
							} else {
								// Recipient is NOT an approved wallet
								errors.push(
									"Transaction recipient is not an approved reserve wallet",
								);
							}
						}
					}
				}
			} catch (blockchainError) {
				errors.push(
					`Blockchain verification error: ${blockchainError.message}`,
				);
				logger.error("Blockchain verification failed", {
					transactionHash,
					error: blockchainError.message,
					stack: blockchainError.stack,
				});
			}
		} else if (!transactionHash) {
			errors.push("Missing transaction hash");
		} else if (!fromAddress) {
			errors.push("Missing from address");
		}

		// Calculate overall validity
		const isValid = errors.length === 0;

		logger.info("Manual purchase validation completed", {
			id,
			username,
			transactionHash: transactionHash?.slice(0, 20) + "...",
			isValid,
			errors: errors.length,
			warnings: warnings.length,
		});

		return {
			isValid,
			errors,
			warnings,
			checks: {
				userValid: details.userExists && details.userActive,
				walletValid: details.walletMatches,
				hashAvailable: details.hashNotUsed,
				packageValid: details.packageValid,
				priceValid: details.priceValid,
				nteValid: details.nteAmountValid,
				blockchainValid: details.blockchainVerified,
			},
			failedPurchase: {
				id: failedPurchase._id,
				username,
				packageName: selectedPackage?.name,
				quantity,
				priceUSD,
				priceNTE,
				status: failedPurchase.status,
			},
		};
	},
};

module.exports = { failedPurchaseService };
