const { newsService } = require("./news.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");

const newsController = {
	// Get all news with search, filters and pagination
	async listNews(req, res, next) {
		try {
			const {
				page = 1,
				limit = 20,
				search = "",
				sortBy = "date",
				sortOrder = "desc",
				admin = false,
				type = "",
				isEmergency = false,
			} = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				search,
				sortBy,
				sortOrder,
				includeInactive: admin === "true" && req.user?.role === "admin",
				type,
				isEmergency:
					isEmergency === "true"
						? true
						: isEmergency === "false"
						? false
						: null,
			};

			const result = await newsService.listNews(options);
			res.json(new ApiResponse(200, result, "News retrieved successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Get a single news item by ID
	async getNewsById(req, res, next) {
		try {
			const { id } = req.params;
			const admin = req.query.admin === "true";

			const includeInactive = admin && req.user?.role === "admin";
			const news = await newsService.getNewsById(id, includeInactive);

			res.json(new ApiResponse(200, { news }, "News retrieved successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Create new news item
	async createNews(req, res, next) {
		try {
			const { title, description, date, isEmergency, type } = req.body;

			if (!title || !description || !date) {
				throw new ApiError(400, "title, description, and date are required");
			}

			const newsData = {
				title: title.trim(),
				description: description.trim(),
				date: date.trim(),
				isEmergency: isEmergency || false,
				type: type ? type.trim() : "public",
			};

			const news = await newsService.createNews(newsData);

			res
				.status(201)
				.json(new ApiResponse(201, { news }, "News created successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Update existing news
	async updateNews(req, res, next) {
		try {
			const { id } = req.params;
			const { title, description, date, active, isEmergency, type } = req.body;

			const updateData = {};
			if (title !== undefined) updateData.title = title.trim();
			if (description !== undefined)
				updateData.description = description.trim();
			if (date !== undefined) updateData.date = date.trim();
			if (active !== undefined) updateData.active = active;
			if (isEmergency !== undefined) updateData.isEmergency = isEmergency;
			if (type !== undefined) updateData.type = type.trim();

			const news = await newsService.updateNews(id, updateData);

			res.json(new ApiResponse(200, { news }, "News updated successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Delete news item
	async deleteNews(req, res, next) {
		try {
			const { id } = req.params;

			const result = await newsService.deleteNews(id);

			res.json(new ApiResponse(200, result, "News deleted successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Toggle active status
	async toggleNewsStatus(req, res, next) {
		try {
			const { id } = req.params;

			const news = await newsService.toggleNewsStatus(id);

			res.json(
				new ApiResponse(200, { news }, "News status toggled successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Toggle emergency status
	async toggleEmergencyStatus(req, res, next) {
		try {
			const { id } = req.params;

			const news = await newsService.toggleEmergencyStatus(id);

			res.json(
				new ApiResponse(200, { news }, "Emergency status toggled successfully"),
			);
		} catch (err) {
			next(err);
		}
	},
};

module.exports = { newsController };
