const { News } = require("./news.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");

const newsService = {
	// Get all news with search, filters and pagination
	async listNews(options) {
		const {
			page = 1,
			limit = 20,
			search = "",
			sortBy = "date",
			sortOrder = "desc",
			includeInactive = false,
			type = "public",
			isEmergency,
		} = options;

		const filter = {};

		// Add search filter
		if (search) {
			filter.$or = [
				{ title: { $regex: search, $options: "i" } },
				{ description: { $regex: search, $options: "i" } },
			];
		}

		if (isEmergency === true) {
			filter.isEmergency = true;
		}

		// Filter by active status unless admin requests all
		if (!includeInactive) {
			filter.active = true;
		}

		if (type) {
			filter.type = type;
		}

		const skip = (page - 1) * limit;
		const sort = { [sortBy]: sortOrder === "asc" ? 1 : -1 };

		const [newsList, total] = await Promise.all([
			News.find(filter).sort(sort).skip(skip).limit(limit).lean(),
			News.countDocuments(filter),
		]);

		return {
			news: newsList,
			pagination: {
				page,
				limit,
				total,
				totalPages: Math.ceil(total / limit),
			},
		};
	},

	// Get a single news item by ID
	async getNewsById(id, includeInactive = false) {
		const filter = { _id: id };
		if (!includeInactive) {
			filter.active = true;
		}

		const news = await News.findOne(filter).lean();
		if (!news) {
			throw new ApiError(404, "News not found");
		}

		return news;
	},

	// Create new news item
	async createNews(newsData) {
		const { title, description, date, isEmergency, type } = newsData;

		// If marking as emergency, unmark all other emergency news
		if (isEmergency) {
			const targetType = type || "public";
			await News.updateMany(
				{ isEmergency: true, type: targetType },
				{ isEmergency: false },
			);
		}

		const news = new News({
			title,
			description,
			date,
			active: true,
			isEmergency: isEmergency || false,
			type: type || "public",
		});

		await news.save();

		logger.info("News created", { newsId: news._id, title });

		return news.toObject();
	},

	// Update existing news
	async updateNews(id, updateData) {
		const { title, description, date, active, isEmergency, type } = updateData;

		const news = await News.findById(id);
		if (!news) {
			throw new ApiError(404, "News not found");
		}

		// If marking as emergency, unmark all other emergency news
		if (isEmergency && !news.isEmergency) {
			const targetType = type !== undefined ? type : news.type;
			await News.updateMany(
				{ _id: { $ne: id }, isEmergency: true, type: targetType },
				{ isEmergency: false },
			);
		}

		if (title !== undefined) news.title = title;
		if (description !== undefined) news.description = description;
		if (date !== undefined) news.date = date;
		if (active !== undefined) news.active = active;
		if (isEmergency !== undefined) news.isEmergency = isEmergency;
		if (type !== undefined) news.type = type;

		await news.save();

		logger.info("News updated", { newsId: id });

		return news.toObject();
	},

	// Delete news item
	async deleteNews(id) {
		const news = await News.findById(id);
		if (!news) {
			throw new ApiError(404, "News not found");
		}

		await News.deleteOne({ _id: id });

		logger.info("News deleted", { newsId: id });

		return { deletedId: id };
	},

	// Toggle active status
	async toggleNewsStatus(id) {
		const news = await News.findById(id);
		if (!news) {
			throw new ApiError(404, "News not found");
		}

		news.active = !news.active;
		await news.save();

		logger.info("News status toggled", { newsId: id, active: news.active });

		return news.toObject();
	},

	// Toggle emergency status
	async toggleEmergencyStatus(id) {
		const news = await News.findById(id);
		if (!news) {
			throw new ApiError(404, "News not found");
		}

		// If marking as emergency, unmark all other emergency news
		if (!news.isEmergency) {
			await News.updateMany(
				{ _id: { $ne: id }, isEmergency: true, type: news.type },
				{ isEmergency: false },
			);
		}

		news.isEmergency = !news.isEmergency;
		await news.save();

		logger.info("News emergency status toggled", {
			newsId: id,
			isEmergency: news.isEmergency,
		});

		return news.toObject();
	},
};

module.exports = { newsService };
