const { newsletterService } = require("./newsletterSubscriber.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");

const newsletterController = {
	// Get all subscribers with search, filters and pagination
	async listSubscribers(req, res, next) {
		try {
			const {
				page = 1,
				limit = 20,
				search = "",
				sortBy = "createdAt",
				sortOrder = "desc",
				admin = false,
			} = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				search,
				sortBy,
				sortOrder,
			};

			const result = await newsletterService.listSubscribers(options);
			res.json(
				new ApiResponse(200, result, "Subscribers retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get a single subscriber by ID
	async getSubscriberById(req, res, next) {
		try {
			const { id } = req.params;

			const subscriber = await newsletterService.getSubscriberById(id);

			res.json(
				new ApiResponse(
					200,
					{ subscriber },
					"Subscriber retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Subscribe to newsletter (public endpoint)
	async subscribe(req, res, next) {
		try {
			const { email, name, phone, interests } = req.body;

			if (!email) {
				throw new ApiError(400, "email is required");
			}

			// Basic email validation
			const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
			if (!emailRegex.test(email)) {
				throw new ApiError(400, "Invalid email format");
			}

			const subscriberData = {
				email: email.trim().toLowerCase(),
				name: name?.trim(),
				phone: phone?.trim(),
				interests: interests?.trim(),
			};

			const result = await newsletterService.subscribe(subscriberData);

			const message = result.isNew
				? "Successfully subscribed to newsletter"
				: "Subscription updated successfully";

			res
				.status(result.isNew ? 201 : 200)
				.json(new ApiResponse(result.isNew ? 201 : 200, result, message));
		} catch (err) {
			next(err);
		}
	},

	// Update subscriber (admin only)
	async updateSubscriber(req, res, next) {
		try {
			const { id } = req.params;
			const { name, phone, interests, active } = req.body;

			const updateData = {};
			if (name !== undefined) updateData.name = name.trim();
			if (phone !== undefined) updateData.phone = phone.trim();
			if (interests !== undefined) updateData.interests = interests.trim();
			if (active !== undefined) updateData.active = active;

			const subscriber = await newsletterService.updateSubscriber(
				id,
				updateData,
			);

			res.json(
				new ApiResponse(200, { subscriber }, "Subscriber updated successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Delete subscriber (admin only)
	async deleteSubscriber(req, res, next) {
		try {
			const { id } = req.params;

			const result = await newsletterService.deleteSubscriber(id);

			res.json(new ApiResponse(200, result, "Subscriber deleted successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Toggle active status (admin only)
	async toggleSubscriberStatus(req, res, next) {
		try {
			const { id } = req.params;

			const subscriber = await newsletterService.toggleSubscriberStatus(id);

			res.json(
				new ApiResponse(
					200,
					{ subscriber },
					"Subscriber status toggled successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},
};

module.exports = { newsletterController };
