const { NewsletterSubscriber } = require("./newsletterSubscriber.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");

const newsletterService = {
	// Get all subscribers with search, filters and pagination
	async listSubscribers(options) {
		const {
			page = 1,
			limit = 20,
			search = "",
			sortBy = "createdAt",
			sortOrder = "desc",
		} = options;

		const filter = {};

		// Add search filter
		if (search) {
			filter.$or = [
				{ email: { $regex: search, $options: "i" } },
				{ name: { $regex: search, $options: "i" } },
				{ phone: { $regex: search, $options: "i" } },
			];
		}

		const skip = (page - 1) * limit;
		const sort = { [sortBy]: sortOrder === "asc" ? 1 : -1 };

		const [subscribers, total] = await Promise.all([
			NewsletterSubscriber.find(filter)
				.sort(sort)
				.skip(skip)
				.limit(limit)
				.lean(),
			NewsletterSubscriber.countDocuments(filter),
		]);

		return {
			subscribers,
			pagination: {
				page,
				limit,
				total,
				totalPages: Math.ceil(total / limit),
			},
		};
	},

	// Get a single subscriber by ID
	async getSubscriberById(id) {
		const filter = { _id: id };

		const subscriber = await NewsletterSubscriber.findOne(filter).lean();
		if (!subscriber) {
			throw new ApiError(404, "Subscriber not found");
		}

		return subscriber;
	},

	// Subscribe to newsletter
	async subscribe(subscriberData) {
		const { email, name, phone, interests } = subscriberData;

		// Check if email already exists
		const existingSubscriber = await NewsletterSubscriber.findOne({ email });

		if (existingSubscriber) {
			// Update existing subscriber by appending new interest
			const currentInterests = existingSubscriber.interests || "";
			const newInterest = interests || "";

			// Only append if the new interest is different and not empty
			let updatedInterests = currentInterests;
			if (newInterest && !currentInterests.includes(newInterest)) {
				updatedInterests = currentInterests
					? `${currentInterests}\n${newInterest}`
					: newInterest;
			}

			// Update name and phone if provided and not already set
			if (name && !existingSubscriber.name) {
				existingSubscriber.name = name;
			}
			if (phone && !existingSubscriber.phone) {
				existingSubscriber.phone = phone;
			}

			existingSubscriber.interests = updatedInterests;
			existingSubscriber.active = true;

			await existingSubscriber.save();

			logger.info("Newsletter subscription updated", {
				subscriberId: existingSubscriber._id,
				email,
			});

			return {
				subscriber: existingSubscriber.toObject(),
				isNew: false,
			};
		}

		// Create new subscriber
		const subscriber = new NewsletterSubscriber({
			email,
			name,
			phone,
			interests,
			active: true,
		});

		await subscriber.save();

		logger.info("New newsletter subscription", {
			subscriberId: subscriber._id,
			email,
		});

		return {
			subscriber: subscriber.toObject(),
			isNew: true,
		};
	},

	// Update subscriber
	async updateSubscriber(id, updateData) {
		const { name, phone, interests, active } = updateData;

		const subscriber = await NewsletterSubscriber.findById(id);
		if (!subscriber) {
			throw new ApiError(404, "Subscriber not found");
		}

		if (name !== undefined) subscriber.name = name;
		if (phone !== undefined) subscriber.phone = phone;
		if (interests !== undefined) subscriber.interests = interests;
		if (active !== undefined) subscriber.active = active;

		await subscriber.save();

		logger.info("Subscriber updated", { subscriberId: id });

		return subscriber.toObject();
	},

	// Delete subscriber
	async deleteSubscriber(id) {
		const subscriber = await NewsletterSubscriber.findById(id);
		if (!subscriber) {
			throw new ApiError(404, "Subscriber not found");
		}

		await NewsletterSubscriber.deleteOne({ _id: id });

		logger.info("Subscriber deleted", { subscriberId: id });

		return { deletedId: id };
	},

	// Toggle active status
	async toggleSubscriberStatus(id) {
		const subscriber = await NewsletterSubscriber.findById(id);
		if (!subscriber) {
			throw new ApiError(404, "Subscriber not found");
		}

		subscriber.active = !subscriber.active;
		await subscriber.save();

		logger.info("Subscriber status toggled", {
			subscriberId: id,
			active: subscriber.active,
		});

		return subscriber.toObject();
	},
};

module.exports = { newsletterService };
