const emailService = require("./email.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");
const { enqueueJob } = require("../../queues/job.queue");
const { v4: uuidv4 } = require("uuid");

class EmailController {
	async sendEmail(req, res, next) {
		try {
			const { from, to, cc, bcc, subject, text, html, contentType } = req.body;

			if (!to || (Array.isArray(to) && to.length === 0)) {
				throw new ApiError(400, "Recipient email(s) required");
			}

			if (!subject) {
				throw new ApiError(400, "Email subject required");
			}

			if (!text && !html) {
				throw new ApiError(400, "Email content required");
			}

			const sentBy = req.user?.username || req.user?.email;

			const result = await emailService.sendEmail({ from, to, cc, bcc, subject, text, html, contentType, sentBy });

			return res
				.status(200)
				.json(new ApiResponse(200, result, "Email sent successfully"));
		} catch (error) {
			next(error);
		}
	}

	async sendBulkEmail(req, res, next) {
		try {
			const { 
				from, 
				recipients, 
				cc, 
				bcc, 
				subject, 
				text, 
				html, 
				contentType,
				batchSize,
				delayBetweenBatches,
				maxRetries
			} = req.body;

			if (!recipients || !Array.isArray(recipients) || recipients.length === 0) {
				throw new ApiError(400, "Recipients array required");
			}

			if (!subject) {
				throw new ApiError(400, "Email subject required");
			}

			if (!text && !html) {
				throw new ApiError(400, "Email content required");
			}

			const sentBy = req.user?.username || req.user?.email;
			const bulkEmailId = uuidv4();

			// Create job for async processing
			const job = await enqueueJob(
				"send-bulk-email",
				{
					from,
					recipients,
					cc,
					bcc,
					subject,
					text,
					html,
					contentType,
					sentBy,
					batchSize,
					delayBetweenBatches,
					maxRetries,
					bulkEmailId,
				},
				{
					priority: 5,
					attempts: 3,
					backoff: { type: "exponential", delay: 5000 },
				}
			);

			return res
				.status(202)
				.json(new ApiResponse(202, {
					bulkEmailId,
					recipientCount: recipients.length,
					status: "processing",
					message: "Bulk email job created and processing in background",
					statusUrl: `/api/emails/bulk/${bulkEmailId}`,
				}, "Bulk email job created successfully. Use the bulkEmailId to track progress."));
		} catch (error) {
			next(error);
		}
	}

	async verifyConnection(req, res, next) {
		try {
			const result = await emailService.verifyConnection();

			return res
				.status(200)
				.json(new ApiResponse(200, result, result.message));
		} catch (error) {
			next(error);
		}
	}

	async getEmailHistory(req, res, next) {
		try {
			const { status, limit, page } = req.query;

			const result = await emailService.getEmailHistory({
				status,
				limit: parseInt(limit) || 50,
				page: parseInt(page) || 1,
			});

			return res
				.status(200)
				.json(new ApiResponse(200, result, "Email history retrieved"));
		} catch (error) {
			next(error);
		}
	}

	async getEmailById(req, res, next) {
		try {
			const { id } = req.params;

			const email = await emailService.getEmailById(id);

			return res
				.status(200)
				.json(new ApiResponse(200, email, "Email retrieved"));
		} catch (error) {
			next(error);
		}
	}

	async getBulkEmailStatus(req, res, next) {
		try {
			const { bulkId } = req.params;

			const result = await emailService.getBulkEmailStatus(bulkId);

			return res
				.status(200)
				.json(new ApiResponse(200, result, "Bulk email status retrieved"));
		} catch (error) {
			next(error);
		}
	}
}

module.exports = new EmailController();
